function hidePreviousAiChatDialogs(container) {
    const previousDialog = container ? container.find(".ai-chat-dialog") : $(".ai-chat-dialog");
    if (previousDialog.length > 0) {
        previousDialog[0].aiDialog.closeDialog();
    }
}

class AiChatDialog {
    constructor(aiChatRequest, anchor, logFunnel, showUpgrade, userName, gptData, query, subscribed, focusOnComplete) {
        this.aiChatRequest = aiChatRequest;
        this.anchor = anchor;
        this.logFunnel = logFunnel;
        this.userName = userName;
        this.gptData = gptData;
        this.query = query;
        this.showUpgrade = showUpgrade;
        this.focusOnComplete = focusOnComplete !== undefined ? focusOnComplete : true;
        this.sessionId = URL.createObjectURL(new Blob([])).split('/').pop();

        if(userName) {
            this.welcomeMessages = [i18n("Hi user, can I help you", this.userName), i18n("Type a question or select an option below")];
        } else {
            this.welcomeMessages = [i18n("Hi, can I help you"), i18n("Type a question or select an option below")];
        }

        this.limitReached = false;
        this.questionNumber = 0;
        this.subscribed = subscribed;
    }

    show() {
        hidePreviousAiChatDialogs();

        const dialog = $(this.getTemplate());
        this.dialog = dialog;
        this.dialog[0].aiDialog = this;
        this.anchor.append(dialog);

        this.input = $(".ai-input input", dialog);
        this.inputButton = $(".ai-input button", dialog);

        dialog.on("click", ".close-button", async () => {
            await this.closeDialog();
        });

        const performInput = async () => {
            const userMessage = this.input.val().trim();
            this.input.val("");
            if (userMessage.length === 0) {
                this.addMessage("chatbot", i18n("Please ask a question"));
            } else {
                await this.sendRequest(userMessage, false);
            }
        }

        dialog.on("click", ".ai-input .input-button", performInput);
        this.input.on("keyup", async (event) => {
            if (event.keyCode === 13) {
                await performInput();
            }
        });

        this.welcomeMessages.forEach((message, index) => {
            this.addTimedMessage("chatbot", message, (index + 1) * 500);
        });
        this.addTimedMessage("option", i18n("I didn't understand this step"), 1000);
        this.addTimedMessage("option", i18n("Explain this more simply"), 1000);

        dialog.on("click", ".message.option", async (e) => {
            const userMessage = e.currentTarget.textContent;
            $(e.currentTarget).remove();
            await this.sendRequest(userMessage, true);
        });

        if (this.anchor[0].getBoundingClientRect().y + dialog[0].getBoundingClientRect().height > window.visualViewport.height) {
            dialog[0].scrollIntoView({behavior: "smooth", "block": "center"});
        }

        if(this.focusOnComplete) {
            setTimeout(() => {
                this.input.focus();
            }, 500);
        }
    }

    addMessage(type, text) {
        const mainContent = $(".main-content", this.anchor);
        const message = $("<div>", {class: `message ${type}`});
        if (text) {
            message.text(text);
        }
        if (type === "chatbot") {
            message.append($("<div>"));
        }
        mainContent.append(message);
        this.scrollDown();

        return message
    }

    addToMessage(messageDiv, text) {
        let lastElement = messageDiv.children().last();
        let lastText = lastElement.text();

        for (let i = 0; i < text.length; i++) {
            const ch = text[i];
            if (ch === "\n") {
                lastElement.text(lastText);
                lastElement = $("<div>");
                messageDiv.append(lastElement);
                lastText = "";
            } else {
                lastText += ch;
            }
        }
        lastElement.text(lastText);
    }

    addTimedMessage(type, text, timeout) {
        setTimeout(() => {
            this.addMessage(type, text);
        }, timeout);
    }

    scrollDown() {
        const explainPane = $(".main-content", this.dialog);
        if (explainPane) {
            explainPane[0].scrollTop = explainPane[0].scrollHeight;
        }
    }

    async sendRequest(userMessage, userTextCameFromButton) {
        this.enableInput(false);
        const $this = this;

        this.addMessage("user", userMessage);
        this.questionNumber++;

        if (!this.subscribed && (!userTextCameFromButton || sy_var !== '2' || this.questionNumber > 1)) {
            const messageDiv = this.addMessage("chatbot");
            this.addToMessage(messageDiv, this.questionNumber  === 1 ? i18n("Chat Subscribe required") : i18n("Chat Continue Subscribe required"));
            const btn = messageDiv.children().append("<div class='chat-action'>" + i18n("Upgrade") + "</div>");
            btn.on("click", () => {
                this.showUpgrade("AI_Chat_Funnel", false);
            });
            this.scrollDown();
            $(".message.option").remove();
            $this.logFunnel('SeenFeature', "AI_Chat_Funnel");
            return;
        }

        let {messageDiv, waitingElement} = this.prepareForRequest();

        const state = {
            messageDiv,
            waitingElement,
            lastResponseLen: 0,

            callbacks: {
                receiveResponseProgress: this.receiveResponseProgress.bind(this),
                requestError: this.requestError.bind(this),
                requestComplete: this.requestComplete.bind(this)
            }
        }

        setTimeout(async () => {
            await $this.aiChatRequest.sendRequest(userMessage, $this.gptData, $this.query, $this.sessionId, userTextCameFromButton, state);
        }, 500);
    }

    prepareForRequest() {
        const messageDiv = this.addMessage("chatbot");
        let waitingElement = $("<div>", {class: "waiting"});
        const addDot = (element, delay) => {
            const dotElement = $("<span>.</span>");
            dotElement.css({'animation-delay': `${delay}s`});
            element.append(dotElement);
        }
        addDot(waitingElement, 0);
        addDot(waitingElement, 0.1);
        addDot(waitingElement, 0.2);
        messageDiv.append(waitingElement)
        this.scrollDown();

        return {messageDiv, waitingElement};
    }

    receiveResponseProgress(response, state) {
        const $this = this;

        state.waitingElement.remove();

        let thisResponse = response.substring(state.lastResponseLen);
        state.lastResponseLen = response.length;

        const noErrorResponse = thisResponse.replace("<!-ERROR->", "");
        if (noErrorResponse !== thisResponse) {
            $this.limitReached = true;
            thisResponse = noErrorResponse;
        }
        if (thisResponse.length > 0) {
            $this.addToMessage(state.messageDiv, thisResponse)
            $this.scrollDown();
        }
    }

    requestError(status, state) {
        const $this = this;
        if (status !== 200) {
            $this.addToMessage(state.messageDiv, "I am sorry - something went wrong, please try again...");
            $this.scrollDown();
        }
    }

    requestComplete() {
        if (!this.limitReached) {
            this.enableInput(true);
            if(this.focusOnComplete) {
                this.input.focus();
            }
        }
    }

    enableInput(enable) {
        this.input.prop( "disabled", !enable);
        this.inputButton.prop( "disabled", !enable);
        $(".ai-chat-dialog .message.option").css( "pointer-events", enable ? "all" : "none");
    }

    async closeDialog() {
        this.dialog.remove();
    }

    getTemplate() {
        return `
            <div class="ai-chat-dialog">
                <header>
                    <svg ><use href="#ai-help"></use></svg>
                    <div class="dialog-title">
                        <div>${i18n("Chat with Symbo")}</div>     
                        <div class="tooltip-container">
                            <svg><use href="#definition-i-icon-black"></use></svg>
                            <svg class="red"><use href="#definition-i-icon-red"></use></svg>
                            <div class="tooltip">${i18n("AI may present inaccurate or offensive content")}</div>                                
                        </div>
                    </div>
                    <button class="close-button"><svg ><use href="#close-svg"></use></svg></button>                    
                </header>
                <div class="main-content"></div>
                <div class="dialog-footer">
                    <div class="ai-input">
                        <input placeholder="${i18n("Ask a question")}">
                        <button class="input-button"><svg ><use href="#send"></use></svg></button>
                    </div>
                    <div class="disclaimer">${i18n("Do not enter any personal information")}</div>    
                </div>
            </div>
        `;
    }
}