/*
 * Alipay.com Inc.
 * Copyright (c) 2004-2015 All Rights Reserved.
 *
 * common.js
 *
 */

// 命名空间 alipay mobile cashier
var amc = {};

// 平台信息
amc.platform = document.platform || 'chrome';
amc.isIOS = (amc.platform === 'iOS');
amc.isAndroid = (amc.platform === 'android');
amc.path = amc.isAndroid ? 'com.alipay.android.app/' : 'AlipaySDK.bundle/';

// 兼容收银台H5容器-end

amc.rpcData = flybird.rpcData || {};
amc.isFlybird = (amc.isIOS || amc.isAndroid);

// 页面是否初始化完成
amc.pageInited = false;

// key为spmId，value为bool值，用来标识某个曝光埋点是否执行过
amc.spmExposured = {};

if (amc.isIOS) {
    amc.isPlus = (document.getCustomAttr('isIphonePlus') === 'true');
    amc.isIPhoneX = (document.getCustomAttr('isIphoneX') === 'true');
    amc.osVersion = document.getCustomAttr('osVersion');
    amc.supportAuthType = document.getCustomAttr('supportAuthType');
}

if (amc.isFlybird) {
    amc.sdkVersion = document.getCustomAttr('sdkVersion') || '0.0.0';

    // 专业版SDK类型: ios_钱包,android_钱包,ios_淘宝,android_淘宝分别就是1, 2, 3, 4
    // 参考(callflag): http://gitlab.alipay-inc.com/mpay/cashierProject/wikis/msp_request_params
    amc.mspType = document.getCustomAttr('mspType') || '1';
    // 是否是专业版SDK(支付宝App外)
    amc.isSDK = (amc.mspType != '1' && amc.mspType != '2' && amc.mspType != '1_hk' && amc.mspType != '2_hk');
}

// display='flex'使隐藏元素可见
amc.VISIBLE = 'flex';
// 公共方法名字空间 function
amc.fn = {};
amc.fn.getById = function(id) {
    return document.getElementById(id);
};

amc.iOSGestureBackMode = 'disabled';
amc.iOSGestureBackCallback = null;

/**
 * 版本比较
 * 备注:合法的版本号只包含点号和数字，每个点号间用数字分隔(10..0属不合法);
 *     如果两个版本号长度不同(1.0与1.0.9),短的补零进行比较(1.0.0与1.0.9)
 * v1(string) 待比较版本
 * v2(string) 待比较版本
 * return: 1)NaN: v1,v2任意一个不合法 2)1: v1 > v2 3) 0: v1 = v2 4)-1: v1 < v2
 */
amc.fn.versionCompare = function(v1, v2) {
    v1 = amc.fn.isString(v1) ? v1 : '';
    v2 = amc.fn.isString(v2) ? v2 : '';

    v1parts = v1.split('.'),
    v2parts = v2.split('.');

    function isValidPart(x) {
        return (/^\d+$/).test(x);
    }

    if (!v1parts.every(isValidPart) || !v2parts.every(isValidPart)) {
        return NaN;
    }

    while (v1parts.length < v2parts.length) v1parts.push('0');
    while (v2parts.length < v1parts.length) v2parts.push('0');


    v1parts = v1parts.map(Number);
    v2parts = v2parts.map(Number);

    for (var i = 0; i < v1parts.length; ++i) {

        if (v1parts[i] == v2parts[i]) {
            continue;
        } else if (v1parts[i] > v2parts[i]) {
            return 1;
        } else {
            return -1;
        }
    }

    return 0;
};

amc.fn.sdkGreaterThan = function(v) {
    return (amc.fn.versionCompare(amc.sdkVersion, v) === 1);
};

/**
 * SDK 10.7.0开始支持
 */
amc.fn.sdkGreaterThanOrEqual = function(v) {
    return (amc.fn.versionCompare(amc.sdkVersion, v) === 1 || amc.fn.versionCompare(amc.sdkVersion, v) === 0);
};
/**
 * 埋点方法
 * biz(string) 埋点类型：1. 'promotion' 广告类 2. 'spmtracker', ''
 * info(object) 埋点参数
 * @deprecated 不应该使用本方法，应该使用 amc.fn.mqpSpm 并以 amc.spm.bizCode 指定 SPM 埋点的业务类型。
 */
amc.fn.feedback = function(biz, info) {
    if (!info) {
        return;
    }

    var paramObj = {};
    var feedbackType = biz || 'promotion';
    paramObj[feedbackType] = info;

    var actionName = "loc:feedback('" + JSON.stringify(paramObj) + "')";
    var obj = { 'action': { 'name': actionName } };
    document.submit(obj);
};

/**
 * actionType 不能传整数，只能传字符串
 * 支持版本: 钱包9.9.6, sdk 10.6.6
 * 记录action日志
 * 对应日志字段: initial,[actionType],[action],41,529,1038,277,155,QUICKPAY@open-pwd-check-flex,1,-,169,2,13:08:40:787),
 */
amc.fn.logAction = function(action, actionType) {
    if (!action) return;

    if(!window._pageId) {
       window._pageId = Math.random().toString(36).substr(2,3);
    }

    var paramObj = {};
    paramObj['name'] = action;

    // 备注: log中iOS不支持整数，否则会crash
    paramObj['type'] = (actionType + '|' + window._pageId);

    var actionName = "loc:continue;loc:log('" + JSON.stringify({ 'action': paramObj }) + "')";
    var obj = { 'action': { 'name': actionName } };
    document.submit(obj);
};

/**
 * 支持版本: 钱包9.9.6, sdk 10.6.6
 * 记录错误日志
 * 日志字段: ([code],[desc])
 */
amc.fn.logError = function(code, desc) {
    if (!code || !desc) return;

    if(!window._pageId) {
       window._pageId = Math.random().toString(36).substr(2,3);
    }
    desc = desc + '|' + window._pageId;

    var paramObj = {};
    paramObj['code'] = code;
    paramObj['desc'] = desc;

    var actionName = "loc:continue;loc:log('" + JSON.stringify({ 'error': paramObj }) + "')";
    var obj = { 'action': { 'name': actionName } };
    document.submit(obj);
};

/**
 * 支持版本: 钱包10.0.0, sdk 10.7.0才提供该方法
 * 获取当前模板名称，如: cashier-activity-flex
 * flybird.local作为Native传递本地信息的对象
 */
amc.fn.getTplId = function() {
    if (amc.fn.sdkGreaterThan('10.6.9') && flybird && flybird.local) {
        return flybird.local.tplId || '';
    } else {
        return '';
    }
};

/**
 * 打印日志:记录次数, 支持版本: 钱包9.9.6, sdk 10.6.6
 * 备注: biz/op/desc均按照自己分析日志的需求制定，日志分析联系:@子学
 * 日志字段: (@@count@@[biz],[op],[desc],-),
 */
amc.fn.logCount = function(biz, op, desc) {
    if (!biz || !op) return;

    var paramObj = {};
    paramObj['biz'] = biz;
    paramObj['op'] = op;
    paramObj['desc'] = desc;

    var actionName = "loc:continue;loc:log('" + JSON.stringify({ 'count': paramObj }) + "')";
    var obj = { 'action': { 'name': actionName } };
    document.submit(obj);
};

/**
 * 支持版本: 钱包9.9.6, sdk 10.6.6
 * 上报页面init函数执行完毕
 * 日志字段: (@@count@@tpl,init,[tplId],-),
 */
amc.fn.logPageInit = function(success) {
    if (!success) {
        amc.fn.logCount('tpl-init-fail', amc.fn.getTplId(), '-');
    } else {
        amc.pageInited = true;
    }
};

/**
 * 支持版本: 钱包9.9.6, sdk 10.6.6
 * 设置返回结果。
 * 备注: result['doNotExit'] 表示 不退出收银台
 */
amc.fn.setResult = function(result) {
    if (!result) {
        return;
    }

    document.submit({ 'action': { 'name': 'loc:returnData', 'params': result } });
};

/*
 * 支持版本: 钱包9.9.6, sdk 10.6.6
 * 提交页面结果
 */
amc.fn.submitData = function(resultPageData) {
    if (!resultPageData) {
        return;
    }

    var result = { resultPageData: resultPageData, doNotExit: true };
    amc.fn.setResult(result);
};

/*
 * 创建节点
 * tagName(string): 要创建的节点的类型名称，如'div'
 * className(string): CSS类名
 * parent(tag): 父节点
 */
amc.fn.create = function(tagName, className, parent) {
    var tag = document.createElement(tagName);
    if (!tag) {
        return;
    }

    if (className) {
        tag.className = className;
    }

    if (parent) {
        parent.appendChild(tag);
    }

    return tag;
};

/*
 * 创建节点
 * data(obj): data['pageloading'] === '1'或'0',分别表示显示indicator和隐藏indicator
 * btnTitleId(string): 需要被隐藏或显示的按钮文案的id
 * indicatorId(string): 需要被显示或隐藏的indicator图片的id
 */
amc.fn.shouldShowBtnIndicator = function(data, btnTitleId, indicatorId) {
    if (!data) {
        return false;
    }

    if (data['pageloading'] === '1') {
        amc.fn.hide(btnTitleId);
        amc.fn.show(indicatorId);
        return true;
    } else if (data['pageloading'] === '0') {
        amc.fn.show(btnTitleId);
        amc.fn.hide(indicatorId);
    }

    return false;
};

// 公共图片资源名字空间 resource
amc.res = {};
//全屏返回按钮
amc.res.navBack = amc.path + 'alipay_msp_back';
amc.res.navBackWhite = amc.path + 'alipay_msp_back_white';
amc.res.navClose = amc.path + 'alipay_msp_back_close';
amc.res.navMore = amc.path + 'alipay_msp_mini_three_point';
//半屏返回按钮
amc.res.arrowLeft = amc.path + 'alipay_msp_arrow_left';
amc.res.close = amc.path + 'alipay_msp_close';
amc.res.arrowRight = amc.path + 'alipay_msp_arrow_right';
// 菊花
amc.res.loading = amc.path + 'alipay_msp_loading.gif';
amc.res.success = amc.path + 'alipay_msp_success.gif';

amc.res.help = amc.path + 'alipay_msp_help';
amc.res.helpWhite = amc.path + 'alipay_msp_help_icon_white';
//小菊花
amc.res.indicator = 'indicatior';

amc.specs = {};
// iPhone X底部
amc.specs.bottomHeight = amc.isIPhoneX ? 34 : 0;
amc.specs.navHeight = amc.isAndroid ? 48 : 44; // iPhone 状态栏高度均为 64 point
amc.specs.marqueeHeight = 36;
//安卓中body包含导航栏、iOS中body不包含导航栏
amc.specs.bodyHeight = window.innerHeight - amc.specs.navHeight;
// 减去状态栏
if (amc.isIOS) {
    if (amc.isIPhoneX) {
        amc.specs.bodyHeight -= 44;
    } else {
        amc.specs.bodyHeight -= 20;
    }
}
amc.specs.iBodyMinH = amc.isPlus ? 518 : 400;
amc.specs.iBodyHeight = (0.66 * window.innerHeight > amc.specs.iBodyMinH) ? Math.round(0.66 * window.innerHeight) : amc.specs.iBodyMinH;
amc.specs.iNavHeight = amc.isPlus ? 60 : 46; //45.5的高度+1px的横线

amc.fn.exit = function() {
    document.submit({ 'action': { 'name': 'loc:exit' } });
};
amc.fn.back = function() {
    document.submit({ 'action': { 'name': 'loc:back' } });
};

/*
 * 在外部(浏览器或淘宝等)打开链接(在支付过程中，如果在支付过程中跳转会影响支付成功率)
 * url(string): 要跳转的地址
 * willReturn(string):
 * '0': iOS: 退出收银台，不给业务方结果, Android: 不会退出收银台
 * '1': 退出收银台，给业务方结果
 * '2': 不退出收银台，不给业务方结果
 * '4': 隐藏收银台，跳外部业务，可恢复；钱包10.1.28支持，钱包10.1.30版本可通过该函数调用
 * '3': 退出收银台，且可恢复
 * '5': 退出收银台，不可恢复，不给结果，目前结合回到钱包首页的url使用
 */
amc.fn.openurl = function(url, returnType) {
    if (!url) {
        return;
    }

    var returnTypes = ['0', '1', '2', '3', '4', '5'];
    if (!(returnType in returnTypes)) {
        returnType = '0';
    }

    document.submit({ 'action': { 'name': "loc:openurl('" + url + "','" + returnType + "')" } });
};

/*
 * 在内置浏览器打开链接(如:打开协议内容)
 * url(string): 要打开的链接地址
 */
amc.fn.openweb = function(url) {
    if (!url) {
        return;
    }
    document.submit({ 'action': { 'name': "loc:openweb('" + url + "')" } });
};

amc.fn.getMarquee = function(txt) {
    var marquee = create('marquee');
    marquee.className = 'amc-marquee';
    marquee.innerText = txt || '';
    return marquee;
};
//### 播放gif图片
amc.fn.playGif = function(gifNode, gifImg) {
    if (gifImg) {
        gifNode.src = gifImg;
        amc.fn.show(gifNode);
    } else {
        amc.fn.hide(gifNode);
    }
};

amc.fn.hideKeyboard = function() {
    //### 收起全部键盘
    var inputArray = document.querySelectorAll('input');
    for (var i = 0; i < inputArray.length; i++) {
        inputArray[i].blur();
    }
};
//### 退出确认
amc.fn.exitConfirm = function(msg) {
    msg = msg || '{{confirm_exit}}';
    var confirmObj = {
        'title': '',
        'message': msg,
        'okButton': '{{confirm_btn}}',
        'cancelButton': '{{cancel}}'
    };

    document.confirm(confirmObj, function(result) {
        if (result.ok) {
            document.submit({ "action": { "name": "loc:exit" } });

        }
    });
};

//### 协议个数不定时打开协议
amc.fn.showProtocolList = function(pList) {
    if (!pList || pList.length <= 0) {
        return;
    }
    // 一个协议
    if (pList.length == 1) {
        document.submit({ "action": { "name": "loc:openweb('" + pList[0]["url"] + "','" + pList[0]["text"] + "')" } });
        return;
    }
    // 多个协议
    var protocolArray = new Array();
    for (var i = 0; i < pList.length; i++) {
        protocolArray.push(pList[i]["text"]);
    }
    document.actionSheet({
        'text': '{{protocol}}',
        'btns': protocolArray,
        'cancelBtn': '{{cancel}}'
    }, function(data) {
        if (data.index < pList.length) {
            document.submit({ "action": { "name": "loc:openweb('" + pList[data.index]["url"] + "','" + pList[data.index]["text"] + "')" } });
        }
    });
};

amc.fn.isString = function(str) {
    return ((str instanceof String) || typeof str === 'string');
};

amc.fn.isArray = function(o) {
    return Object.prototype.toString.call(o) === '[object Array]';
};

amc.fn.isObject = function(o) {
    return Object.prototype.toString.call(o) === '[object Object]';
};

amc.fn.show = function(tag) {
    if (tag) {
        tag = amc.fn.isString(tag) ? document.getElementById(tag) : tag;

        if (tag) {
            tag.style.display = amc.VISIBLE;
        }
    }
};

amc.fn.hide = function(tag) {
    if (tag) {
        tag = amc.fn.isString(tag) ? document.getElementById(tag) : tag;

        if (tag) {
            tag.style.display = 'none';
        }
    }
};

/**
 *
 * @param {bool} show 是否显示loading
 * @param {bool} keep 改写iajax标 1) false:(iajax=0), 2) true: (iajax=1) 3) undefined: 不去改写iajx
 */
amc.fn.showLoading = function(show, keep) {
    document.submit({
        'action': {
            'name': 'loc:loading'
        },
        'param': {
            'show': !!show,
            'keeploading': keep
        }
    });
};

/*
 * 视图范例: < 返回     支付宝      完成/···  (/表示互斥)
 * lImg(string): 左侧图片(通常作为返回按钮), 默认id=“navImgL”;false则不创建
 * lTxt(string): 最左侧的文字(“返回”):如果为undefined/""则不创建;默认id="navTxtL";
 * mTxt(string): 中间文案(作为Title):如果为undefined/""则不创建;
 * rTxt(string): 最左侧的文字(“完成”/"取消"等):默认id="navTxtR"; 如果为undefined/""则不创建
 * rImg(string): 最右侧的图片(例如:作为"more"按钮),默认id="navImgR"; false则不创建
 * onLeft(function): 左侧box按下的回调
 * onRight(function): 右侧box按下的回调
 * option(object): 手势返回参数 钱包10.0.0才支持
 */
amc.fn.iOSNav = function(lImg, lTxt, mTxt, rTxt, rImg, onLeft, onRight, option) {
    var create = amc.fn.create;

    // nav
    var _nav = create('nav');
    _nav.className = 'amc-nav-box';

    // left box
    var _lBox = create('div');
    _lBox.className = 'amc-nav-l-box';
    _lBox.id = 'navBoxL';

    // left img
    if (lImg) {
        var _lImg = create('img');
        _lImg.className = 'amc-nav-l-img';
        _lImg.id = 'navImgL';
        _lImg.src = lImg;
        _lBox.appendChild(_lImg);

        //voice over
        if (lImg === amc.res.navBack) {
            _lBox.alt = '{{return}}';

            amc.iOSGestureBackCallback = onLeft;
            amc.iOSGestureBackMode = 'gesture';
            if (option && option.backMode) {
                amc.iOSGestureBackMode = option.backMode;
            }
        }
        _lBox.onclick = onLeft;
    }

    // left label
    if (lTxt) {
        var _lTxt = create('label');
        _lTxt.className = 'amc-nav-l-text';
        _lBox.alt = lTxt;

        //如果左侧没有图片，则不需要间距
        if (!lImg) {
            _lTxt.style.marginLeft = 0;
        }
        _lTxt.id = 'navTxtL';
        _lTxt.innerText = lImg ? "" : lTxt;
        _lBox.appendChild(_lTxt);

        _lBox.onclick = onLeft;
    }

    // middle box
    var _mBox = create('div');
    _mBox.className = 'amc-nav-m-box amc-v-box';
    // 6p: 两边padding 40, l-box/r-box 各84, 总和208;
    // 5s: padding 30, l-box/r-box 各70 总和 170
    // 算上间距，稍微多减一点
    _mBox.style.maxWidth = window.innerWidth - (amc.isPlus ? (210) : (180));
    if (mTxt) {
        var _mTxt = create('label');
        _mTxt.id = 'navTxtM';
        _mTxt.innerText = mTxt;
        _mTxt.className = 'amc-nav-m-text';
        _mBox.appendChild(_mTxt);
    }

    var _rBox = create('div');
    _rBox.className = 'amc-nav-r-box';
    _rBox.id = 'navBoxR';


    if (rTxt) {
        var _rTxt = create('label');
        _rTxt.className = 'amc-nav-r-text';
        _rTxt.innerText = rTxt;
        _rTxt.id = 'navTxtR';
        _rBox.appendChild(_rTxt);
        _rBox.onclick = onRight;
    } else if (rImg) {
        var _rImg = create('img');
        _rImg.className = 'amc-nav-r-img';
        _rImg.id = 'navImgR';
        _rImg.src = rImg;
        _rBox.appendChild(_rImg);
        _rBox.onclick = onRight;
    }

    _nav.appendChild(_lBox);
    _nav.appendChild(_mBox);
    _nav.appendChild(_rBox);

    return _nav;
};


/*
 * 视图范例: < | 返回       ···
 * 视图范例: 支付宝         |完成
 * lImg(string): 左侧图片(通常作为返回按钮), 默认id=“navImgL”;
 * lTxt(string): 安卓没有左侧文字,该变量作为占位符; 该参数为Null
 * mTxt(string): 中间文案(作为Title):
 * rTxt(string): 最右侧的文字(“完成”/"取消"等):默认id="navTxtR";
 * rImg(string): 最右侧的图片(例如:作为"more"按钮),默认id="navImgR";
 * onLeft(function): 左侧box按下的回调
 * onRight(function): 右侧box按下的回调
 * 注意: 以上参数若为''/undefined/false/null则不创建
 */
amc.fn.androidNav = function(lImg, lTxt, mTxt, rTxt, rImg, onLeft, onRight) {
    var create = amc.fn.create;

    // 导航栏外容器(为了在导航栏下方放置横线)
    var _navContainer = create('div', 'amc-nav-container-android');

    // 导航栏
    var _nav = create('div', 'amc-nav-box-android', _navContainer);
    // 导航栏下方横线
    var line = create('div', 'amc-nav-horiz-line-android', _navContainer);

    var _lBox = create('div', 'amc-nav-l-box-android', _nav);
    _lBox.id = 'navBoxL';

    // <
    if (lImg) {
        var _lImgBox = create('div', 'amc-nav-l-img-box-android', _lBox);
        _lImgBox.onclick = onLeft;
        _lImgBox.id = 'navImgBoxL';

        var _lImg = create('img', 'amc-nav-l-img-android', _lImgBox);
        _lImg.id = 'navImgL';
        _lImg.contentmode = "Center";
        _lImg.src = lImg;

        // |
        // var _lBar = create('div', 'amc-nav-line-android', _lBox);

        //voice over
        if (lImg === amc.res.navBack) {
            _lImg.alt = '{{return}}';
        } else if (lImg === amc.res.navClose) {
            _lImg.alt = '{{exit}}';
        }
    }

    // 返回 or 支付宝
    var _mTxt = create('label', 'amc-nav-m-text-android', _lBox);
    _mTxt.innerText = mTxt || '';
    _mTxt.id = 'navTxtM';


    var _rBox = create('div');
    _rBox.className = 'amc-nav-r-box-android';
    _rBox.id = 'navBoxR';

    // ···
    if (rImg) {
        var _rImg = create('img', 'amc-nav-r-img-android', _rBox);
        _rImg.id = 'navImgR';
        _rImg.src = rImg;

        _rBox.onclick = onRight;
    } else if (rTxt) {
        // | 设置
        var _rBar = create('div', 'amc-nav-line-android', _nav);

        var _rTxt = create('label', 'amc-nav-r-text-android', _rBox);
        _rTxt.id = 'navTxtR';
        _rTxt.innerText = rTxt;
        _rBox.onclick = onRight;
    }
    _nav.appendChild(_rBox);

    //安卓平台需要有nav标签才能能判断是全屏
    var _isFullScreen = create('nav', 'amc-hidden', _navContainer);

    return _navContainer;
};

amc.fn.endsWith = function (origin, search) {
    if (!origin || !search) {
        return false
    }
    return origin.substring(origin.length - search.length, origin.length) === search;
};

amc.fn.revertImgPath = function (path) {
    if (path && (amc.fn.endsWith(path, 'alipay_msp_back') || amc.fn.endsWith(path, 'alipay_msp_help')
        || amc.fn.endsWith(path, 'alipay_msp_refresh') || amc.fn.endsWith(path, 'alipay_msp_mini_three_point'))) {
        return path + '_nav';
    }
    return path;
};

amc.fn.getNav = function (lImg, lTxt, mTxt, rTxt, rImg, onLeft, onRight, option) {
    if (amc.isAndroid) {
        return amc.fn.androidNav(amc.fn.revertImgPath(lImg), lTxt, mTxt, rTxt, amc.fn.revertImgPath(rImg),
            onLeft, onRight);
    } else {
        return amc.fn.iOSNav(amc.fn.revertImgPath(lImg), lTxt, mTxt, rTxt, amc.fn.revertImgPath(rImg),
            onLeft, onRight, option);
    }
};

amc.fn.pressableElement = function(elDiv, el) {
    if (elDiv && el) {
        elDiv.onmousedown = function() {
            el.style.opacity = '0.5';
        };
        elDiv.onmouseup = function() {
            el.style.opacity = '1';
        };
    }
};

/*
 * isBack 左上角为返回按钮、否则为取消(安卓为关闭)
 * title(string): 中间文案(作为Title),id: iNavTxtM
 * rTxt(string): 最左侧的文字(“完成”/"取消"等):
 * rImg(string): 最右侧的图片(例如:作为"more"按钮):默认id="inavImgR";
 * onLeft(function): 左侧box按下的回调
 * onRight(function): 右侧box按下的回调
 * 注意: 以上参数若为''/undefined/false/null则不创建
 */
amc.fn.navBack = function(isBack, title, rTxt, rImg, onLeft, onRight, option) {
    if (isBack) {
        return amc.fn.getNav(amc.res.navBack, '{{return}}', title, rTxt, rImg, onLeft, onRight, option);
    } else {
        return amc.fn.getNav(amc.isAndroid ? amc.res.navClose : null, '{{cancel}}', title, rTxt, rImg, onLeft, onRight, option);
    }
};


/*
 * 视图范例: x    付款详情   ?/设置
 * lImg(string): 左侧按钮图片
 * title(string): 中间文案(作为Title),id: iNavTxtM
 * rTxt(string): 最左侧的文字(“完成”/"取消"等):
 * rImg(string): 最右侧的图片(例如:作为"more"按钮):默认id="inavImgR";
 * onLeft(function): 左侧box按下的回调
 * onRight(function): 右侧box按下的回调
 * mImg(string): 导航栏中间logo
 * subTitle(string): 副标题
 * 注意: 以上参数若为''/undefined/false/null则不创建
 * mImg pre-confirm页特殊需求，需要添加一个img
 */
amc.fn.iNav = function(lImg, title, rTxt, rImg, onLeft, onRight, mImg, subtitle) {
    var create = amc.fn.create;
    var _iNav = create('div');
    _iNav.className = 'amc-i-nav-box';

    var _lBox = create('div');
    _lBox.className = 'amc-i-nav-l-box';
    _lBox.id = 'iNavBoxL';

    if (lImg) {
        var _lImg = create('img');
        _lImg.className = 'amc-i-nav-l-img';
        _lImg.src = lImg;
        _lImg.id = 'iNavImgL';
        _lBox.appendChild(_lImg);
        amc.fn.pressableElement(_lBox, _lImg);

        if (lImg === amc.res.arrowLeft) {
            _lBox.alt = '{{return}}';
        } else if (lImg === amc.res.close) {
            _lBox.alt = '{{exit}}';
        }
        _lBox.accessibilityTraits = 'Button';
        _lBox.onclick = onLeft;
    }

    var _mBox = create('div');
    _mBox.className = 'amc-i-nav-m-box';
    if (mImg) {
        var _mImg = create('img');
        _mImg.id = 'iNavImgM';
        _mImg.src = mImg;
        _mImg.className = 'amc-i-nav-m-img';
        _mBox.appendChild(_mImg);
    }

    var _mTitleBox = create('div', 'amc-v-box', _mBox);
    _mTitleBox.id = 'iNavTitleBox';
    if (title) {
        // 防止icon被文案挤到最左边
        if (mImg) {
            _mTitleBox.style.maxWidth = window.innerWidth / 2 - 50;
        } else {
            _mTitleBox.className = 'amc-v-box amc-flex-1';
        }

        // 有subtitle时字体略小一些
        var fontClassName = subtitle ? 'amc-i-nav-m-text-with-subtitle' : 'amc-i-nav-m-text';
        var _title = create('label', fontClassName + ' amc-ellipsis', _mTitleBox);
        _title.innerText = title;
        _title.id = 'iNavTitle';
    }

    if (subtitle) {
        var _subTitle = create('label', 'amc-i-nav-subtitle amc-ellipsis', _mTitleBox);
        _subTitle.innerText = subtitle;
        _subTitle.id = 'iNavSubtitle';
    }

    var _rBox = create('div');
    _rBox.className = 'amc-i-nav-r-box';
    _rBox.id = 'iNavBoxR';

    if (rImg) {
        var _rImg = create('img');
        _rImg.className = 'amc-i-nav-r-img';
        _rImg.id = 'iNavImgR';
        _rImg.src = rImg;
        _rBox.appendChild(_rImg);
        amc.fn.pressableElement(_rBox, _rImg);
        _rBox.onclick = onRight;
    } else if (rTxt) {
        var _rTxt = create('label');
        _rTxt.className = 'amc-i-nav-r-text amc-flex-1 amc-ellipsis';
        _rTxt.id = 'iNavTxtR';
        _rTxt.innerText = rTxt;
        _rBox.appendChild(_rTxt);
        amc.fn.pressableElement(_rBox, _rTxt);
        _rBox.onclick = onRight;
    }

    _iNav.appendChild(_lBox);
    _iNav.appendChild(_mBox);
    _iNav.appendChild(_rBox);

    return _iNav;
};



/**
 * 根据key值获取国际化文案对应的value值
 * @param key(string). 国际化语言的key值,形如:{{key}}/key
 * @return (string) 返回国际化语言对应的value
 */
amc.fn.i18nValueForKey = function(key) {
    if (!key) {
        return key;
    }

    var tmpKey = key;
    if (key.indexOf('{{') < 0) {
        tmpKey = '{{' + key + '}}';
    }

    var global = document;
    global.i18nInvisibleTag = global.i18nInvisibleTag || amc.fn.create('label');

    if (!global.i18nInvisibleTag) {
        return key;
    }

    // 将key值赋值给UI控件时进行国际化语言替换
    document.i18nInvisibleTag.innerText = tmpKey;

    return global.i18nInvisibleTag.innerText;
};

/*
 * 进行占位符替换，将字符串中的#key1#替换成对应的value1.
 * @param str(string) 被替换的字符串
 * @param keyValue(object/string) 键值对/单值:
 * 1. 例: function('总数为#count#，总金额为#amount#', {count: 20, amount: '100.0'});
 * 2. 如果只有一个替换值的形式可以简写成: function('总数为#count#', 20);
 */
amc.fn.keyValueReplace = function(str, keyValue) {

    if (!str || !keyValue) {
        return str;
    }

    // 如果只有一个替换值时，keyValue可以简写成该值，此时被替换字符为#val#
    if (!amc.fn.isObject(keyValue)) {
        keyValue = { 'val': keyValue };
    }

    for (var key in keyValue) {
        str = str.replace(new RegExp('#' + key + '#', "g"), keyValue[key]);
    }

    return str;
};

/*
 * 进行占位符替换
 * @param key(string) 国际化语言的key值
 */
amc.fn.i18nPlaceholderReplace = function(key, keyValue) {
    return amc.fn.keyValueReplace(amc.fn.i18nValueForKey(key), keyValue);
};

/**
 * 从中间将文字截断(从钱包10.0.0， SDK 10.6.9开始支持)
 * str(string) 需要截断的字符串
 * headNum(number) 头部保留字数
 * tailNum(number) 末尾保留字数
 */
amc.fn.stringTruncate = function(str, headNum, tailNum) {
    if (!str || (!headNum && !tailNum)) {
        return str;
    }

    if (isNaN(headNum) || headNum < 0) {
        headNum = 0;
    }

    if (isNaN(tailNum) || tailNum < 0) {
        tailNum = 0;
    }

    if (headNum + tailNum >= str.length) {
        return str;
    }

    var head = str.substr(0, headNum);
    str = str.substr(headNum, str.length - headNum);

    var tail = '';
    tail = str.substr(str.length - tailNum);

    return head + '...' + tail;
};

/***
 * 安全invoke接口，对invoke接口的封装，具体为iOS可以直接invoke，安卓需要通过setTimeout
 * @since (5.4.8)
 */
amc.fn.safeInvoke = function(funcName, params, callback) {
    var invoke = function() {
        document.invoke(funcName, param, callback);
    };
    if (amc.isIOS) {
        invoke();
    } else {
        // 页面未渲染完成，需要setTimeout更久
        setTimeout(invoke, pageInited ? 1 : 20);
    }
}

/**
 * 渲染本地模板
 * @param  {string} tplId   模板ID, 如:QUICKPAY@open-pwd-check-flex
 * @param  {object} tplMeta 模板信息，由服务端下发
 * @param  {object} data    模板数据, 在目标模板内通过rpc关键词访问
 */
amc.fn.renderLocalTemplate = function(tplId, tplMeta, data) {
    if (!tplId) {
        return;
    }

    var obj = {};
    obj['action'] = { 'name': 'loc:bnvb' };
    obj['param'] = {};

    obj['param']['tplid'] = tplId;
    obj['param']['tpl'] = tplMeta;
    obj['param']['data'] = data;

    document.submit(obj);
};

amc.zhugeConfig = {};
amc.zhugeLogData = {};
amc.zhugeSpmIdLogData = {};

amc.fn.loadZhugeConfig = function() {
    // 诸葛数据默认为zgConfig字段
    if (amc.rpcData.zhugeConfig && amc.fn.isObject(amc.rpcData.zhugeConfig)) {
        amc.zhugeConfig = amc.rpcData.zhugeConfig;
        var tmpZgConfig = amc.zhugeConfig;
        for (var zgkey in tmpZgConfig) {
            var tmpConfig = tmpZgConfig[zgkey];
            if (tmpConfig) {
                var tmpLogData = tmpConfig['logData'];
                if (tmpLogData && amc.fn.isObject(tmpLogData)) {
                    var tmpSpmId = tmpConfig['spmId'];
                    if (tmpSpmId && amc.fn.isArray(tmpSpmId)) {
                        tmpSpmId.forEach(function(tmpSpmIdKey){
                            amc.zhugeSpmIdLogData[tmpSpmIdKey] = JSON.parse(JSON.stringify(tmpLogData));
                        });
                    } else {
                        amc.zhugeLogData = tmpLogData;
                    }
                }
            }
        }
    }
};

amc.fn.getZhugeConfigAction = function(name) {
    if (name) {
        var zhugeConfig = amc.zhugeConfig[name];
        if (zhugeConfig) {
            return zhugeConfig['action'] || {};
        } else {
            return {};
        }
    } else {
        return {};
    }
};

amc.fn.loadZhugeConfig();

// spm命名空间
amc.spm = {};
amc.spm.bizCode = 'pay';

// 实验命名空间
amc.ab = {};
amc.ab.expId = flybird.local.expId;
amc.ab.expInfo = flybird.local.expInfo;
amc.ab.expLog = flybird.local.expLog;

// 当前渲染的模板信息
amc.tplInfo = {};
try {
    var tplInfoStr = flybird.local.tplInfo;
    if (tplInfoStr) {
        var tplInfo = JSON.parse(tplInfoStr);
        if (tplInfo) {
            amc.tplInfo.fbVer = tplInfo.tplVersion;
            amc.tplInfo.fbId = tplInfo.tplId;
            amc.tplInfo.fbTime = tplInfo.time;
        }
    }
} catch (ignored) {}

/**
 * 收银台链路的spm埋点(收银自己上传, 不走钱包通道)
 * @param  {string} type      'click'/'exposure'/'create'/'destroy' 与收银台native约定的类型
 * @param  {string} actionId  钱包埋点行为ID，由PD申请
 * @param  {string} spmId     spmId
 * @param  {object} param4Map 扩展参数,最多4个
 */
amc.fn.mqpSpm = function(type, actionId, spmId, param4Map) {
    if (!type || !spmId) {
        return;
    }

    // 默认给所有埋点添加tradeNo和uid
    if (actionId != 'create' && actionId != 'click') {
        param4Map = param4Map || {};

        flybird.local = flybird.local || {};
        if (!param4Map['tradeNo']) {
            param4Map['tradeNo'] = flybird.local.tradeNo || amc.rpcData.tradeNo;
        }

        if (!param4Map['uid']) {
            param4Map['uid'] = flybird.local.uid || amc.rpcData.uid;
        }
    }

    // 合并诸葛logdata进spm扩展字段
    var spmLogData = amc.zhugeSpmIdLogData[spmId];
    if (!spmLogData) {
        spmLogData = amc.zhugeLogData;
    }
    for (var spmZgkey in spmLogData) {
        param4Map[spmZgkey] = spmLogData[spmZgkey];
    }

    var expLogObj = amc.ab.expLog;
    if (expLogObj) {
        for (var spmExpKey in expLogObj) {
            param4Map[spmExpKey] = expLogObj[spmExpKey];
        }
    }

    var tplInfoObj = amc.tplInfo;
    if (tplInfoObj) {
        for (var spmExpKey in tplInfoObj) {
            param4Map[spmExpKey] = tplInfoObj[spmExpKey];
        }
    }

    // 调用埋点之前需要先创建
    if (actionId !== 'create' && !amc.spmPageCreated) {
        amc.fn.spmPageCreate();
    }

    try {
        var obj = {
            type: type,
            spmId: spmId,
            param4: param4Map != null ? JSON.stringify(param4Map) : '',
            actionId: actionId || 'click',
            logLevel: '2',
            bizCode: amc.spm.bizCode
        };

        amc.fn.feedback('mqpspm', obj);
    } catch (e) {

    }
};

/**
 * 点击埋点
 */
amc.fn.spmClick = function(spmId, param4Map) {
    amc.fn.mqpSpm('click', 'clicked', spmId, param4Map);
};

/**
 * 曝光埋点
 * 备注: 必须先调用spmPageCreate (参考统一结果页)
 */
amc.fn.spmExposure = function(spmId, param4Map, doNotResume) {
    if (!doNotResume) {
        amc.spmExposured[spmId] = param4Map || {};
    }
    amc.fn.mqpSpm('exposure', 'exposure', spmId, param4Map);
};

/**
 * 曝光埋点恢复
 */
amc.fn.spmExposureResume = function() {
    for (var spmId in amc.spmExposured) {
        amc.fn.mqpSpm('exposure', 'exposure', spmId, amc.spmExposured[spmId]);
    }
};

/**
 * 创建埋点
 * 备注: 后续埋点均依赖该创建操作,spmId 为a.b
 */
amc.fn.spmPageCreate = function(spmId, param4Map) {
    if (amc.spmPageCreated) {
        return;
    }

    amc.spmPageDestroyed = false;
    amc.spmPageCreated = true;
    amc.fn.mqpSpm('create', 'pageMonitor', spmId, param4Map);
};

/**
 * 页面关闭埋点,spmId 为a.b
 */
amc.fn.spmPageDestroy = function(spmId, param4Map) {
    if (amc.spmPageDestroyed) {
        return;
    }
    amc.fn.mqpSpm('destroy', 'pageMonitor', spmId, param4Map);
    amc.spmPageDestroyed = true;
    amc.spmPageCreated = false;
};

/**
 * 弹出Native 弹窗
 *
 * @param  {Function} callback 回调函数，回调入参为点击index, 从0开始, 与传入btns数组序号一致
 * @param  {Array} btns 按钮文案数组
 * @param  {string} message 提示文案
 * 备注:  不支持<font color=''></font>标签 投放方需要注意
 */
amc.fn.nativeAlert = function(btns, callback, message) {
    if (!btns || !btns.length || !(typeof callback === 'function')) {
        return;
    }

    // Native弹窗配置信息
    var dialogConfig = {
        // title: title,   // 只有iOS支持
        message: message || ''
    };



    // 3个按钮时候，iOS显示顺序依次为: 按钮2、按钮3、按钮1(粗体)
    // 在这里进行一次转换

    var needResort = amc.isIOS && btns.length == 3;
    if (needResort) {
        var tmp = btns[2];
        btns[2] = btns[1];
        btns[1] = btns[0];
        btns[0] = tmp;
    }

    // 一个按钮
    if (btns.length >= 1) {
        dialogConfig['cancelButton'] = btns[0] || '{{cancel}}';
    }

    // 两个按钮
    if (btns.length >= 2) {
        dialogConfig['okButton'] = btns[1] || '{{confirm_btn}}';
    }

    // 三个按钮
    if (btns.length >= 3) {
        dialogConfig['otherButton'] = btns[2] || '其他';
    }

    var cbFunction = (function(result) {
        if (callback && result && !isNaN(result.index)) {
            var index = result.index;
            if (needResort) {
                if (index == 2) {
                    index = 1;
                } else if (index == 1) {
                    index = 0;
                } else {
                    index = 2;
                }
            }

            callback(index);

            // 安卓中可能会重复多次回调callback, 为避免发生重复操作, 调用一次之后清除回调
            callback = null;
        }
    });

    if (amc.isIOS) {
        document.invoke('alert', dialogConfig, cbFunction);
    } else {
        var obj = { 'action': { 'name': "loc:alert('" + JSON.stringify(dialogConfig) + "')" } };

        document.asyncSubmit(obj, cbFunction);
    }
};

/**
 * 获取本地缓存
 * @param cacheId 本地缓存的id
 * @param callback 本地缓存回调，结果为json对象
 * @param persist 缓存是否采用持久化，持久化缓存会进一步到磁盘查询
 */
amc.fn.fetchCache = function(cacheId, callback, persist) {
    document.invoke('loc', {
        'action': {
            'name': 'loc:cache',
            'params': {
                'type': 'get',
                'persist': persist,
                'zone': 'mqp_abtest_' + cacheId
            }
        }
    }, function(data) {
        setTimeout(function() {
            if (callback) {
                callback(data['mqp_abtest_' + cacheId] || {});
            }
        }, 1);
    });
};

/**
 * 写入本地缓存
 * @param cacheId 缓存索引Id
 * @param data 缓存数据
 * @param persist 是否持久化，如果为true，data将写入磁盘
 */
amc.fn.putCache = function(cacheId, data, persist) {
    document.invoke('loc', {
        'action': {
            'name': 'loc:cache',
            'params': {
                'type': 'set',
                'persist': persist,
                'zone': 'mqp_abtest_' + cacheId,
                'data': data
            }
        }
    }, function(result) {
        // ignore
    });
};

/**
 * 发送rpc请求到mobilecashier
 * @param action 请求action
 * @param params 业务参数
 * @param callback 返回结果
 */
amc.fn.fetchCashierRpc = function(action, params, callback) {
    document.invoke('rpc', {
        'type': 'json',
        'operationType': 'alipay.msp.cashier.dispatch.json',
        'action': {
            'name': action,
            'params': params
        }
    }, function(result) {
        setTimeout(function() {
            if (!result || result.errCode != 0) {
                return;
            }

            if (result.params && result.params.data) {
                callback(result.params.data);
            }
        }, 1);
    });
};

/**
 * 获取当前页面所处交易的UserId
 */
amc.fn.getUserId = function() {
    return flybird.local.uid || amc.rpcData.uid;
};

/**
 * 获取当前页面所处交易的订单号
 */
amc.fn.getTradeNo = function() {
    return flybird.local.tradeNo || amc.rpcData.tradeNo;
};

/*
 * 用于鸟巢页面感知页面的生命周期，页面中可覆盖此方法
 * 备注: 每次从被遮挡到显示都会回调
 * 仅适用于iOS，从SDK 10.7.3，钱包10.0.8开始支持
 */
document.viewDidAppear = function() {};

/**
 * 配置类生命周期方法，native主动调用，获取导航栏，状态栏颜色
 * 返回空，所有设定使用默认的：白色导航栏，dark状态栏
 * 手势返回默认是 disabled，每个页面自己控制是否启用
 * {
 *   navi: {
 *     naviBarColor: 导航栏颜色；安卓 iOS通用
 *     statusBarStyle: 状态栏style；dark/light对应iOS中状态栏style；
 *                   安卓目前没有修改状态栏的需求，后续有需求再适配
 *   },
 *   gestureBack: {
 *     mode: gesture
 *   }
 * }
 */
amc.fn.docConfig = function() {
    return JSON.stringify({
        gestureBack: {
            mode: amc.iOSGestureBackMode
        }
    });
};

/**
 * iOS 收银台自己的手势返回触发回调
 */
amc.fn.onGestureBack = function() {
    if (amc.iOSGestureBackCallback) {
        amc.iOSGestureBackCallback();
    }
};

/**
 * 发送native通知
 * @param {string} name 通知名称
 * @param {map} params 通知携带的参数，单层map
 */
amc.fn.postNotification = function(name, params) {
    document.submit({
        action: {
            name: 'loc:postNotification',
            params: {
                notifyName: name,
                data: params || {}
            }
        }
    });
};

amc.fn.processImageUrl = function(url) {
    if (amc.fn.isString(url)) {
        return url.replace(/^local:/, amc.path);
    }
};

/**
 * 检查指令是否可被调用，>=10.1.58
 * @param {string} apiName 指令名称
 * @param {function} succCallback 可被调用callback
 * @param {function} failCallback 不可被调用callback
 */
amc.fn.checkApiInfo = function (apiName, succCallback, failCallback) {
    document.invoke('queryInfo', {
            'queryKey': 'apiInfo',
            'apiName': apiName
        }, function (info) {
            if (info.available) {
                succCallback && succCallback();
            } else {
                failCallback && failCallback();
            }
        }
    );
};

/*
 * 覆盖鸟巢提供的document.confirm方法
 * 参数参考: http://birdnest.alipay.net/books/js/#confirm.html
 * 本方法在钱包9.6.9之后才能支持
 */

(function() {
    // 备份鸟巢自己的接口
    document._confirm = document.confirm;
    document.confirm = function(btn, callback, doInvoke) {
        if (!btn || !callback ||
            !(typeof btn === 'object') ||
            !(typeof callback === 'function')) {
            return;
        }

        if (amc.isIOS || doInvoke) {
            document.invoke('alert', btn, function(result) {
                callback(result);
            });
        } else {
            var obj = { 'action': { 'name': "loc:alert('" + JSON.stringify(btn) + "')" } };
            document.asyncSubmit(obj, function(result) {
                callback(result);
            });
        }
    };
})();

/**
 * 注册Tracker
 */
(function(global) {
    function Tracker(pageName) {
        this.page = pageName;
    }
    Tracker.prototype = {
        _wrap: function(type, errMsg, callback) {
            var wrapper = function() {
                try {
                    callback && callback.apply(null, arguments);
                } catch (ex) {
                    amc.fn.logError(this.page + '|' + type + '|' + errMsg, 'ex:' + ex);
                }
            };
            return wrapper.bind(this);
        },
        onClick: function(errMsg, onclick) {
            return this._wrap('click', errMsg, onclick);
        }
    };
    global.Tracker = Tracker;
})(window);
