﻿//
// Created by Herb on 2019/1/10.
// Copyright (c) 2019 DingTalk. All rights reserved.
//

#ifndef GAEA_BASE_ASYNC_TASK_MANAGER_H_
#define GAEA_BASE_ASYNC_TASK_MANAGER_H_

#include <memory>
#include <chrono>
#include <functional>
#include "gaea/base/blocking_queue.h"
#include "gaea/base/macros.h"

namespace gaea {
namespace base {

/**
 * @class AsyncTask
 * @brief 异步任务 抽象类
 */
class AsyncTask {
 public:
  AsyncTask() {}
  virtual ~AsyncTask() {}

 public:
  virtual void Execute() = 0;
};

typedef std::shared_ptr<AsyncTask> AsyncTaskPtr;

class LambdaAsyncTask : public gaea::base::AsyncTask {
 public:
  LambdaAsyncTask(std::function<void()> func)
  : func_(func){}
  void Execute() {
    if (func_) {
      func_();
    }
  }
  
 private:
  std::function<void ()> func_;
};
/**
 * @class AsyncTaskManager
 * @brief 异步任务对象管理类
 */
class AsyncTaskManager {
 public:
  /**
   * @function AsyncTaskManager
   * @brief 管理类构造函数，默认最大缓冲管理 1024 个任务对象
   */
  AsyncTaskManager(size_t max_size = 1024);
  ~AsyncTaskManager();

 public:
  /**
   * @function Add
   * @brief 添加任务对象
   * @return true 添加成功; false 添加失败
   */
  bool Add(AsyncTaskPtr task);
  
  /**
   * @function Remove
   * @brief 删除任务对象
   * @return true 删除成功; false 删除失败
   */
  bool Remove(AsyncTaskPtr task);

  /**
   * @function Process
   * @brief 循环消费内部缓冲的任务对象，并执行任务块，直到清空缓冲队列
   */
  void Process();

  template <class Rep, class Period>
  void ProcessFor(const std::chrono::duration<Rep, Period>& duration) {
    if (duration <= std::chrono::milliseconds(0) || tasks_.Size() > 0) {
      Process();
      return;
    }
    tasks_.WaitFor(duration);
    Process();
  }

  /**
   * @function MaxSize
   * @brief 获取任务队列最大异步任务缓冲量
   */
  size_t MaxSize() const;
  
  /**
   * @function SetMaxSize
   * @brief 配置任务队列最大异步任务缓冲量
   */
  void SetMaxSize(size_t max_size);

  /**
   * @function SetEnableMaxSizeLimit
   * @brief 是否开启队列节点最大个数限制
   */
  void SetEnableMaxSizeLimit(bool enable) { tasks_.set_enable_max_size_limit(enable); }
  
  /**
   * @function Empty
   * @brief 判断当前任务队列是否为空
   * @return true 当前空队列; false 当前队列非空
   */
  bool Empty();
  
  /**
   * @function Size
   * @brief 当前任务队列缓冲任务个数
   * @return 返回当前队列中任务个数
   */
  size_t Size();
  
  /**
   * @function Clear
   * @brief 清空当前任务队列
   */
  void Clear();
  
  /**
   * @function Stop
   * @brief 复位队列缓冲，并通知退出逻辑
   */
  void Stop();

 private:
  BlockingQueue<AsyncTaskPtr> tasks_;
};

} // end of namespace base
} // end of namespace gaea

#endif /** GAEA_BASE_ASYNC_TASK_MANAGER_H_ */
