//
//  logging.h
//  Gaea
//
//  Created by jinxi on 4/21/16.
//  Copyright © 2016 jinxi. All rights reserved.
//

#ifndef GAEA_BASE_LOGGING_H_
#define GAEA_BASE_LOGGING_H_

#include "gaea/base/macros.h"

#ifdef GAEA_OS_WINDOWS
#include <direct.h>
#include <winsock2.h>
#include <windows.h>
#endif

#ifdef _WIN32
#if !defined(IOVEC_HAS_DEFINED)
#define IOVEC_HAS_DEFINED  1
struct iovec {
	void* iov_base;
	size_t iov_len;
};
#endif
#ifndef __func__
#define __func__ __FUNCTION__
#endif
#else
#include <sys/uio.h>
#endif

#include <set>
#include <sstream>
#include <string>
#include <vector>
#include <functional>
#include "gaea/base/macros.h"
#include "gaea/base/string_util.h"

namespace gaea {
namespace base {

class Properties;
class Logger;
class Splitter;
typedef std::function<void (int level, struct timeval*, const char*, size_t)> CustomLogHanlder;
class LogWriter {
 public:
  LogWriter() {}
  virtual ~LogWriter() {}

 public:
  virtual int Trace(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Debug(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Info(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Notice(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Warn(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Error(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
  virtual int Fatal(const std::string& message, const std::string& file, int lineno, const std::string& function) = 0;
};

class DefaultLogWriter : public LogWriter {
 public:
  DefaultLogWriter() {}
  ~DefaultLogWriter() {}

 public:
  int Trace(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Debug(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Info(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Notice(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Warn(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Error(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
  int Fatal(const std::string& message, const std::string& file, int lineno, const std::string& function) override;
};

class LogLayout;
class LogAppender;

class LogLayout {
 public:
  LogLayout() {}
  virtual ~LogLayout() {}

 public:
  virtual bool Init(Properties* properties) { return true; }
  virtual void FormatAndAppend(int level,
                               struct ::timeval* time,
                               const char* message,
                               size_t length,
                               const char* file,
                               int lineno,
                               const char* function) = 0;

  LogAppender* appender() const { return appender_; };
  void set_appender(LogAppender* appender) { appender_ = appender; }

 protected:
  LogAppender* appender_;
};

class LogAppender {
 public:
  LogAppender() {}
  virtual ~LogAppender() {}

 public:
  void set_name(const std::string& name) { name_ = name; }
  const std::string& name() const { return name_; }
  void set_layout(LogLayout* layout) { layout_ = layout; }
  LogLayout* layout() const { return layout_; }

  virtual bool Init(Properties* properties) { return true; }
  virtual void Write(struct timeval* time, const char* message, size_t length) = 0;
  virtual void WriteWithLevel(int level, struct timeval* time, const char* message, size_t length) { Write(time, message, length); };
  virtual void WriteV(struct timeval* time, const struct iovec* iov, int iovcnt);

 private:
  std::string name_;
  LogLayout* layout_;
};

class Logger {
 public:
  enum LoggerLevelType {
    kAllLevel = 0,
    kTraceLevel = 1,
    kDebugLevel = 2,
    kInfoLevel = 3,
    kNoticeLevel = 4,
    kWarnLevel = 5,
    kErrorLevel = 6,
    kFatalLevel = 7,
    kOffLevel = 8,
  };

  static uint32_t FromLevelString(const std::string& levelString);
  static const std::string ToLevelString(uint32_t level);

 public:
  Logger();
  ~Logger();

 public:
  void AddAppender(LogAppender* appender);
  const std::vector<LogAppender*>& appenders() const { return appenders_; }
  void set_log_level(uint32_t level) { log_level_ = level; }
  uint32_t log_level() { return log_level_; }
  bool IsEnabledFor(uint32_t level) const { return level >= log_level_; }
  
  Logger& set_unique_mark(const std::string& mark);
  const std::string& unique_mark() const { return unique_mark_; }
  
  void Trace(const std::string& message, const char* file, int lineno, const char* function) const;
  void Debug(const std::string& message, const char* file, int lineno, const char* function) const;
  void Info(const std::string& message, const char* file, int lineno, const char* function) const;
  void Notice(const std::string& message, const char* file, int lineno, const char* function) const;
  void Warn(const std::string& message, const char* file, int lineno, const char* function) const;
  void Error(const std::string& message, const char* file, int lineno, const char* function) const;
  void Fatal(const std::string& message, const char* file, int lineno, const char* function) const;

  void Log(
      uint32_t level, const char* message, size_t length, const char* file, int lineno, const char* function) const;

 private:
  std::string unique_mark_;
  std::vector<LogAppender*> appenders_;
  uint32_t log_level_;
};

class LoggerFactory {
 public:
  static LoggerFactory* GetInstance();
  static void SetInstance(LoggerFactory* instance);
  static Logger GetLogger(const std::string& category);

 private:
  static LoggerFactory* instance_;

 public:
  LoggerFactory() {}
  virtual ~LoggerFactory() {}

 public:
  /**
   * @function Init
   * @brief Properties 属性描述:
   *        以 key = "log4gaea.logger.xxx" 注册 xxx 日志模块, 注册 logger;
   *           value = "level, Appender0, Appender1" 的方式注册对应的 value;
   *        以 key = "log4gaea.appender.xxxApender" 注册对应 Appender 的属性;
   *           value = "xxx" 对应 appender 的值
   */
  virtual bool Init(Properties* properties) = 0;
  virtual Logger GetLoggerImpl(const std::string& category) = 0;
};

class GaeaLoggerFactory : public LoggerFactory {
 public:
  GaeaLoggerFactory();
  ~GaeaLoggerFactory();

  bool Init(Properties* properties) override;
  Logger GetLoggerImpl(const std::string& category) override;

  virtual void AddCustomAppender(const std::string& appenderName,
                                 gaea::base::CustomLogHanlder func);
  
 private:
  LogAppender* CreateAppender(Properties* properties, const std::string& appenderName);

 private:
  Splitter* comma_splitter_;
  Splitter* dot_splitter_;
  std::set<LogAppender*> appenders_;
  std::set<LogLayout*> layouts_;
  std::map<std::string, Logger> loggers_; /** 模块名 对应 Appender  */
  std::map<std::string, CustomLogHanlder> custom_appender_map_;
};
  
#define GAEA_LOG_DEBUG_ENABLE(logger)   logger.IsEnabledFor(gaea::base::Logger::kDebugLevel)
#define GAEA_LOG_INFO_ENABLE(logger)    logger.IsEnabledFor(gaea::base::Logger::kInfoLevel)
#define GAEA_LOG_WARN_ENABLE(logger)    logger.IsEnabledFor(gaea::base::Logger::kWarnLevel)
#define GAEA_LOG_ERROR_ENABLE(logger)   logger.IsEnabledFor(gaea::base::Logger::kErrorLevel)

#define GAEA_LOG_TRACE(logger, logEvent)                   \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kTraceLevel)) { \
      break;                                                 \
    }                                                        \
    std::ostringstream oss;                                  \
    oss << logger.unique_mark() << "| " << logEvent ;    \
    logger.Trace(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_DEBUG(logger, logEvent)                   \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kDebugLevel)) { \
      break;                                                 \
    }                                                        \
    std::ostringstream oss;                                  \
    oss << logger.unique_mark() << "| " << logEvent ;    \
    logger.Debug(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_INFO(logger, logEvent)                   \
  do {                                                      \
    if (!logger.IsEnabledFor(gaea::base::Logger::kInfoLevel)) { \
      break;                                                \
    }                                                       \
    std::ostringstream oss;                                 \
    oss << logger.unique_mark() << "| " << logEvent ;   \
    logger.Info(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_NOTICE(logger, logEvent)                   \
  do {                                                        \
    if (!logger.IsEnabledFor(gaea::base::Logger::kNoticeLevel)) { \
      break;                                                  \
    }                                                         \
    std::ostringstream oss;                                   \
    oss << logger.unique_mark() << "| " << logEvent ;     \
    logger.Notice(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_WARN(logger, logEvent)                   \
  do {                                                      \
    if (!logger.IsEnabledFor(gaea::base::Logger::kWarnLevel)) { \
      break;                                                \
    }                                                       \
    std::ostringstream oss;                                 \
    oss << logger.unique_mark() << "| " << logEvent ;   \
    logger.Warn(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_ERROR(logger, logEvent)                   \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kErrorLevel)) { \
      break;                                                 \
    }                                                        \
    std::ostringstream oss;                                  \
    oss << logger.unique_mark() << "| " << logEvent ;    \
    logger.Error(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_FATAL(logger, logEvent)                   \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kFatalLevel)) { \
      break;                                                 \
    }                                                        \
    std::ostringstream oss;                                  \
    oss << logger.unique_mark() << "| " << logEvent ;    \
    logger.Fatal(oss.str(), __FILE__, __LINE__, __func__);   \
  } while (0)

#define GAEA_LOG_TRACE_STR(logger, logEvent)               \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kTraceLevel)) { \
      break;                                                 \
    }                                                        \
    logger.Trace(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_DEBUG_STR(logger, logEvent)               \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kDebugLevel)) { \
      break;                                                 \
    }                                                        \
    logger.Debug(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_INFO_STR(logger, logEvent)               \
  do {                                                      \
    if (!logger.IsEnabledFor(gaea::base::Logger::kInfoLevel)) { \
      break;                                                \
    }                                                       \
    std::ostringstream oss;                                 \
    oss << logEvent;                                        \
    logger.Info(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_NOTICE_STR(logger, logEvent)               \
  do {                                                        \
    if (!logger.IsEnabledFor(gaea::base::Logger::kNoticeLevel)) { \
      break;                                                  \
    }                                                         \
    logger.Notice(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_WARN_STR(logger, logEvent)               \
  do {                                                      \
    if (!logger.IsEnabledFor(gaea::base::Logger::kWarnLevel)) { \
      break;                                                \
    }                                                       \
    logger.Warn(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_ERROR_STR(logger, logEvent)               \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kErrorLevel)) { \
      break;                                                 \
    }                                                        \
    logger.Error(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)

#define GAEA_LOG_FATAL_STR(logger, logEvent)               \
  do {                                                       \
    if (!logger.IsEnabledFor(gaea::base::Logger::kFatalLevel)) { \
      break;                                                 \
    }                                                        \
    logger.Fatal(logEvent, __FILE__, __LINE__, __func__);    \
  } while (0)
  
} // end of namespace base
} // end of namespace gaea

#endif /* GAEA_BASE_LOGGING_H_ */
