﻿//
//  string_util.h
//  Atlas
//
//  Created by 金喜 on 4/21/16.
//  Copyright © 2016 金喜. All rights reserved.
//

#ifndef GAEA_BASE_STRING_UTIL_H_
#define GAEA_BASE_STRING_UTIL_H_

#include <cstdint>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <vector>
#include <list>

#include "gaea/base/macros.h"

namespace gaea {
namespace base {

typedef std::vector<std::string> Strings;
typedef std::list<std::string> StringList;
typedef std::set<std::string> StringSet;
typedef std::map<std::string, std::string> StringMap;
typedef std::map<std::string, int> StringIntMap;

class StringUtil {
 public:
  /**
   * 转换成小写字母，仅支持ascii；输入必须是UTF-8编码字符串；
   */
  static std::string LowerCaseCopy(const std::string& val);
  /**
   * 转换成大写字母，仅支持ascii；输入必须是UTF-8编码字符串；
   */
  static std::string UpperCaseCopy(const std::string& val);
  
  /**
   * 转换成小写字母，仅支持ascii；输入必须是UTF-8编码字符串；
   * 原地修改参数字符串，输入参数为null时不做任何处理
   */
  static void LowerCase(std::string* val);
  /**
   * 转换成大写字母，仅支持ascii；输入必须是UTF-8编码字符串；
   * 原地修改参数字符串，输入参数为null时不做任何处理
   */
  static void UpperCase(std::string* val);

  /**
   * 去除首位空白字符，返回结果字符串的拷贝
   * 注意：空白字符指 std::isspace返回true的字符串；暂不支持不同locale的空白字符，且依赖启动时设置set locale 为 "C"
   */
  static std::string TrimCopy(const std::string& val);
  /**
   * 去除开头的空白字符，返回结果字符串的拷贝
   * 注意：空白字符指 std::isspace返回true的字符串；暂不支持不同locale的空白字符，且依赖启动时设置set locale 为 "C"
   */
  static std::string TrimLeftCopy(const std::string& val);
  /**
   * 去除末尾的空白字符，返回结果字符串的拷贝
   * 注意：空白字符指 std::isspace返回true的字符串；暂不支持不同locale的空白字符，且依赖启动时设置set locale 为 "C"
   */
  static std::string TrimRightCopy(const std::string& val);

  /*
   TODO: 支持非Copy的Trim操作
  static void Trim(std::string* val);
  static void TrimLeft(std::string* val);
  static void TrimRight(std::string* val);
   */

  /**
   * 判断是否是有效的数字，
   * 支持：整数（含负数）、浮点数
   * 不支持：超大数字检测，超出double/int64_t范围和精度的数字检测暂不考虑支持
   */
  static bool IsNumber(const std::string& val);
  
  /**
   * 判断 字符串是否 以某个特定的prefix开头
   */
  static bool StartsWith(const std::string& val, const std::string& prefix);
  /**
   * 判断 字符串是否 以某个特定的suffix结尾
   */
  static bool EndsWith(const std::string& val, const std::string& suffix);

  /**
   * 字符串转成int32的整数，不做有效性校验
   * TODO: 增加 bool SafeToInt32(const string &val, int32_t *) 函数，输入无效时，返回false
   */
  static int32_t ToInt32(const std::string& val);
  /**
   * 字符串转成int64的整数，不做有效性校验
   * TODO: 增加 bool SafeToInt64(const string &val, int64_t *) 函数，输入无效时，返回false
   */
  static int64_t ToInt64(const std::string& val);
  /**
   * 字符串转成double的整数，不做有效性校验
   * TODO: 增加 bool SafeToDouble(const string &val, double *) 函数，输入无效时，返回false
   */
  static double ToDouble(const std::string& val);

  /**
   * 返回 "true" / "false" 小写字符串
   */
  static std::string ToString(bool val);

  /**
   * 返回对象参数类型对应的 ostream 输出表达式
   */
  template <typename Digits>
  static std::string ToString(Digits digits) {
    std::stringstream ss;
    ss << digits;
    return ss.str();
  }

  /**
   * 字符替换，并返回拷贝的数据，对输入字符串不做修改
   * TOOD: 增加Repalce（原地修改，返回void）
   * TOOD: 增加 子字符串 替换
   */
  static std::string ReplaceCopy(const std::string& val, char ch, char replacement);

  /**
   * 二进制数据转成Hex形式输出；
   * TODO: 对称的UnHexDump待开发
   */
  static std::string HexDump(const std::string& binary);
  
  /*
  static std::wstring Utf8ToWideString(const std::string& utf8_string);
  static std::string WideStringToUtf8(const std::wstring& wide_string);
  */
#ifdef _WIN32
  static std::wstring SysMultiByteToWide(const std::string& mb, int code_page);
  static std::string SysWideToMultiByte(const std::wstring& wide, int code_page);
  static std::string UTF8ToGBK(const std::string& utf8Str);
  static std::string GBKToUTF8(const std::string& gbkStr);
#endif // WIN32
};

template <typename Type>
struct StringTransform {
};

template <>
struct StringTransform<int32_t > {
    inline static int32_t From(std::string&& str) {
      return StringUtil::ToInt32(str);
    }
};

template <>
struct StringTransform<std::string > {
    inline static std::string&& From(std::string&& str) {
          return std::move(str);
    }
};

template<>
struct StringTransform<int64_t> {
  inline static int64_t From(std::string &&str) {
    return StringUtil::ToInt64(str);
  }
};

template<>
struct StringTransform<double> {
  inline static double From(std::string &&str) {
    return StringUtil::ToDouble(str);
  }
};

} // end of namespace base
} // end of namespace gaea

#endif /* GAEA_BASE_STRING_UTIL_H_ */
