//
//  config_service.h
//  UniverConfig
//
//  Created by Herb on 2019/1/9.
//  Copyright © 2019年 DingTalk. All rights reserved.
//

#ifndef GAEA_CONFIG_SERVICE_H_
#define GAEA_CONFIG_SERVICE_H_

#include <memory>
#include <map>
#include <mutex>
#include <string>
#include <thread>
#include <future>
#include <chrono>

#include "gaea/base/async_task_manager.h"
#include "gaea/base/error_result.h"
#include "gaea/base/logging.h"
#include "gaea/base/singleton.h"
#include "gaea/config/config_service_delegate.h"
#include "gaea/config/config_service_interface_delegate.h"
#include "gaea/config/user_info_model.h"
#include "gaea/interface/http/http_interface.h"
#include "gaea/interface/database/kv_database_interface.h"
#include "gaea/interface/trace/trace_interface.h"


#if DEBUG
  class ConfigServiceTest;
#endif

namespace gaea {
namespace config {
  
/**
 * 配置服务错误码分类
 * */
enum ErrorCode {
  kSuccess = 0,
  kInvalidParam,
  kInitialDatabaseFailed,
  kDatabaseUpdateFailed,
  kConfigDataInvalid,
  kDownloadModulesConfigFailed,
  kLackModuleConfig,
  kDownloadConfigFailed,
  kConfigServiceStop,
  kServiceUpdating,
  kConfigDataException,
  kLocalStorageException,
  kSwitchConfigFailed,
  kLocalDownloadWithoutImple,
  kDownloadInvaildParam,
};
const std::string TransforErrCode2Descript(int code);

class ConfigDownload;

class ConfigCache;
typedef std::shared_ptr<ConfigCache> ConfigCachePtr;

class ConfigServiceInterface;
typedef std::shared_ptr<ConfigServiceInterface> ConfigServiceInterfacePtr;

/**
 * 配置服务类
 * */
class ConfigService
  : public gaea::base::Singleton<ConfigService>,
    public ConfigServiceInterfaceDelegate {

#ifdef GAEA_CONFIG_TEST_ENABLED
  friend ConfigServiceTest;
#endif
  friend class ConfigDownload;

 public:
  ConfigService();
  virtual ~ConfigService();
  
  void set_delegate(ConfigServiceDelegatePtr delegate) { delegate_ = delegate; }
  ConfigServiceDelegatePtr delegate() const { return delegate_; }
  
  bool optimise_start() { return optimise_start_; }
  void set_optimise_start(bool enable_optimise) { optimise_start_ = enable_optimise; }
  virtual bool GetStartOptimiseSwitch() override { return optimise_start(); }
  
  void set_user_info(const UserInfoModel& user_info) { user_info_ = user_info; }
  virtual const UserInfoModel& user_info() override { return user_info_; }
  
  gaea::service::KvDatabaseInterfacePtr database() const { return database_; }
  void set_database(gaea::service::KvDatabaseInterfacePtr database) {
      database_ = database;
  }
      
  void set_enable_database(bool enable_database);
  
  void set_storage_root_path(const std::string& path);
  const std::string storage_root_path() const { return storage_root_path_; }
  
  void set_http_implement(gaea::service::HttpInterface* http_implement);
      
  void set_trace_service(gaea::service::TraceInterfacePtr trace_service);
  
  /**
   * @function Start
   * @brief 安装启动配置服务&初始化
   *        可以在 ConfigServiceDelegate::OnStartedConfigService
   *        代理接口中注册实现服务启动状态回调
   **/
  void Start();
  
  /**
  * @function SyncStart
  * @brief 安装启动配置服务&初始化
  *        同步启动方式
  **/
  void SyncStart();

  /**
   * @function Shutdown
   * @brief 关闭服务
   **/
  void Shutdown();

  /**
   * @function GetConfig
   * @brief 获取配置数据对外接口，目前提供两个接口，分别获取字符串和bool值的捞取数据接口
   * @param module 模块名
   * @param key 关键 key 值
   * @param default_value 对应 key 默认值
   * @return
   *   0 设置成功;
   *  <0 配置失败;
   * 0. 如果引入fastconfig，则fastconfig将会支持高优先级紧急止血的功能，优先返回
   * 1. 优先取人角色上是否有值，有则返回
   * 2.1. 有一个企业为true就认为是true
   * 2.2. 有值但值为false返回false~
   * 2.3. 一个都没有返回则继续检查
   * 3.1. 判定百分比（用户id百分比或者企业id百分比）是否命中，命中则返回
   * 3.2. 不命中则最终使用default_value
   * */

  bool GetGraySwitch(
      const std::string& module,
      const std::string& key,
      const bool default_value,
      bool* is_hit = nullptr);
  
  bool GetGraySwitchWithOrgId(
      const std::string& module,
      const std::string& key,
      const bool default_value,
      const int64_t orgid,
      bool* is_hit = nullptr);
  
  /*
   * 0. 如果引入fastconfig，则fastconfig将会支持高优先级紧急止血的功能，优先返回
   * 1. 优先取人角色上是否有值，有则返回
   * 2.1. 如果有值的，返回的是企业id从小到达排序后第一个有值的value返回
   * 2.2. 如果有值，但没有和default_value不一样的值，则返回default_value
   * 2.3  如果一个有效值都没有则继续检查
   * 3.1. 判定百分比（用户id百分比或者企业id百分比）是否命中，命中则返回
   * 3.2. 不命中则最终使用default_value
   *
   */
  std::string GetString(
      const std::string& module,
      const std::string& key,
      const std::string& default_value,
      bool* is_hit = nullptr);
  
  std::string GetStringWithOrgId(
      const std::string& module,
      const std::string& key,
      const std::string& default_value,
      const int64_t orgid,
      bool* is_hit = nullptr);
  
  /**
   * @function UpdateConfig
   * @brief 触发配置服务检查本地配置与远端配置接口
   */
  void UpdateConfig(int64_t index_version = 0, const std::string& index_url = "");

  /**
  * @function AsyncUpdateSyncData
  * @brief 异步更新单个配置，结果通过callback确定
  */
  void AsyncUpdateSyncData(
      const std::string& sync_data,
      std::function<void(const gaea::base::ErrorResult& result)> callback);
  
  /**
  * @function AsyncUpdateSyncData
  * @brief 异步更新一组配置，结果通过callback确定，只触发一次，
           只要有一个更新失败，则callback的内容就是失败
  */
  void AsyncUpdateSyncData(const std::vector<std::string>& sync_data_vector,
                           std::function<void(const gaea::base::ErrorResult& result)> callback);
  
  /**
  * @function GetConfQueryCondition
  * @brief TooLoog2时获取最新版本号信息，包含个人、企业、全局，
           然后将对象序列化得到字符串结果，通过指针传出；
           解析出来的版本号如果为0，代表该类型最高版本号获取失败
  */
  void GetConfQueryCondition(std::vector<int64_t> org_ids, std::string* pack_latest_version);
  
  /**
  * @function GetLatestGlobalSettingVersion
  * @brief TooLoog2时获取最新全局类型的版本号，返回为0表示获取失败
  */
  const int64_t GetLatestGlobalSettingVersion();
  
  //返回为0表示查找失败
  int64_t GetIndexVersion();
      
  void CommitServiceStartStatus(const std::string& service_type,
                                int error_code,
                                const std::string& error_msg,
                                const std::string& is_started);
  
 private:
  /**
   * @function UpdateConfig
   * @brief 校验当前主配置信息是否发生改变，内部以异步的方式触发配置信息校验
   * @param callback 外部传入回调
   */
  void UpdateConfig(
      std::function<void(
          int error_code,
          const std::string& error_reason)> callback);
  
  /**
   * @function InnerGetConfig
   * @brief 内部接口，内部统一获取数据接口
   * @param module 目标模块
   * @param key 目标关键字
   * @param target_value 返回的数据值
   * @return
   *   true 得到目标关键字数据
   *   false 未得到关键字数据
   */
  bool InnerGetConfig(
      const std::string& module,
      const std::string& key,
      std::string* target_value,
      int64_t orgid);
  
  /**
   * @function processTasks
   * @brief 处理异步任务
   * */
  void ProcessTasks();
  
  /**
   * @brief 注册配置服务
   */
  ConfigServiceInterfacePtr GetService(int service_id);
  void RegisterServer(ConfigServiceInterfacePtr server);
  std::map<int, ConfigServiceInterfacePtr>* mutable_services_map() {
    return &services_map_;
  }
  
  /**
   * @brief 初始化数据库
   */
  bool OpenDatabase();
  void CloseDatabase();
  
  
  /** 内部代理接口 */
  virtual int64_t GetRemoteIndexVersion() override;
  
  virtual const std::string GetRemoteIndexDownloadUrl() override;
  
  virtual void OnStarted(
      int service_id,
      int code,
      const std::string& error_msg) override;
      
  void OnSyncStarted(int service_id,
                     int code,
                     const std::string& error_msg);
  
  virtual void OnUpdated(
      int service_id,
      int code,
      const std::string& error_msg,
      const std::multimap<std::string, std::string>& change_list) override;
  
  void PostAsyncTask(gaea::base::AsyncTaskPtr task);
  
 private:
  /**
   * 日志输出对象
   * */
  gaea::base::Logger logger_;
  
  bool optimise_start_;
  
  /**
   * 用户数据信息
   */
  UserInfoModel user_info_;
  
  /**
   * 外部代理
   * */
  ConfigServiceDelegatePtr delegate_;
  
  /**
   * 配置服务当前状态
   */
  std::mutex service_mutex_;

  /**
   * 内部异步任务管理
   * */
  std::shared_ptr<gaea::base::AsyncTaskManager> tasks_mananger_;
  std::thread* task_mananger_thread_;
  
  /**
   * 服务集合，将任务优先级和对应的服务进行绑定，建立服务映射表
   */
  std::map<int, ConfigServiceInterfacePtr> services_map_;
  
  /**
   * 配置缓存
   */
  ConfigCachePtr cache_;
  gaea::service::KvDatabaseInterfacePtr database_;
  std::string storage_root_path_;
  bool enable_database_;
  gaea::service::TraceInterfacePtr trace_service_;

  bool stop_;
  bool shutdown_complete_;
  std::promise<bool> all_start_promise_;
  std::future<bool> all_start_future_;
  bool sync_start_enable_;
  bool is_common_config_started_;
  bool is_fast_config_started_;
  bool has_common_config_try_start_;
  bool has_fast_config_try_start_;
  std::chrono::time_point<std::chrono::steady_clock> service_start_time_;
};

} // end of namespace config
} // end of namespace gaea

#endif /* GAEA_CONFIG_SERVICE_H_ */
