/*
* =====================================================================================
*
*       Filename:  model_msgpack_helper.h
*
*    Description:
*
*        Version:  1.0
*        Created:  2019-05-05
*       Revision:  none
*       Compiler:  gcc
*
*         Author:  jack.kj@alibaba-inc.com
*
* =====================================================================================
*/
#ifndef GAEA_IDL_MODEL_MSGPACK_HELPER_H_
#define GAEA_IDL_MODEL_MSGPACK_HELPER_H_

#include <cassert>
#include <functional>
#include <map>
#include <string>
#include <vector>
#include "gaea/idl/gaea_idl_define.h"
#include "gaea/idl/msgpack_helper.h"

namespace gaea {
namespace idl {

class BaseModel;

class ModelMsgpackHelper {
public:
  // List
  static bool Pack(const BaseModel& model, cmp_ctx_t *context);
  static bool Pack(int32_t value, cmp_ctx_t * cmp);
  static bool Pack(const std::string& value, cmp_ctx_t * cmp);
  static bool Pack(float value, cmp_ctx_t * cmp);
  static bool Pack(double value, cmp_ctx_t * cmp);
  static bool Pack(int64_t value, cmp_ctx_t * cmp);
  static bool Pack(char value, cmp_ctx_t * cmp);
  static bool Pack(bool value, cmp_ctx_t * cmp);

  template <typename Type>
  static bool Pack(const Type& value, std::string* out) {
    cmp_ctx_t cmp;
    cmp_init(&cmp, out, nullptr, MsgPackHelper::MsgpackStringWriter);

    return Pack(value, &cmp);
  }

  template <typename Type>
  static bool Unpack(const std::string& value, Type* out,
      bool* is_missing_fields = nullptr) {
    cmp_ctx_t cmp;
    gaea::idl::MsgPackHelper::MsgpackContext ctx{};
    gaea::idl::MsgPackHelper::MsgpackContext *context = &ctx;
    context->data = value.data();
    context->size = static_cast<int64_t>(value.length());
    context->begin = 0;
    context->cmp = &cmp;
    cmp_init(&cmp, (void *)context, MsgPackHelper::MsgpackContextReader, nullptr);
    bool is_nil = true;

    return Unpack(out, &cmp, &is_nil, is_missing_fields);
  }

  static bool Unpack(BaseModel* model, cmp_ctx_t *context, bool *is_nil,
      bool* is_missing_fields = nullptr);
  static bool Unpack(int32_t* value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(std::string* value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(float* value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(double * value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(int64_t * value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(char * value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  static bool Unpack(bool * value, cmp_ctx_t* context, bool *is_nil,
                     bool* is_missing_fields = nullptr);
  
  template <typename Key, typename Value>
  static bool Pack(const std::map<Key, Value> &map_value, cmp_ctx_t *cmp) {
    cmp_write_map(cmp, (uint32_t)map_value.size());
    for (auto iter : map_value) {
      if (!Pack(iter.first, cmp)) {
        return false;
      }

      if (!Pack(iter.second, cmp)) {
        return false;
      }
    }

    return true;
  }

  template <typename Key, typename Value>
  static bool Unpack(std::map<Key, Value>* map_value, cmp_ctx_t *cmp,
      GAEA_IDL_SAFE_POINT(bool) is_nil, bool* is_missing_fields) {
    // is_nil cannot be nil now，this method is only be invoked in Unpack of model,and it's member function Unpack
    assert(is_nil != nullptr);
    cmp_object_t object;
    if (!cmp_read_object(cmp, &object)) {
      return false;
    }

    if (cmp_object_is_nil(&object)) {
      *is_nil = true;
      return true;
    }

    uint32_t mapSize;
    if (!cmp_object_as_map(&object, &mapSize)) {
      return false;
    }

    *is_nil = false;
    for (uint32_t i = 0; i < mapSize; ++i) {
      // Read Key;
      Key key;
      bool is_key_nil = true;
      if (!Unpack(&key, cmp, &is_key_nil, is_missing_fields)) {
        return false;
      }

      Value value;
      bool is_value_nil = true;
      if (!Unpack(&value, cmp, &is_value_nil, is_missing_fields)) {
        return false;
      }

      if (!is_key_nil && !is_value_nil) {
        map_value->emplace(std::make_pair(std::move(key), std::move(value)));
      }
    }

    return true;
  }

  static bool GetSize(cmp_ctx_t *cmp, int *size, bool *is_nil);

  template <typename T>
  static bool Unpack(std::vector<T> *output, cmp_ctx_t *cmp, bool *is_nil,
                     bool *is_missing_fields) {
    int size = 0;
    if (!GetSize(cmp, &size, is_nil)) {
      return false;
    }

    if (*is_nil) {
      return true;
    }

    output->reserve(static_cast<unsigned long>(size));
    for (int i = 0; i < size; ++i) {
      T tmp;
      bool temp_is_nil = true;
      if (!Unpack(&tmp, cmp, &temp_is_nil, is_missing_fields)) {
        return false;
      }

      if (!temp_is_nil) {
        output->push_back(tmp);
      }
    }

    if (!output->empty()) {
      *is_nil = false;
    }

    return true;
  }

  template <typename T>
  static bool Pack(const std::vector<T> &values, cmp_ctx_t *cmp) {
    if (!cmp_write_array(cmp, (uint32_t)values.size())) {
      return false;
    }

    for (const auto &value : values) {
      if (!Pack(value, cmp)) {
        return false;
      }
    }

    return true;
  }

private:
  template <typename Type>
  static bool UnpackValue(Type *value, cmp_ctx_t *cmp, bool *is_nil);
};

} // namespace idl
} // namespace gaea

#endif
