//
//  translate_service_interface.hpp
//  gaea_interface
//
//  Created by Herb on 2020/4/20.
//  Copyright © 2020 DingTalk. All rights reserved.
//

#ifndef GAEA_SERVICE_TRANSLATE_SERVICE_INTERFACE_H_
#define GAEA_SERVICE_TRANSLATE_SERVICE_INTERFACE_H_

#include <functional>
#include <string.h>
#include "gaea/base/error_result.h"
#include "gaea/interface/base_interface.h"
#include "gaea/interface/file/file_defines.h"

namespace gaea {
namespace service {


typedef enum RecognitionType {
  kTranslateUndefine = 0,   /* 初始无效值 */
  kRecordTranslate = 1,     /* 语音录制翻译 */
  kMessageTranslate = 2,    /* 语音消息转文字 */
  kRealtimeTranslate = 3,   /* 实时语音翻译 */
} RecognitionType;


typedef struct AsrInfo {
  int32_t sentence_index;      // index of current sentence
  std::string text;            // the content of the sentence
  bool is_sentence_end;        // present current sentence reach the end
  int64_t sentence_time;       // the current timestamp from the stream begin
  int64_t sentence_begin_time; // the sentence begin timestamp from the stream begin
} AsrInfo;


class AsrProgress {
 public:
  AsrProgress() {}
  virtual ~AsrProgress();
  
  const AsrInfo content() const { return content_; }
  void set_content(const AsrInfo& info) { content_ = info; }
 protected:
  AsrInfo content_;
};
typedef std::shared_ptr<AsrProgress> AsrProgressPtr;


class AsrResult {
 public:
  AsrResult() {}
  virtual ~AsrResult();
  
  const AsrInfo content() const { return content_; }
  void set_content(const AsrInfo& info) { content_ = info; }
 protected:
  AsrInfo content_;
};
typedef std::shared_ptr<AsrResult> AsrResultPtr;


typedef enum AudioSampleRateType {
  kSampleRateUndefine = 0,
  kSampleRate8000 = 1,
  kSampleRate16000 = 2,
} AudioSampleRateType;


typedef enum AudioFmtType {
  kAudioFmtUndefine = 0,
  kAudioFmtPCM = 1,
  kAudioFmtOpus = 2,
} AudioFmtType;


typedef enum SyncType {
  kWukongNotify = 1,
  kSyncServer = 2,
  kImpaasSyncServer = 3,
} SyncType;


class BaseAsrTask {
 public:
  BaseAsrTask();
  virtual ~BaseAsrTask();
  
  RecognitionType task_type() const { return task_type_; }
  void set_task_type(RecognitionType type) { task_type_ = type; }
  
  const std::string& task_id() const { return task_id_; }
  
  SyncType sync_type() const { return sync_type_; }
  void set_sync_type(SyncType type) { sync_type_ = type; }
  
 protected:
  RecognitionType task_type_;
  std::string task_id_;
  SyncType sync_type_;
};
typedef std::shared_ptr<BaseAsrTask> BaseAsrTaskPtr;


class MsgTranslateTask : public BaseAsrTask {
 public:
  MsgTranslateTask();
  virtual ~MsgTranslateTask();
  
  const std::string& entity_id() const { return entity_id_; }
  void set_entity_id(const std::string& entity_id) { entity_id_ = entity_id; }
  
  int entity_type() const { return entity_type_; }
  void set_entity_type(int entity_type) { entity_type_ = entity_type; }
  
  const std::string& media_id() const { return media_id_; }
  void set_media_id(const std::string& media_id) { media_id_ = media_id; }
  
  int64_t duration() const { return duration_; }
  void set_duration(int64_t duration) { duration_ = duration; }
  
  int64_t sender_id() const { return sender_id_; }
  void set_sender_id(int64_t sender_id) { sender_id_ = sender_id; }

  // record and translate
  const std::string& mime_type() const { return mime_type_; }
  void set_mime_type(const std::string& mime_type) { mime_type_ = mime_type; }
  
  const std::string& biz_type() const { return biz_type_; }
  void set_biz_typ(const std::string& biz_type) { biz_type_ = biz_type; }
  
  gaea::service::AuthType auth_type() const { return auth_type_; }
  void set_auth_type(gaea::service::AuthType auth_type) { auth_type_ = auth_type; }

 private:
  std::string entity_id_;
  int entity_type_;
  std::string media_id_;
  int64_t duration_;
  int64_t sender_id_;
  std::string mime_type_;
  std::string biz_type_;
  gaea::service::AuthType auth_type_;
};
typedef std::shared_ptr<MsgTranslateTask> MsgTranslateTaskPtr;
typedef std::weak_ptr<MsgTranslateTask> MsgTranslateTaskWeakPtr;


class RealtimeTranslateTask : public BaseAsrTask {
 public:
  RealtimeTranslateTask();
  virtual ~RealtimeTranslateTask();
  
  const std::string& biz_type() const { return biz_type_; }
  void set_biz_type(const std::string& biz_type) { biz_type_ = biz_type; }
  
  const std::string& biz_id() const { return biz_id_; }
  void set_biz_id(const std::string& biz_id) { biz_id_ = biz_id; }
  
  bool is_first() const { return is_first_; }
  void set_is_first(bool is_first) { is_first_ = is_first; }
  
  const std::string& biz_transaction_id() const { return biz_transaction_id_; }
  void  set_biz_transaction_id(const std::string& biz_transaction_id) { biz_transaction_id_ = biz_transaction_id; }
  
  AudioSampleRateType sample_rate() const { return sample_rate_; }
  void set_sample_rate(AudioSampleRateType sample_rate) { sample_rate_ = sample_rate; }
  
  AudioFmtType input_format() const { return input_format_; }
  void set_input_format(AudioFmtType input_format) { input_format_ = input_format; }
  
  const std::string& extern_data() const { return extern_data_; }
  void  set_extern_data(const std::string& extern_data) { extern_data_ = extern_data; }
  
 private:
  std::string biz_type_;
  std::string biz_id_;
  bool is_first_;
  std::string biz_transaction_id_;
  AudioSampleRateType sample_rate_;
  AudioFmtType input_format_;
  std::string extern_data_;
};
typedef std::shared_ptr<RealtimeTranslateTask> RealtimeTranslateTaskPtr;
typedef std::weak_ptr<RealtimeTranslateTask> RealtimeTranslateTaskWeakPtr;


class AsrInterface : public BaseInterface {
 public:
  AsrInterface() {}
  virtual ~AsrInterface();
  
  virtual void CreateStream(
      BaseAsrTaskPtr task,
      std::function<void()> OnStart,
      std::function<void(const AsrProgress&)> OnProgress,
      std::function<void(const AsrResult&)> OnSuccess,
      std::function<void(const gaea::base::ErrorResult&)> OnFailure) = 0;

  virtual void CancelStream(const std::string& task_id) = 0;
  
  virtual void SendStreamData(
      const std::string& task_id,
      void* data,
      int64_t data_size,
      bool eof);
};
typedef std::shared_ptr<AsrInterface> AsrInterfacePtr;
typedef std::weak_ptr<AsrInterface> AsrInterfaceWeakPtr;


} // end of namespace service
} // end of namespace gaea
#endif /* GAEA_SERVICE_TRANSLATE_SERVICE_INTERFACE_H_ */
