//
//  file_service_interface.h
//  gaeaLwp
//
//  Created by guodi.ggd on 2019/6/2.
//  Copyright © 2019 DingTalk. All rights reserved.
//

#ifndef file_service_interface_h
#define file_service_interface_h

#include <memory>
#include <string>
#include <functional>

#include "gaea/lwp/gaea_define.h"
#include "gaea/lwp/common/common_transaction_manager_delegate.h"

#if defined(UTEST_PREPROCESS_MODULE)
  class FileServiceInterfaceTest;
#endif

GAEA_LWP_NAMESPACE_BEGIN

class UserAgent;
class FileServiceInterface;

class FileTransaction;
typedef std::shared_ptr<FileTransaction> FileTransactionPtr;

class CommonTransactionManager;
typedef std::shared_ptr<CommonTransactionManager> CommonTransactionManagerPtr;

class FileServiceInterface : public CommonTransactionManagerDelegate,
                             public std::enable_shared_from_this<FileServiceInterface> {
#if defined(UTEST_PREPROCESS_MODULE)
 friend FileServiceInterfaceTest;
#endif

 public:
  FileServiceInterface(UserContextPtr context);
  virtual ~FileServiceInterface() {};
  
  void set_get_token_handler(OnGetTokenHandler handler);
  void set_async_require_token_handler(OnAsyncGetTokenHandler handler);
  const std::string GetToken();
  const std::string CheckTokenIfValid(
      gaea::base::ErrorResult* error_result);
  
  /* Check file task if valid to exec */
  virtual bool CheckUploadTaskValid(
      FileUploadTaskPtr task,
      gaea::base::ErrorResult* result);
  
  virtual bool CheckDownloadTaskValid(
      FileDownloadTaskPtr task,
      gaea::base::ErrorResult* result);
  
  std::weak_ptr<UserAgent> user_agent() { return user_agent_; }
  void set_useragent(std::weak_ptr<UserAgent> user_agent) { user_agent_ = user_agent; }
  
  virtual void Upload(
      FileUploadTaskPtr task,
      std::function<void()> onStart,
      std::function<void(const FileProgress&)> onProgress,
      std::function<void(const FileUploadResult&)> onSuccess,
                         std::function<void(const gaea::base::ErrorResult&)> onFailure) {}
  
  virtual void Download(
      FileDownloadTaskPtr task,
      std::function<void()> onStart,
      std::function<void(const FileProgress&)> onProgress,
      std::function<void(const FileDownloadResult&)> onSuccess,
                         std::function<void(const gaea::base::ErrorResult&)> onFailure) {}

  virtual void FlushTransaction(bool external_trigger = false);
  virtual void CancelTask(const std::string& transaction_id);
  virtual void Clean(const gaea::base::ErrorResult& final_result);
  
  bool has_trigger_auth() const { return has_trigger_auth_; }
                               
  FileTransactionPtr CreateTransaction(
      const std::string& transaction_id,
      FileDownloadTaskPtr download_task,
      FileUploadTaskPtr upload_task);

 protected:
  void UpdateTransactionTimer(BaseTransactionPtr transaction, int64_t expired_interval);
  virtual void OnStartTransactionExpiredTimer(BaseTransactionPtr transaction) override;
  virtual bool OnCheckIfReady(BaseTransactionPtr transaction, gaea::base::ErrorResult* result) override;
  virtual void OnStart(BaseTransactionPtr transaction) override;

  virtual bool CheckUploadTaskIfReady(FileUploadTaskPtr task, gaea::base::ErrorResult* result);
  virtual bool CheckDownloadTaskIfReady(FileDownloadTaskPtr task, gaea::base::ErrorResult* result);
  virtual void OnTerminated(BaseTransactionPtr transaction) override;
  virtual void OnRemoveTimer(BaseTransactionPtr transaction) override;
  bool CommitFileTransactionStat(BaseTransactionPtr transaction);
  
 protected:
  gaea::base::Logger logger_;
  bool has_trigger_auth_;
  UserContextPtr user_context_;
  CommonTransactionManagerPtr task_manager_;
  OnGetTokenHandler get_token_handler_;
  OnAsyncGetTokenHandler async_require_token_handler_;
  std::weak_ptr<UserAgent> user_agent_;
};
typedef std::shared_ptr<FileServiceInterface> FileServiceInterfacePtr;
  

GAEA_LWP_NAMESPACE_END

#endif /* file_service_interface_h */
