#version 300 es
precision highp float;
precision highp sampler2D;

uniform sampler2D uPassTex;
uniform float uTime;
uniform vec2 uResolution;
uniform vec2 uMouse;

in vec2 vUV;
in vec2 vFragCoord;
out vec4 fragColor;

const float WIDTH = 160.0;
const float HEIGHT = 60.0;

// Signed distance function for a rounded rectangle
float sdRoundedRect(vec2 pos, vec2 halfSize, vec4 cornerRadius) {
    cornerRadius.xy = (pos.x > 0.0) ? cornerRadius.xy : cornerRadius.zw;
    cornerRadius.x = (pos.y > 0.0) ? cornerRadius.x : cornerRadius.y;
    
    vec2 q = abs(pos) - halfSize + cornerRadius.x;
    return min(max(q.x, q.y), 0.0) + length(max(q, 0.0)) - cornerRadius.x;
}

// Wrapper for box SDF
float boxSDF(vec2 uv) {
    return sdRoundedRect(uv, vec2(WIDTH, HEIGHT), vec4(HEIGHT));
}

// 2D hash function
vec2 randomVec2(vec2 co) {
    return fract(sin(vec2(
        dot(co, vec2(127.1, 311.7)),
        dot(co, vec2(269.5, 183.3))
    )) * 43758.5453);
}

// 高斯模糊采样偏移
const vec2 gaussianOffsets[9] = vec2[](
    vec2(-1.0, -1.0), vec2(0.0, -1.0), vec2(1.0, -1.0),
    vec2(-1.0,  0.0), vec2(0.0,  0.0), vec2(1.0,  0.0),
    vec2(-1.0,  1.0), vec2(0.0,  1.0), vec2(1.0,  1.0)
);

const float gaussianWeights[9] = float[](
    0.0625, 0.125, 0.0625,
    0.125,  0.25,  0.125,
    0.0625, 0.125, 0.0625
);

// 简化的模糊采样
vec3 sampleWithBlur(vec2 uv, float blurAmount) {
    vec2 pixelSize = 1.0 / uResolution;
    vec3 color = vec3(0.0);
    float totalWeight = 0.0;
    
    for(int i = 0; i < 9; i++) {
        vec2 offset = gaussianOffsets[i] * pixelSize * blurAmount;
        vec2 sampleUV = clamp(uv + offset, 0.0, 1.0);
        float weight = gaussianWeights[i];
        
        color += texture(uPassTex, sampleUV).rgb * weight;
        totalWeight += weight;
    }
    
    return color / totalWeight;
}

// 获取模糊后的颜色
vec3 getBlurredColor(vec2 uv, float blurStrength) {
    // 添加一些随机偏移
    vec2 noiseOffset = (randomVec2(uv + vec2(uTime)) - 0.5) * 0.01;
    return sampleWithBlur(uv + noiseOffset, blurStrength);
}

vec3 saturate(vec3 color, float factor) {
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    return mix(vec3(gray), color, factor);
}

// Compute refractive offset from SDF using screen-space gradients
vec2 computeRefractOffset(float sdf) {
    if (sdf < 0.1) {
        return vec2(0.0);
    }

    vec2 grad = normalize(vec2(dFdx(sdf), dFdy(sdf)));
    float offsetAmount = pow(abs(sdf), 12.0) * -0.1;
    return grad * offsetAmount;
}

float highlight(float sdf) {
    if (sdf < 0.1) {
        return 0.0;
    }

    vec2 grad = normalize(vec2(dFdx(sdf), dFdy(sdf)));
    return 1.0 - clamp(pow(1.0 - abs(dot(grad, vec2(-1.0, 1.0))), 0.5), 0.0, 1.0);
}

void main() {
    // 基础纹理采样
    vec3 baseTex = texture(uPassTex, vUV).rgb;
    
    // 计算相对于鼠标的位置
    vec2 centeredUV = vFragCoord - uMouse;
    float sdf = boxSDF(centeredUV);
    
    // 如果在玻璃区域外，直接显示原始图像
    if (sdf > HEIGHT) {
        fragColor = vec4(baseTex, 1.0);
        return;
    }

    // 计算玻璃效果强度
    float normalizedInside = clamp(1.0 - (sdf / HEIGHT), 0.0, 1.0);
    float edgeBlendFactor = pow(normalizedInside, 2.0);

    // 计算扭曲
    vec2 distortion = vec2(
        sin(vUV.y * 10.0 + uTime) * 0.02,
        cos(vUV.x * 10.0 + uTime) * 0.02
    ) * edgeBlendFactor;

    // 计算采样UV
    vec2 sampleUV = clamp(vUV + distortion, 0.0, 1.0);
    
    // 获取模糊效果
    float blurStrength = mix(1.0, 5.0, edgeBlendFactor);
    vec3 blurredTex = getBlurredColor(sampleUV, blurStrength);
    
    // 添加高光
    float highlightIntensity = pow(1.0 - abs(normalizedInside - 0.5) * 2.0, 2.0);
    vec3 highlightColor = vec3(1.0) * highlightIntensity * 0.2;
    
    // 混合颜色
    vec3 finalColor = mix(baseTex, blurredTex + highlightColor, edgeBlendFactor * 0.7);
    
    // 计算透明度
    float alpha = mix(0.4, 0.8, edgeBlendFactor);
    
    fragColor = vec4(finalColor, alpha);
}
}