#version 300 es
precision highp float;

in vec2 v_originTexCoord;
in float rateRev;
in float rate2;
in float rate3;
in float xr;
in vec2 texCoord;
in vec2 pos;
in vec2 pos2;
in vec2 hr;
in float radius;
in vec2 vp1;
in vec2 vp2;
in float bgRate;
in vec2 rect2Size;
in vec2 rectSize;
in float hlWidth;
in float shadowSize;
in vec3 highlightColor;
in vec2 lightDir1;
in vec2 lightDir2;
in float shadowIntensity;
in vec2 distortSpeed;
in float distortFreq;
in float F0;
in float specularPower;
in float highlightEdgeSmoothness;
in float glassTransparency;
in float transitionWidth;

uniform sampler2D uPassTex;
uniform sampler2D uMainTex;

uniform float u_aspect_ratio;
uniform float uTime;
uniform float uRate;

// Highlight configuration
uniform float uHighlightWidth;
uniform float uHighlightIntensity;

// Turbulence configuration
uniform float uTurbulenceStrength;

// Saturation configuration
uniform float uSaturation;

uniform float uLockBgType;
uniform float uGlassBlur;
uniform float uHomeGlassBlur;

uniform float uDistortionStrength;
uniform float uCircleRadius;
uniform float uBrightness;

out vec4 fragColor;

// Random function
float rand(float n) {
    return fract(sin(dot(vec2(n, n), vec2(12.9898, 78.233))) * 43758.5453);
}

// 优化: 简化扰动效果计算，使用预计算参数
vec2 distortShap(vec2 uv) {
    vec2 distortS = vec2(sin(uv.x), cos(uv.y));

    // 展开循环以减少开销
    // 第一次迭代
    float ifloat = 1.0;
    float r1 = rand(ifloat + 1.0) * 2.0 - 1.0;
    float r2 = rand(ifloat + 2.0) * 2.0 - 1.0;

    vec2 turb2;
    turb2.x = sin(uv.x * (1.0 + r1 * distortFreq) + distortS.y * ifloat + uTime * distortSpeed.x * r2);
    turb2.y = cos(uv.y * (1.0 + r1 * distortFreq) + distortS.x * ifloat + uTime * distortSpeed.y * r2);

    distortS = mix(distortS, turb2, 0.5);

    // 第二次迭代
    ifloat = 2.0;
    r1 = rand(ifloat + 1.0) * 2.0 - 1.0;
    r2 = rand(ifloat + 2.0) * 2.0 - 1.0;

    turb2.x = sin(uv.x * (1.0 + r1 * distortFreq) + distortS.y * ifloat + uTime * distortSpeed.x * r2);
    turb2.y = cos(uv.y * (1.0 + r1 * distortFreq) + distortS.x * ifloat + uTime * distortSpeed.y * r2);

    return mix(distortS, turb2, 0.5);
}

// 优化: 高光计算使用预计算参数
vec3 calculateHighlight(float sdf, vec2 uv, vec3 baseColor) {
    // 计算边缘平滑过渡
    float edge = 1.0 - smoothstep(0.0, hlWidth, abs(sdf));
    
    // 计算表面法线
    vec2 gradientDir = normalize(vec2(dFdx(sdf), dFdy(sdf)));
    
    // 视角方向
    vec2 viewDir = normalize(-uv);
    
    // 菲涅尔效应 - 使用预计算的F0
    float fresnel = F0 + (1.0 - F0) * pow(1.0 - max(0.0, dot(gradientDir, viewDir)), 5.0);
    
    // 高光计算 - 使用预计算的指数
    float specular1 = pow(max(0.0, dot(gradientDir, lightDir1)), specularPower);
    float specular2 = pow(max(0.0, dot(gradientDir, lightDir2)), specularPower);
    
    // 合并高光 - 使用预计算的平滑系数
    float highlight = edge * (specular1 + specular2) * highlightEdgeSmoothness;

    // 计算反射率
    float reflectivity = fresnel * highlight * 1.5;

    // 优化: 减少乘法操作
    vec3 brightBase = baseColor * 1.15;
    vec3 transColor = mix(brightBase, brightBase * glassTransparency, 0.3);
    //vec3 c = highlightColor * vec3(130./255.,173.0/255.,160.0/255.) * 1.2;
    return mix(transColor, highlightColor, reflectivity);
}

// 优化: 简化圆角矩形SDF计算
// 优化: 使用更高效的圆角矩形SDF计算
float superRoundedRect(vec2 p, vec2 size, float r, float n) {
    p = abs(p) - size + r;
    vec2 q = max(p, 0.0);
    float d = pow(pow(q.x + 1e-4, n) + pow(q.y + 1e-4, n), 1.0 / n);
    return d + min(max(p.x, p.y), 0.0) - r;
}


// 优化: 简化圆形SDF计算
float circleSDF(vec2 p, float r) {
    return length(p) - r;
}

// 优化: metaball函数改为更高效版本
float metaball(float d1, float d2, float size) {
    float k = size * 0.5;
    float h = clamp(0.5 + 0.5 * (d2 - d1) / k, 0.0, 1.0);
    return mix(d2, d1, h) - k * h * (1.0 - h);
}

// 优化: 调整饱和度函数以提高效率
vec3 adjustSaturation(vec3 color, float saturation) {
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    return mix(vec3(gray), color, saturation);
}

void main() {
    // 获取背景
    vec4 bg = texture(uPassTex, v_originTexCoord);

    // 计算距离场
    float distR = superRoundedRect(vp1, rectSize, radius, 3.5);
    float distR2 = superRoundedRect(vp2, rect2Size, radius, 3.5);
    float distC = circleSDF(vp1, uCircleRadius);
    float distC2 = circleSDF(vp2, uCircleRadius);

    // 混合圆形和矩形
    float dist1 = mix(distC, distR, rate2);
    float dist2 = mix(distC2, distR2, rate2);

    // 计算metaball距离场
    float dist = metaball(dist1, dist2, xr);

    // 添加扰动
    if (uRate < 1.0) {
        vec2 distortS = (distortShap(texCoord) * 2.0 - 1.0) * uTurbulenceStrength;
        dist += distortS.x * rateRev;
    }

    // 计算平滑边缘
    float distSmooth = smoothstep(0.0, -0.001, dist);

    // 初始化颜色
    vec4 color = vec4(1.0);

    // 玻璃区域
    if (dist < 0.0) {
        // 计算两个区域的混合因子
        float mixFactor = smoothstep(-transitionWidth, transitionWidth, dist1 - dist2);

        // 计算扭曲偏移
        float blend = clamp(-dist / uDistortionStrength, 0.0, 1.0);
        float pb = sin(pow(blend, 0.15) * 1.57);
        vec2 offset1 = (v_originTexCoord - pos) * pb + pos;
        vec2 offset2 = (v_originTexCoord - pos2) * pb + pos2;

        // 混合两个偏移位置
        vec2 offsetUV = mix(offset1, offset2, mixFactor);

        // 纹理采样 - 优化分支操作
        if (uGlassBlur < 0.5) {
            if (uHomeGlassBlur < 0.5){
                color = texture(uMainTex, offsetUV);
            } else {
                color = mix(texture(uMainTex, offsetUV), texture(uPassTex, offsetUV), uRate);
            }
        } else {
            if (uHomeGlassBlur < 0.5){
                color = mix(texture(uPassTex, offsetUV), texture(uMainTex, offsetUV), uRate);
            } else {
                color = texture(uPassTex, offsetUV);
            }
        }

        // 调整饱和度和高光
        color.rgb = adjustSaturation(color.rgb, uSaturation);
        color.rgb = calculateHighlight(dist, v_originTexCoord, color.rgb) + uBrightness;
    } else if (dist <= shadowSize){
        if (uLockBgType > 0.5){
            bg = texture(uMainTex, v_originTexCoord);
            float shadow = (dist) * (1.0 / shadowSize) * 0.05 + 0.95;
            bg.rgb *= shadow;
        }else {
            color = texture(uMainTex, v_originTexCoord);
            float shadow = (dist) * (1.0 / shadowSize) * 0.05 + 0.95;
            bg.rgb *= shadow;
            bg = mix(color, bg, bgRate);
        }
    } else {
        if (uLockBgType > 0.5){
            bg = texture(uMainTex, v_originTexCoord);
        }else {
            color = texture(uMainTex, v_originTexCoord);
            bg = mix(color, bg, bgRate);
        }
    }

    // 最终颜色混合
    fragColor = mix(bg, color, distSmooth);
}