package com.inmovation.tools.image.load;

import java.io.FileNotFoundException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.inmovation.tools.FileUtil;
import com.inmovation.tools.InmovationLogger;
import com.inmovation.tools.LogUtil;
import com.inmovation.tools.StringUtil;
import com.inmovation.tools.image.IFTP;
import com.inmovation.tools.image.ImageObserver;

import android.app.ActivityManager;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Environment;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.webkit.URLUtil;
import android.widget.ImageView;

public class AsyncImageLoaderFtpManager {

	public static final String TAG = AsyncImageLoaderFtpManager.class
			.getSimpleName();
	InmovationLogger mLogger = InmovationLogger.getLogger(TAG);
	ExecutorService mExecutorService;
	static FileCache mFileCache;

	IFTP iFTP;

	public AsyncImageLoaderFtpManager(Context context, IFTP iftp) {

		int nThreads = Runtime.getRuntime().availableProcessors();
		mExecutorService = Executors.newFixedThreadPool(nThreads);
		mFileCache = new FileCache(context);
		this.iFTP = iftp;
	}

	public AsyncImageLoaderFtpManager(IFTP iftp) {
		int nThreads = Runtime.getRuntime().availableProcessors();
		mExecutorService = Executors.newFixedThreadPool(nThreads);
		this.iFTP = iftp;
	}

	public static void init(Context context) {
		// 得到应用的实际可用内存大小
		int Memory = ((ActivityManager) context
				.getSystemService(Context.ACTIVITY_SERVICE)).getMemoryClass();
		ImageMemoryCacheManager.init(Memory / 8);
		mFileCache = new FileCache(context);
	}

	/**
	 * imageurl是图片的直接路径
	 * 
	 * @param imageUrl
	 * @param imageCallback
	 * @return
	 */
	public Bitmap loadImage(final String imgurl,
			final ImageLoadCallBack imageCallback) {
		ImageObserver mImageObserver = new ImageObserver(imgurl);
		Bitmap bitmap = ImageMemoryCacheManager.getInstance()
				.getBitmapFromMemoryCache(mImageObserver.getMemoryCacheKey());

		if (bitmap != null) {
			return bitmap;
		}
		final Handler handler = new Handler() {
			public void handleMessage(Message message) {
				imageCallback.imageLoaded((ImageObserver) message.obj);
			}
		};

		Runnable loadTask;
		if (mImageObserver.isFtp()) {
			loadTask = new LoaderImageByFtp(mImageObserver, handler);
		} else if (!URLUtil.isValidUrl(imgurl)) {
			loadTask = new LoaderImageSdTask(mImageObserver, handler);
		} else {
			loadTask = new LoaderImageTask(mImageObserver, handler);
		}
		mExecutorService.execute(loadTask);
		return null;
	}

	/**
	 * 包含文件缓存的，缓存机制
	 * 
	 * @param mImageObserver
	 *            需要确定图片的id，和
	 * @param imageCallback
	 * @return
	 */
	public Bitmap loadImageWithFile(ImageObserver mImageObserver,
			final ImageLoadCallBack imageCallback) {

		Bitmap bitmap = ImageMemoryCacheManager.getInstance()
				.getBitmapFromMemoryCache(mImageObserver.getMemoryCacheKey());
		if (bitmap != null) {
			mImageObserver.setBitmap(bitmap);
			return bitmap;
		}
		// 判断文件中是否存在
		if (mFileCache != null) {
			String path = mFileCache.obtainCachePath(mImageObserver);
			if (!StringUtil.isEmpty(path)) {
				mImageObserver.setUrl(path);
			}
		}

		final Handler handler = new Handler() {
			public void handleMessage(Message message) {
				imageCallback.imageLoaded((ImageObserver) message.obj);
			}
		};

		Runnable loadTask;
		if (mImageObserver.isFtp()) {
			loadTask = new LoaderImageByFtp(mImageObserver, handler);
		} else if (!mImageObserver.isValidUrl()) {
			loadTask = new LoaderImageSdTask(mImageObserver, handler);
		} else {
			loadTask = new LoaderImageTask(mImageObserver, handler);
		}
		mExecutorService.execute(loadTask);
		return null;
	}

	public void saveBitmapToFileCache(String filename, String srcPath) {
		// 判断文件中是否存在
		if (mFileCache != null) {
			mFileCache.saveImage(filename, srcPath);
		}
	}

	class LoaderImageTask implements Runnable {

		Handler handler;
		ImageObserver mImageObserver;

		boolean no_use_cache_but_does;

		public LoaderImageTask(ImageObserver mImageObserver, Handler handler) {
			this.handler = handler;
			this.mImageObserver = mImageObserver;
		}

		public LoaderImageTask(ImageObserver mImageObserver, Handler handler,
				boolean no_use_cache_but_does) {
			this.handler = handler;
			this.mImageObserver = mImageObserver;
			this.no_use_cache_but_does = no_use_cache_but_does;
		}

		@Override
		public void run() {
			Bitmap bitmap = ImageMemoryCacheManager.getInstance()
					.getBitmapFromMemoryCache(
							mImageObserver.getMemoryCacheKey());

			if (no_use_cache_but_does || bitmap == null) {
				bitmap = ImageLoader.loadImageFromUrl(mImageObserver.getUrl());
			}
			if (mImageObserver.getMemoryCacheKey() != null && bitmap != null) {
				ImageMemoryCacheManager.getInstance().addBitmapToMemoryCache(
						mImageObserver.getMemoryCacheKey(), bitmap);
			}
			mImageObserver.setBitmap(bitmap);
			Message message = handler.obtainMessage(0, mImageObserver);
			handler.sendMessage(message);

			if (bitmap != null && mFileCache != null)
				mFileCache.saveBitmap(mImageObserver, bitmap);
		}

	}

	class LoaderImageSdTask implements Runnable {
		Handler handler;

		ImageObserver mImageObserver;

		public LoaderImageSdTask(ImageObserver mImageObserver, Handler handler) {
			this.handler = handler;
			this.mImageObserver = mImageObserver;
		}

		@Override
		public void run() {

			Bitmap bitmap = ImageMemoryCacheManager.getInstance()
					.getBitmapFromMemoryCache(
							mImageObserver.getMemoryCacheKey());
			if (bitmap == null) {
				try {
					bitmap = ImageLoader.loadImageFromSDCard(mImageObserver
							.getUrl());
				} catch (FileNotFoundException e) {
					e.printStackTrace();
					Log.e(TAG, "图片路径未找到：" + mImageObserver.getUrl());
				} catch (OutOfMemoryError error) {
					Log.e(TAG, "图片内存溢出：" + mImageObserver.getUrl());
					System.gc();
				}
			}
			if (mImageObserver.getMemoryCacheKey() != null && bitmap != null) {
				ImageMemoryCacheManager.getInstance().addBitmapToMemoryCache(
						mImageObserver.getMemoryCacheKey(), bitmap);
			}
			mImageObserver.setBitmap(bitmap);
			Message message = handler.obtainMessage(0, mImageObserver);
			handler.sendMessage(message);
		}
	}

	class LoaderImageByFtp implements Runnable {
		Handler handler;

		ImageObserver mImageObserver;

		public LoaderImageByFtp(ImageObserver mImageObserver, Handler handler) {
			this.handler = handler;
			this.mImageObserver = mImageObserver;
		}

		@Override
		public void run() {

			LogUtil.d(TAG, "LoaderImageByFtp:" + mImageObserver);
			Bitmap bitmap = ImageMemoryCacheManager.getInstance()
					.getBitmapFromMemoryCache(
							mImageObserver.getMemoryCacheKey());
			if (bitmap == null) {
				try {
					String path = Environment.getExternalStorageDirectory()
							.toString() + "/" + System.currentTimeMillis();
					LogUtil.d(TAG,
							"LoaderImageByFtp-url:" + mImageObserver.getUrl());
					iFTP.download(mImageObserver.getUrl(), path);
					bitmap = ImageLoader.loadImageFromSDCard(path);
					FileUtil.deleteFile(path);
				} catch (FileNotFoundException e) {
					e.printStackTrace();
					Log.e(TAG, "图片路径未找到：" + mImageObserver.getUrl());
				} catch (OutOfMemoryError error) {
					Log.e(TAG, "图片内存溢出：" + mImageObserver.getUrl());
					System.gc();
				}
			}
			if (mImageObserver.getMemoryCacheKey() != null && bitmap != null) {
				ImageMemoryCacheManager.getInstance().addBitmapToMemoryCache(
						mImageObserver.getMemoryCacheKey(), bitmap);
			}
			mImageObserver.setBitmap(bitmap);
			
			if (bitmap != null && mFileCache != null)
				mFileCache.saveBitmap(mImageObserver, bitmap);
			
			Message message = handler.obtainMessage(0, mImageObserver);
			handler.sendMessage(message);
		}
	}
}
