package com.whroid.tool.audiorecorder.voice;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaRecorder;
import android.os.Message;
import android.util.Log;

import com.inmovation.android.Exception.CommonException;
import com.whroid.tool.audiorecorder.DataEncodeThread;
import com.whroid.tool.audiorecorder.PCMFormat;
import com.whroid.tool.audiorecorder.RingBuffer;
import com.whroid.tool.audiorecorder.SimpleLame;

public class VoiceMp3Recorder extends AbstractRecorder {

	private static final String TAG = VoiceMp3Recorder.class.getSimpleName();

	static {
		System.loadLibrary("mp3lame");
	}

	private static final int DEFAULT_SAMPLING_RATE = 22050;

	private static final int FRAME_COUNT = 160;

	/* Encoded bit rate. MP3 file will be encoded with bit rate 32kbps */
	private static final int BIT_RATE = 16;

	private AudioRecord audioRecord = null;

	private int bufferSize;

	private File mp3File;

	private RingBuffer ringBuffer;

	private byte[] buffer;

	private FileOutputStream os = null;

	private DataEncodeThread encodeThread;

	private int samplingRate;

	private int channelConfig;

	private PCMFormat audioFormat;

	private boolean isRecording = false;
	public float voiceAmplitude; // 音量大小

	public VoiceMp3Recorder(int samplingRate, int channelConfig,
			PCMFormat audioFormat) {
		this.samplingRate = samplingRate;
		this.channelConfig = channelConfig;
		this.audioFormat = audioFormat;
		initVoice();
		setRecorderType(".mp3");
	}

	/**
	 * Default constructor. Setup recorder with default sampling rate 1 channel,
	 * 16 bits pcm
	 */
	public VoiceMp3Recorder() {
		this(DEFAULT_SAMPLING_RATE, AudioFormat.CHANNEL_IN_MONO,
				PCMFormat.PCM_16BIT);
	}

	/**
	 * Start recording. Create an encoding thread. Start record from this
	 * thread.
	 * 
	 * @throws IOException
	 */
	public void start() throws CommonException {
		if (isRecording)
			return;
		// }
		String state = android.os.Environment.getExternalStorageState();
		if (!state.equals(android.os.Environment.MEDIA_MOUNTED)) {
			throw new CommonException("SD Card is not mounted,It is  " + state
					+ ".");
		}
		voicePath = rootDir + "/voice" + System.currentTimeMillis() + voiceType;
		File directory = new File(voicePath).getParentFile();
		if (!directory.exists() && !directory.mkdirs()) {
			throw new CommonException("Path to file could not be created");
		}
		// if (audioRecord == null) {
		try {
			initAudioRecorder();
		} catch (IOException e) {
			throw new CommonException("VoiceMp3Recorder:start", e);
		}

		audioRecord.startRecording();

		new Thread() {

			@Override
			public void run() {
				isRecording = true;
				voiceTimer.start();
				currentStatus = STATUS_ING;
				while (isRecording) {
					int bytes = audioRecord.read(buffer, 0, bufferSize);
					if (bytes > 0) {
						ringBuffer.write(buffer, bytes);
					}

					int v = 0;
					// 将 buffer 内容取出，进行平方和运算
					for (int i = 0; i < buffer.length; i++) {
						// 这里没有做运算的优化，为了更加清晰的展示代码
						v += buffer[i] * buffer[i];
					}
					// 平方和除以数据总长度，得到音量大小。可以获取白噪声值，然后对实际采样进行标准化。
					// 如果想利用这个数值进行操作，建议用 sendMessage 将其抛出，在 Handler 里进行处理。
					Log.e("spl", String.valueOf(v / (float) bytes));
					voiceAmplitude = (v / (float) bytes);

					// int value = (int) (Math.abs((int)(value1/10000)) );
					// Log.e("spl", "value:"+value+" values1:"+value1);
					// Message msg = mHandler.obtainMessage();
					// msg.obj = value1;
					// mHandler.sendMessage(msg);
				}

				// release and finalize audioRecord
				try {
					audioRecord.stop();
					audioRecord.release();
					audioRecord = null;
					voiceTimer.stop();
					currentStatus = STATUS_NONE;
					// stop the encoding thread and try to wait
					// until the thread finishes its job
					Message msg = Message.obtain(encodeThread.getHandler(),
							DataEncodeThread.PROCESS_STOP);
					msg.sendToTarget();

					Log.d(TAG, "waiting for encoding thread");
					encodeThread.join();
					Log.d(TAG, "done encoding thread");
				} catch (InterruptedException e) {
					Log.d(TAG, "Faile to join encode thread");
				} finally {
					if (os != null) {
						try {
							os.close();
						} catch (IOException e) {
							e.printStackTrace();
						}
					}
				}

			}
		}.start();
	}

	/**
	 * 
	 * @throws IOException
	 */
	public void stop() throws CommonException {
		isRecording = false;
	}

	@Override
	public int getAmplitude() {
		// TODO Auto-generated method stub
		int volume = (int) (Math.abs(voiceAmplitude / 10000) * MAX_VU_SIZE);
		return volume;
	}

	/**
	 * Initialize audio recorder
	 */
	private void initAudioRecorder() throws IOException {
		int bytesPerFrame = audioFormat.getBytesPerFrame();
		/*
		 * Get number of samples. Calculate the buffer size (round up to the
		 * factor of given frame size)
		 */
		int frameSize = AudioRecord.getMinBufferSize(samplingRate,
				channelConfig, audioFormat.getAudioFormat()) / bytesPerFrame;
		if (frameSize % FRAME_COUNT != 0) {
			frameSize = frameSize + (FRAME_COUNT - frameSize % FRAME_COUNT);
			Log.d(TAG, "Frame size: " + frameSize);
		}

		bufferSize = frameSize * bytesPerFrame;

		/* Setup audio recorder */
		audioRecord = new AudioRecord(MediaRecorder.AudioSource.MIC,
				samplingRate, channelConfig, audioFormat.getAudioFormat(),
				bufferSize);

		// Setup RingBuffer. Currently is 10 times size of hardware buffer
		// Initialize buffer to hold data
		ringBuffer = new RingBuffer(10 * bufferSize);
		buffer = new byte[bufferSize];

		// Initialize lame buffer
		// mp3 sampling rate is the same as the recorded pcm sampling rate
		// The bit rate is 32kbps
		SimpleLame.init(samplingRate, 1, samplingRate, BIT_RATE);

		mp3File = new File(voicePath);
		os = new FileOutputStream(mp3File);

		// Create and run thread used to encode data
		// The thread will
		encodeThread = new DataEncodeThread(ringBuffer, os, bufferSize);
		encodeThread.start();
		audioRecord.setRecordPositionUpdateListener(encodeThread,
				encodeThread.getHandler());
		audioRecord.setPositionNotificationPeriod(FRAME_COUNT);
	}

	public interface OnVoumeChangedListener {

		public void onValue(float value);
	}

}