package com.union.security.base;

import java.security.AlgorithmParameters;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.InvalidParameterException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.ShortBufferException;
import javax.crypto.spec.IvParameterSpec;

import com.union.crypto.BlockCipher;
import com.union.crypto.BufferedBlockCipher;
import com.union.crypto.CipherParameters;
import com.union.crypto.DataLengthException;
import com.union.crypto.InvalidCipherTextException;
import com.union.crypto.OutputLengthException;
import com.union.crypto.paddings.BlockCipherPadding;
import com.union.crypto.paddings.PaddedBufferedBlockCipher;
import com.union.crypto.params.KeyParameter;
import com.union.crypto.params.ParametersWithIV;
import com.union.crypto.params.ParametersWithRandom;
import com.union.util.Strings;

import com.union.crypto.modes.CBCBlockCipher;

public class BaseBlockCipher extends CipherSpi{
	
	private Class[] availableSpecs = { IvParameterSpec.class };
	
	protected AlgorithmParameters     engineParams = null;
	
	private int ivLength = 0;
	private boolean padded;
	
	private String modeName = null;
	
	private GenericBlockCipher cipher;
	private ParametersWithIV ivParam;
	private BlockCipher baseEngine;
	
	protected BaseBlockCipher(BlockCipher engine) {
		baseEngine = engine;

		cipher = new BufferedGenericBlockCipher(engine);
	}

	@Override
	protected void engineSetMode(String mode) throws NoSuchAlgorithmException {
		modeName = Strings.toUpperCase(mode);

		if (modeName.equals("ECB")) {
			ivLength = 0;
			cipher = new BufferedGenericBlockCipher(baseEngine);
		} else if (modeName.equals("CBC")) {
			ivLength = baseEngine.getBlockSize();
			cipher = new BufferedGenericBlockCipher(new CBCBlockCipher(
					baseEngine));
		} else {
			throw new NoSuchAlgorithmException("can't support mode " + mode);
		}
	}

	@Override
	protected void engineSetPadding(String padding)
			throws NoSuchPaddingException {

		String paddingName = Strings.toUpperCase(padding);

		if (paddingName.equals("NOPADDING")) {
			if (cipher.wrapOnNoPadding()) {
				cipher = new BufferedGenericBlockCipher(
						new BufferedBlockCipher(cipher.getUnderlyingCipher()));
			}
		} else {
			padded = true;

			if (isAEADModeName(modeName)) {
				throw new NoSuchPaddingException(
						"Only NoPadding can be used with AEAD modes.");
			} else if (paddingName.equals("PKCS5PADDING")
					|| paddingName.equals("PKCS7PADDING")) {
				cipher = new BufferedGenericBlockCipher(
						cipher.getUnderlyingCipher());
			} else {
				throw new NoSuchPaddingException("Padding " + padding
						+ " unknown.");
			}
		}
	
	}
	
	private boolean isAEADModeName(String modeName) {
		return "CCM".equals(modeName) || "EAX".equals(modeName)
				|| "GCM".equals(modeName) || "OCB".equals(modeName);
	}

	@Override
	protected int engineGetBlockSize() {
		return baseEngine.getBlockSize();
	}

	@Override
	protected int engineGetOutputSize(int inputLen) {
		return cipher.getOutputSize(inputLen);
	}

	@Override
	protected byte[] engineGetIV() {
		return (ivParam != null) ? ivParam.getIV() : null;
	}

	@Override
	protected AlgorithmParameters engineGetParameters() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	protected void engineInit(int opmode, Key key, SecureRandom random)
			throws InvalidKeyException {
		try {
			engineInit(opmode, key, (AlgorithmParameterSpec) null, random);
		} catch (InvalidAlgorithmParameterException e) {
			throw new InvalidKeyException(e.getMessage());
		}
	}

	@Override
	protected void engineInit(int opmode, Key key,
			AlgorithmParameterSpec params, SecureRandom random)
			throws InvalidKeyException, InvalidAlgorithmParameterException {

		CipherParameters param;

		//
		// basic key check
		//
		if (!(key instanceof SecretKey)) {
			throw new InvalidKeyException("Key for algorithm "
					+ key.getAlgorithm()
					+ " not suitable for symmetric enryption.");
		}

		//
		// for RC5-64 we must have some default parameters
		//
		if (params == null
				&& baseEngine.getAlgorithmName().startsWith("RC5-64")) {
			throw new InvalidAlgorithmParameterException(
					"RC5 requires an RC5ParametersSpec to be passed in.");
		}

		//
		// a note on iv's - if ivLength is zero the IV gets ignored (we don't
		// use it).
		//
		if (params == null) {
			param = new KeyParameter(key.getEncoded());
		} else if (params instanceof IvParameterSpec) {
			if (ivLength != 0) {
				IvParameterSpec p = (IvParameterSpec) params;

				if (p.getIV().length != ivLength && !isAEADModeName(modeName)) {
					throw new InvalidAlgorithmParameterException("IV must be "
							+ ivLength + " bytes long.");
				}

				
				param = new ParametersWithIV(new KeyParameter(
							key.getEncoded()), p.getIV());
				ivParam = (ParametersWithIV) param;
			} else {
				if (modeName != null && modeName.equals("ECB")) {
					throw new InvalidAlgorithmParameterException(
							"ECB mode does not use an IV");
				}

				param = new KeyParameter(key.getEncoded());
			}
		} else {
			throw new InvalidAlgorithmParameterException(
					"unknown parameter type.");
		}

		if ((ivLength != 0) && !(param instanceof ParametersWithIV)) {
			SecureRandom ivRandom = random;

			if (ivRandom == null) {
				ivRandom = new SecureRandom();
			}

			if ((opmode == Cipher.ENCRYPT_MODE) || (opmode == Cipher.WRAP_MODE)) {
				byte[] iv = new byte[ivLength];

				ivRandom.nextBytes(iv);
				param = new ParametersWithIV(param, iv);
				ivParam = (ParametersWithIV) param;
			} else if (cipher.getUnderlyingCipher().getAlgorithmName()
					.indexOf("PGPCFB") < 0) {
				throw new InvalidAlgorithmParameterException(
						"no IV set when one expected");
			}
		}

		if (random != null && padded) {
			param = new ParametersWithRandom(param, random);
		}

		try {
			switch (opmode) {
			case Cipher.ENCRYPT_MODE:
			case Cipher.WRAP_MODE:
				cipher.init(true, param);
				break;
			case Cipher.DECRYPT_MODE:
			case Cipher.UNWRAP_MODE:
				cipher.init(false, param);
				break;
			default:
				throw new InvalidParameterException("unknown opmode " + opmode
						+ " passed");
			}
		} catch (Exception e) {
			throw new InvalidKeyException(e.getMessage());
		}
	
	}

	@Override
	protected void engineInit(int opmode, Key key, AlgorithmParameters params,
			SecureRandom random) throws InvalidKeyException,
			InvalidAlgorithmParameterException {

		AlgorithmParameterSpec paramSpec = null;

		if (params != null) {
			for (int i = 0; i != availableSpecs.length; i++) {
				if (availableSpecs[i] == null) {
					continue;
				}

				try {
					paramSpec = params.getParameterSpec(availableSpecs[i]);
					break;
				} catch (Exception e) {
					// try again if possible
				}
			}

			if (paramSpec == null) {
				throw new InvalidAlgorithmParameterException(
						"can't handle parameter " + params.toString());
			}
		}

		engineInit(opmode, key, paramSpec, random);

		engineParams = params;
	
	}

	@Override
	protected byte[] engineUpdate(byte[] input, int inputOffset, int inputLen) {
		int length = cipher.getUpdateOutputSize(inputLen);

		if (length > 0) {
			byte[] out = new byte[length];

			int len = cipher.processBytes(input, inputOffset, inputLen, out, 0);

			if (len == 0) {
				return null;
			} else if (len != out.length) {
				byte[] tmp = new byte[len];

				System.arraycopy(out, 0, tmp, 0, len);

				return tmp;
			}

			return out;
		}

		cipher.processBytes(input, inputOffset, inputLen, null, 0);

		return null;
	}

	@Override
	protected int engineUpdate(byte[] input, int inputOffset, int inputLen,
			byte[] output, int outputOffset) throws ShortBufferException {
		try {
			return cipher.processBytes(input, inputOffset, inputLen, output,
					outputOffset);
		} catch (DataLengthException e) {
			throw new ShortBufferException(e.getMessage());
		}
	}

	@Override
	protected byte[] engineDoFinal(byte[] input, int inputOffset, int inputLen)
			throws IllegalBlockSizeException, BadPaddingException {

		int len = 0;

		byte[] tmp = new byte[engineGetOutputSize(inputLen)];

		if (inputLen != 0) {
			len = cipher.processBytes(input, inputOffset, inputLen, tmp, 0);

		}

		try {
			len += cipher.doFinal(tmp, len);
		} catch (DataLengthException e) {
			throw new IllegalBlockSizeException(e.getMessage());
		}

		if (len == tmp.length) {
			return tmp;
		}

		byte[] out = new byte[len];

		System.arraycopy(tmp, 0, out, 0, len);

		return out;
	}

	@Override
	protected int engineDoFinal(byte[] input, int inputOffset, int inputLen,
			byte[] output, int outputOffset) throws ShortBufferException,
			IllegalBlockSizeException, BadPaddingException {
		try {
			int len = 0;

			if (inputLen != 0) {
				len = cipher.processBytes(input, inputOffset, inputLen, output,
						outputOffset);
			}

			return (len + cipher.doFinal(output, outputOffset + len));
		} catch (OutputLengthException e) {
			throw new ShortBufferException(e.getMessage());
		} catch (DataLengthException e) {
			throw new IllegalBlockSizeException(e.getMessage());
		}
	}
	
	private static class BufferedGenericBlockCipher implements
			GenericBlockCipher {
		private BufferedBlockCipher cipher;

		BufferedGenericBlockCipher(BufferedBlockCipher cipher) {
			this.cipher = cipher;
		}

		BufferedGenericBlockCipher(
				com.union.crypto.BlockCipher cipher) {
			
			this.cipher = new com.union.crypto.paddings.PaddedBufferedBlockCipher(
						cipher);
		}

		BufferedGenericBlockCipher(
				com.union.crypto.BlockCipher cipher,
				BlockCipherPadding padding) {
			this.cipher = new PaddedBufferedBlockCipher(cipher, padding);
		}

		public void init(boolean forEncryption, CipherParameters params)
				throws IllegalArgumentException {
			cipher.init(forEncryption, params);
		}

		public boolean wrapOnNoPadding() {
			return true;
		}

		public String getAlgorithmName() {
			return cipher.getUnderlyingCipher().getAlgorithmName();
		}

		public com.union.crypto.BlockCipher getUnderlyingCipher() {
			return cipher.getUnderlyingCipher();
		}

		public int getOutputSize(int len) {
			return cipher.getOutputSize(len);
		}

		public int getUpdateOutputSize(int len) {
			return cipher.getUpdateOutputSize(len);
		}

		public void updateAAD(byte[] input, int offset, int length) {
			throw new UnsupportedOperationException(
					"AAD is not supported in the current mode.");
		}

		public int processByte(byte in, byte[] out, int outOff)
				throws DataLengthException {
			return cipher.processByte(in, out, outOff);
		}

		public int processBytes(byte[] in, int inOff, int len, byte[] out,
				int outOff) throws DataLengthException {
			return cipher.processBytes(in, inOff, len, out, outOff);
		}

		public int doFinal(byte[] out, int outOff)
				throws IllegalStateException, BadPaddingException {
			try {
				return cipher.doFinal(out, outOff);
			} catch (InvalidCipherTextException e) {
				throw new BadPaddingException(e.getMessage());
			}
		}
	}

	
	static private interface GenericBlockCipher {
		public void init(boolean forEncryption, CipherParameters params)
				throws IllegalArgumentException;

		public boolean wrapOnNoPadding();

		public String getAlgorithmName();

		public com.union.crypto.BlockCipher getUnderlyingCipher();

		public int getOutputSize(int len);

		public int getUpdateOutputSize(int len);

		public void updateAAD(byte[] input, int offset, int length);

		public int processByte(byte in, byte[] out, int outOff)
				throws DataLengthException;

		public int processBytes(byte[] in, int inOff, int len, byte[] out,
				int outOff) throws DataLengthException;

		public int doFinal(byte[] out, int outOff)
				throws IllegalStateException, BadPaddingException;
	}
}
