precision highp float;
varying vec2 vTextureCoord;
uniform lowp sampler2D sTexture;

/* --- START BRIGHTNESS --- */
uniform lowp float brightness;
vec4 runBrightness(in vec4 inputColor) {
    return vec4((inputColor.rgb + vec3(brightness)), inputColor.w);
}
/* --- END BRIGHTNESS --- */
/* --- START CONSTAST --- */
uniform lowp float contrast;
vec4 runContrast(in vec4 inputColor) {
    return vec4(((inputColor.rgb - vec3(0.5)) * contrast + vec3(0.5)), inputColor.w);
}
/* --- END CONTRAST --- */
/* --- START SHARPNESS --- */
varying highp vec2 leftTextureCoordinate;
varying highp vec2 rightTextureCoordinate;
varying highp vec2 topTextureCoordinate;
varying highp vec2 bottomTextureCoordinate;

varying float centerMultiplier;
varying float edgeMultiplier;

vec4 runSharpness(in vec4 inputColor) {
    mediump vec3 textureColor = inputColor.rgb;
    mediump vec3 leftTextureColor   = texture2D(sTexture, leftTextureCoordinate).rgb;
    mediump vec3 rightTextureColor  = texture2D(sTexture, rightTextureCoordinate).rgb;
    mediump vec3 topTextureColor    = texture2D(sTexture, topTextureCoordinate).rgb;
    mediump vec3 bottomTextureColor = texture2D(sTexture, bottomTextureCoordinate).rgb;

    mediump vec3 finalColor = (textureColor * centerMultiplier - (
            leftTextureColor * edgeMultiplier
            + rightTextureColor * edgeMultiplier
            + topTextureColor * edgeMultiplier
            + bottomTextureColor * edgeMultiplier
        )
    );
    return vec4(clamp(finalColor, 0.0, 1.0), inputColor.w);
}
/* --- END SHARPNESS --- */
/* --- START GAUSSIAN BLUR --- */
const lowp int GAUSSIAN_SAMPLES = 9;
varying highp vec2 blurCoordinates[GAUSSIAN_SAMPLES];

vec4 runBlurGaussian(in vec4 inputColor) {
    highp vec3 sum = vec3(0.0);

    sum += texture2D(sTexture, blurCoordinates[0]).rgb * 0.05;
    sum += texture2D(sTexture, blurCoordinates[1]).rgb * 0.09;
    sum += texture2D(sTexture, blurCoordinates[2]).rgb * 0.12;
    sum += texture2D(sTexture, blurCoordinates[3]).rgb * 0.15;
    sum += texture2D(sTexture, blurCoordinates[4]).rgb * 0.18;
    sum += texture2D(sTexture, blurCoordinates[5]).rgb * 0.15;
    sum += texture2D(sTexture, blurCoordinates[6]).rgb * 0.12;
    sum += texture2D(sTexture, blurCoordinates[7]).rgb * 0.09;
    sum += texture2D(sTexture, blurCoordinates[8]).rgb * 0.05;

    return vec4(sum, inputColor.w);
}

/* --- END GAUSSIAN BLUR --- */
/* --- START SATURATION --- */
uniform lowp float saturation;
const mediump vec3 luminanceWeighting = vec3(0.2125, 0.7154, 0.0721);

vec4 runSaturation(in vec4 inputColor) {
    lowp float luminance = dot(inputColor.rgb, luminanceWeighting);
    lowp vec3 greyScaleColor = vec3(luminance);
    return vec4(mix(greyScaleColor, inputColor.rgb, saturation), inputColor.w);
}
/* --- END SATURATION--- */
/* --- START EXPOSURE--- */
uniform highp float exposure;

vec4 runExposure(in vec4 inputColor) {
    return vec4(inputColor.rgb * pow(2.0, exposure), inputColor.w);
}
/* --- END EXPOSURE--- */
/* --- START WHITE BALANCE--- */
uniform lowp float temperature;
uniform lowp float tint;

const lowp vec3 warmFilter = vec3(0.93, 0.54, 0.0);

const mediump mat3 RGBtoYIQ = mat3(0.299, 0.587, 0.114, 0.596, -0.274, -0.322, 0.212, -0.523, 0.311);
const mediump mat3 YIQtoRGB = mat3(1.0, 0.956, 0.621, 1.0, -0.272, -0.647, 1.0, -1.105, 1.702);

vec4 runWhiteBalance(in vec4 inputColor) {
    mediump vec3 yiq = RGBtoYIQ * inputColor.rgb;//adjusting tint
    yiq.b = clamp(yiq.b + tint*0.5226*0.1, -0.5226, 0.5226);
    lowp vec3 rgb = YIQtoRGB * yiq;

    lowp vec3 processed = vec3(
    (rgb.r < 0.5 ? (2.0 * rgb.r * warmFilter.r) : (1.0 - 2.0 * (1.0 - rgb.r) * (1.0 - warmFilter.r))), //adjusting temperature
    (rgb.g < 0.5 ? (2.0 * rgb.g * warmFilter.g) : (1.0 - 2.0 * (1.0 - rgb.g) * (1.0 - warmFilter.g))),
    (rgb.b < 0.5 ? (2.0 * rgb.b * warmFilter.b) : (1.0 - 2.0 * (1.0 - rgb.b) * (1.0 - warmFilter.b))));

    return vec4(mix(rgb, processed, temperature), inputColor.w);
}
/* --- END WHITE BALANCE--- */
/* --- START HIGHLIGHTS SHADOWS--- */
uniform lowp float shadows;
uniform lowp float highlights;

vec4 runHighlightsShadows(in vec4 inputColor) {
    mediump float luminance = dot(inputColor.rgb, luminanceWeighting);

    //(shadows+1.0) changed to just shadows:
    mediump float shadow = clamp((pow(luminance, 1.0/shadows) + (-0.76)*pow(luminance, 2.0/shadows)) - luminance, 0.0, 1.0);
    mediump float highlight = clamp((1.0 - (pow(1.0-luminance, 1.0/(2.0-highlights)) + (-0.8)*pow(1.0-luminance, 2.0/(2.0-highlights)))) - luminance, -1.0, 0.0);
    lowp vec3 result = vec3(0.0, 0.0, 0.0) + ((luminance + shadow + highlight) - 0.0) * ((inputColor.rgb - vec3(0.0, 0.0, 0.0))/(luminance - 0.0));

    // blend toward white if highlights is more than 1
    mediump float contrastedLuminance = ((luminance - 0.5) * 1.5) + 0.5;
    mediump float whiteInterp = contrastedLuminance*contrastedLuminance*contrastedLuminance;
    mediump float whiteTarget = clamp(highlights, 1.0, 2.0) - 1.0;
    result = mix(result, vec3(inputColor.w), whiteInterp*whiteTarget);

    // blend toward black if shadows is less than 1
    mediump float invContrastedLuminance = 1.0 - contrastedLuminance;
    mediump float blackInterp = invContrastedLuminance*invContrastedLuminance*invContrastedLuminance;
    mediump float blackTarget = 1.0 - clamp(shadows, 0.0, 1.0);
    result = mix(result, vec3(0.0), blackInterp*blackTarget);

    return vec4(clamp(result, 0.0, 1.0), inputColor.w);
}
/* --- END HIGHLIGHTS SHADOWS--- */
/* --- START VIBRANCE--- */
uniform lowp float vibrance;

vec4 runVibrance(in vec4 inputColor) {
    lowp float average = (inputColor.r + inputColor.g + inputColor.b) / 3.0;
    lowp float mx = max(inputColor.r, max(inputColor.g, inputColor.b));
    lowp float amt = (mx - average) * (-vibrance * 3.0);
    inputColor.rgb = mix(inputColor.rgb, vec3(mx), amt);
    return inputColor;
}
/* --- END VIBRANCE--- */
/* --- START GRAIN--- */
uniform float grainTimer;
uniform float grainAmount;
uniform float grainSize;//grain particle size (1.5 - 2.5)
uniform float grainScale;

const float permTexUnit = 1.0/256.0;// Perm texture texel-size
const float permTexUnitHalf = 0.5/256.0;// Half perm texture texel-size

bool colored = false;//colored noise?
float coloramount = 0.0;
float lumamount = 1.0;//

//a random texture generator, but you can also use a pre-computed perturbation texture
vec4 rnm(in vec2 tc)
{
    float noise =  sin(dot(tc + vec2(grainTimer, grainTimer), vec2(12.9898, 78.233))) * 43758.5453;

    float noiseR = fract(noise)*2.0-1.0;
    float noiseG = fract(noise*1.2154)*2.0-1.0;
    float noiseB = fract(noise*1.3453)*2.0-1.0;
    float noiseA = fract(noise*1.3647)*2.0-1.0;

    return vec4(noiseR, noiseG, noiseB, noiseA);
}

float fade(in float t) {
    return t*t*t*(t*(t*6.0-15.0)+10.0);
}

float pnoise3D(in vec3 p)
{
    vec3 pi = permTexUnit*floor(p)+permTexUnitHalf;// Integer part, scaled so +1 moves permTexUnit texel
    // and offset 1/2 texel to sample texel centers
    vec3 pf = fract(p);// Fractional part for interpolation

    // Noise contributions from (x=0, y=0), z=0 and z=1
    float perm00 = rnm(pi.xy).a;
    vec3  grad000 = rnm(vec2(perm00, pi.z)).rgb * 4.0 - 1.0;
    float n000 = dot(grad000, pf);
    vec3  grad001 = rnm(vec2(perm00, pi.z + permTexUnit)).rgb * 4.0 - 1.0;
    float n001 = dot(grad001, pf - vec3(0.0, 0.0, 1.0));

    // Noise contributions from (x=0, y=1), z=0 and z=1
    float perm01 = rnm(pi.xy + vec2(0.0, permTexUnit)).a;
    vec3  grad010 = rnm(vec2(perm01, pi.z)).rgb * 4.0 - 1.0;
    float n010 = dot(grad010, pf - vec3(0.0, 1.0, 0.0));
    vec3  grad011 = rnm(vec2(perm01, pi.z + permTexUnit)).rgb * 4.0 - 1.0;
    float n011 = dot(grad011, pf - vec3(0.0, 1.0, 1.0));

    // Noise contributions from (x=1, y=0), z=0 and z=1
    float perm10 = rnm(pi.xy + vec2(permTexUnit, 0.0)).a;
    vec3  grad100 = rnm(vec2(perm10, pi.z)).rgb * 4.0 - 1.0;
    float n100 = dot(grad100, pf - vec3(1.0, 0.0, 0.0));
    vec3  grad101 = rnm(vec2(perm10, pi.z + permTexUnit)).rgb * 4.0 - 1.0;
    float n101 = dot(grad101, pf - vec3(1.0, 0.0, 1.0));

    // Noise contributions from (x=1, y=1), z=0 and z=1
    float perm11 = rnm(pi.xy + vec2(permTexUnit, permTexUnit)).a;
    vec3  grad110 = rnm(vec2(perm11, pi.z)).rgb * 4.0 - 1.0;
    float n110 = dot(grad110, pf - vec3(1.0, 1.0, 0.0));
    vec3  grad111 = rnm(vec2(perm11, pi.z + permTexUnit)).rgb * 4.0 - 1.0;
    float n111 = dot(grad111, pf - vec3(1.0, 1.0, 1.0));

    // Blend contributions along x
    vec4 n_x = mix(vec4(n000, n001, n010, n011), vec4(n100, n101, n110, n111), fade(pf.x));

    // Blend contributions along y
    vec2 n_xy = mix(n_x.xy, n_x.zw, fade(pf.y));

    // Blend contributions along z
    float n_xyz = mix(n_xy.x, n_xy.y, fade(pf.z));

    // We're done, return the final noise value.
    return n_xyz;
}

//2d coordinate orientation thing
vec2 coordRot(in vec2 tc, in float angle, in float scale)
{
    float aspect = scale;
    float rotX = ((tc.x*2.0-1.0)*aspect*cos(angle)) - ((tc.y*2.0-1.0)*sin(angle));
    float rotY = ((tc.y*2.0-1.0)*cos(angle)) + ((tc.x*2.0-1.0)*aspect*sin(angle));
    rotX = ((rotX/aspect)*0.5+0.5);
    rotY = rotY*0.5+0.5;
    return vec2(rotX, rotY);
}

vec4 runGrain(in vec4 inputColor) {
    vec3 rotOffset = vec3(1.425, 3.892, 5.835);//rotation offset values
    vec2 rotCoordsR = coordRot(vTextureCoord, grainTimer + rotOffset.x, grainScale);
    vec3 noise = vec3(pnoise3D(vec3(rotCoordsR*vTextureCoord*grainSize*grainScale, 0.0)));

    if (colored)
    {
        vec2 rotCoordsG = coordRot(vTextureCoord, grainTimer + rotOffset.y, grainScale);
        vec2 rotCoordsB = coordRot(vTextureCoord, grainTimer + rotOffset.z, grainScale);
        noise.g = mix(noise.r, pnoise3D(vec3(rotCoordsG*vTextureCoord*grainSize*grainScale, 1.0)), coloramount);
        noise.b = mix(noise.r, pnoise3D(vec3(rotCoordsB*vTextureCoord*grainSize*grainScale, 2.0)), coloramount);
    }

    highp vec3 col = inputColor.rgb;

    //noisiness response curve based on scene luminance
    highp vec3 lumcoeff = vec3(0.299, 0.587, 0.114);
    float luminance = mix(0.0, dot(col, lumcoeff), lumamount);
    float lum = smoothstep(0.2, 0.0, luminance);
    lum += luminance;


    noise = mix(noise, vec3(0.0), pow(lum, 4.0));
    col = col+noise*grainAmount;

    return vec4(col, inputColor.w);
}
/* --- END GRAIN--- */
/* --- START VIGNETTE--- */
uniform lowp vec2 vignetteCenter;
uniform highp float vignetteStart;
uniform highp float vignetteEnd;

vec4 runVignette(in vec4 inputColor) {
    lowp vec3 rgb = inputColor.rgb;
    lowp float d = distance(vTextureCoord, vec2(vignetteCenter.x, vignetteCenter.y));
    lowp float percent = smoothstep(vignetteStart, vignetteEnd, d);
    return vec4(mix(rgb.x, 0.0, percent), mix(rgb.y, 0.0, percent), mix(rgb.z, 0.0, percent), inputColor.w);
}
/* --- END VIGNETTE--- */
/* --- START MADE LOOKUP--- */
uniform sampler2D lutTexture;// lookup texture

uniform int isLutTextureLoaded;

uniform lowp float lutIntensity;
uniform lowp float lutDimension;

highp vec4 sampleAs3DTexture(sampler2D tex, highp vec3 texCoord, highp float size) {
    highp float x = texCoord.x;
    highp float y = texCoord.z;
    highp float z = texCoord.y;

    highp float sliceSize = 1.0 / size;// space of 1 slice
    highp float sliceTexelSize = sliceSize / size;// space of 1 pixel
    highp float texelsPerSlice = size - 1.0;
    highp float sliceInnerSize = sliceTexelSize * texelsPerSlice;// space of size pixels

    highp float zSlice0 = floor(z * texelsPerSlice);
    highp float zSlice1 = min(zSlice0 + 1.0, texelsPerSlice);

    highp float yRange = (y * texelsPerSlice + 0.5) / size;

    highp float xOffset = sliceTexelSize * 0.5 + x * sliceInnerSize;

    highp float z1 = zSlice1 * sliceSize + xOffset;

    #if defined(USE_NEAREST)
    return texture2D(tex, vec2(z0, yRange)).bgra;
    #else
    highp float z0 = zSlice0 * sliceSize + xOffset;
    highp vec4 slice0Color = texture2D(tex, vec2(z0, yRange));
    highp vec4 slice1Color = texture2D(tex, vec2(z1, yRange));
    highp float zOffset = mod(z * texelsPerSlice, 1.0);
    return mix(slice0Color, slice1Color, zOffset).bgra;
    #endif
}
vec4 runMadeLookup(in vec4 inputColor) {
    if (isLutTextureLoaded == 0) {
        return inputColor;
    } else if (inputColor.w == 0.0) {
        return vec4(1.0, 1.0, 1.0, 0.0);
    } else {
        highp vec4 newColor = sampleAs3DTexture(lutTexture, inputColor.rgb, lutDimension);
        //TODO newColor - 0.001
        newColor = newColor + 0.001;
        //TODO newColor + 0.001
        return vec4(mix(inputColor, newColor, lutIntensity).rgb, inputColor.w);
    }
}
/* --- END MADE LOOKUP--- */

uniform int enableBrightness;
uniform int enableContrast;
uniform int enableSharpness;
uniform int enableBlurGaussian;
uniform int enableSaturation;
uniform int enableExposure;
uniform int enableWhiteBalance;
uniform int enableHighlightsShadows;
uniform int enableVibrance;
uniform int enableGrain;
uniform int enableVignette;
uniform int enableLut;

void main() {
    highp vec4 textureColor = texture2D(sTexture, vTextureCoord);
    if (enableBlurGaussian == 1) {
        textureColor = runBlurGaussian(textureColor);
    }
    if (enableSharpness == 1) {
        textureColor = runSharpness(textureColor);
    }
    if (enableHighlightsShadows == 1) {
        textureColor = runHighlightsShadows(textureColor);
    }
    if (enableContrast == 1) {
        textureColor = runContrast(textureColor);
    }
    if (enableGrain == 1) {
        textureColor = runGrain(textureColor);
    }
    if (enableVignette == 1) {
        textureColor = runVignette(textureColor);
    }
    if (enableLut == 1) {
        textureColor = runMadeLookup(textureColor);
    }
    if (enableSaturation == 1) {
        textureColor = runSaturation(textureColor);
    }
    if (enableBrightness == 1) {
        textureColor = runBrightness(textureColor);
    }
    if (enableExposure == 1) {
        textureColor = runExposure(textureColor);
    }
    if (enableVibrance == 1) {
        textureColor = runVibrance(textureColor);
    }
    if (enableWhiteBalance == 1) {
        textureColor = runWhiteBalance(textureColor);
    }
    gl_FragColor = textureColor;
}