﻿//
//  BlockingQueue.h
//  gaeabase
//
//  Created by jinxi on 4/22/16.
//  Copyright © 2016 jinxi. All rights reserved.
//

#ifndef GAEA_BASE_BLOCKING_QUEUE_H_
#define GAEA_BASE_BLOCKING_QUEUE_H_

#include <deque>
#include <condition_variable>

#include "gaea/base/macros.h"

namespace gaea {
namespace base {

template <class T>
class BlockingQueue {
 public:
  BlockingQueue(size_t max_size = 1024) : max_size_(max_size), enable_max_size_limit_(true) {}
  ~BlockingQueue() {}

  void set_max_size(size_t max_size) { max_size_ = max_size; }
  size_t max_size() const { return max_size_; }
  void set_enable_max_size_limit(bool enable) { enable_max_size_limit_ = enable; }
  
  bool Empty() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return queue_.empty();
  }

  size_t Size() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return queue_.size();
  }

  bool Put(const T& value) {
    std::lock_guard<std::mutex> lock(mutex_);
    if (enable_max_size_limit_ && queue_.size() >= max_size_) {
      return false;
    }
    queue_.push_back(value);
    condition_.notify_one();
    return true;
  }

  bool Get(T* value) {
    if (value == nullptr) {
      return false;
    }
    std::unique_lock<std::mutex> lock(mutex_);
    if (max_size_ == 0) {
      return false;
    }
    if (queue_.empty()) {
      condition_.wait(lock);
    }
    bool has_element = !queue_.empty();
    if (has_element) {
      *value = queue_.front();
      queue_.pop_front();
    }
    return has_element;
  }

  /**
   * NOTE: undefined behavior when timeout_millis < 0
   */
  bool Get(T* value, int timeout_millis) {
    if (value == nullptr) {
      return false;
    }
    if (timeout_millis < 0) {
      timeout_millis = 0;
    }
    std::unique_lock<std::mutex> lock(mutex_);

    if (max_size_ == 0) {
      return false;
    }
    if (queue_.empty()) {
      condition_.wait_for(lock, std::chrono::milliseconds(timeout_millis));
    }
    bool has_element = !queue_.empty();
    if (has_element) {
        *value = queue_.front();
        queue_.pop_front();
    }
    return has_element;
  }

  bool RemoveOne(const T& value) {
    std::lock_guard<std::mutex> lock(mutex_);

    if (max_size_ == 0) {
      return false;
    }
    if (queue_.empty()) {
      return false;
    }

    for (auto iter = queue_.begin(); iter != queue_.end(); iter++) {
      if (*iter == value) {
        queue_.erase(iter);
        return true;
      }
    }
    return false;
  }

  /**
   * @function Wait
   * @return different platform with return different values, it's so always return true, cant ignore it 
   */
  bool Wait() {
    std::unique_lock<std::mutex> lock(mutex_);
    condition_.wait(lock);
    return true;
  }

  template <class Rep, class Period>
  bool WaitFor(const std::chrono::duration<Rep, Period>& duration) {
    std::unique_lock<std::mutex> lock(mutex_);
    return std::cv_status::no_timeout == condition_.wait_for(lock, duration);
  }

  bool TryGet(T* value) {
    if (value == nullptr) {
      return false;
    }

    std::lock_guard<std::mutex> lock(mutex_);

    if (queue_.empty()) {
      return false;
    }
    *value = queue_.front();
    queue_.pop_front();
    return true;
  }

  bool Front(T* value) {
    if (value == nullptr) {
      return false;
    }
    std::lock_guard<std::mutex> lock(mutex_);
    if (queue_.empty()) {
      return false;
    }
    *value = queue_.front();
    return true;
  }

  bool Pop() {
    std::lock_guard<std::mutex> lock(mutex_);
    if (queue_.empty()) {
      return false;
    }
    queue_.pop_front();
    return true;
  }

  void Clear() {
    std::lock_guard<std::mutex> lock(mutex_);
    queue_.clear();
  }

  void Stop() {
    std::lock_guard<std::mutex> lock(mutex_);
    max_size_ = 0;
    condition_.notify_all();
  }

 private:
  /** 最大任务缓冲数 */
  size_t max_size_;
  bool enable_max_size_limit_;
  
  mutable std::mutex mutex_;
  std::condition_variable condition_;
  
  /** 缓冲异步任务的队列 */
  std::deque<T> queue_;
};

} // end of namespace base
} // end of namespace gaea

#endif /* GAEA_BASE_BLOCKING_QUEUE_H_ */
