﻿//
//  Splitter.hpp
//  Atlas
//
//  Created by jinxi on 4/21/16.
//  Copyright © 2016 jinxi. All rights reserved.
//

#ifndef GAEA_BASE_SPLITTER_H_
#define GAEA_BASE_SPLITTER_H_

#include <list>
#include <iterator>

#include "gaea/base/string_util.h"

namespace gaea {
namespace base {

class Splitter {
 public:
  Splitter();
  ~Splitter();

 public:
  Splitter& On(char separator);
  Splitter& On(const std::string& separator);
  Splitter& WithKeyValueSeparator(char separator);
  Splitter& WithKeyValueSeparator(const std::string& separator);

  /*
   * @desc: When the fixedLength mode is enabled: the limit/omitEmptyString/trimResults parameter does not work
   *        splitToMap does not work
   */
  Splitter& FixedLength(int length);
  /*
   * @desc: Does not work under the following conditions
   *        1. fixedLength mode
   *        2. split to map
   */
  Splitter& Limit(int limit);
  /*
   * @desc: Does not work under the following conditions
   *        1. fixedLength mode
   *        2. split to map
   */
  Splitter& OmitEmptyString();
  /*
   * @desc: Does not work under the following conditions
   *        1. fixedLength mode
   *        2. split to map
   */
  Splitter& TrimResults(const std::string& chars = " \r\n\t\0\x0B");

  Strings SplitToStrings(const std::string& str) const {
    Strings result;
    SplitListImpl(str, std::back_inserter(result));
    return result;
  }

  StringList SplitToList(const std::string& str) const {
    StringList result;
    SplitListImpl(str, std::back_inserter(result));
    return result;
  }

  StringSet SplitToSet(const std::string& str) const {
    StringSet result;
    SplitListImpl(str, std::inserter(result, result.begin()));
    return result;
  }

  StringMap SplitToMap(const std::string& str) const {
    StringMap result;
    Strings items = SplitToStrings(str);

    Strings::const_iterator it = items.begin();
    Strings::const_iterator itEnd = items.end();
    for (; it != itEnd; ++it) {
      const std::string& item = *it;
      size_t pos = item.find(key_value_separator_);
      if (pos == std::string::npos) {
        continue;
      }
      result[item.substr(0, pos)] = item.substr(pos + key_value_separator_.size());
    }
    return result;
  }

  template <class OutputIterator>
  void SplitList(const std::string& str, OutputIterator inserter) const {
    SplitListImpl(str, inserter);
  }

 private:
  template <class OutputIterator>
  void SplitListImpl(const std::string& str, OutputIterator inserter) const {
    size_t startPos = 0;
    size_t pos = 0;

    if (fixed_length_ > 0) {
      while (startPos < str.length()) {
        if (startPos + fixed_length_ <= str.length()) {
          *inserter = str.substr(startPos, fixed_length_);
          startPos += fixed_length_;
        } else {
          *inserter = str.substr(startPos);
          break;
        }
      }
      return;
    }

    if (separator_.empty() || str.empty()) {
      return;
    }
    int count = 0;
    std::string item;
    do {
      pos = str.find(separator_.data(), startPos, separator_.size());
      if (pos != std::string::npos) {
        if ((!omit_empty_string_) || (pos > startPos)) {
          if (trim_results_) {
            item = Trim(str.substr(startPos, pos - startPos));
            if ((!omit_empty_string_) || item.size()) {
              *inserter = item;
              if (limit_ > 0 && limit_ <= ++count) {
                startPos = pos + separator_.length();
                break;
              }
            }
          } else {
            *inserter = str.substr(startPos, pos - startPos);
            if (limit_ > 0 && limit_ <= ++count) {
              startPos = pos + separator_.length();
              break;
            }
          }
        }
        startPos = pos + separator_.length();
      } else {
        if ((!omit_empty_string_) || (startPos < str.length())) {
          if (trim_results_) {
            item = Trim(str.substr(startPos));
            if ((!omit_empty_string_) || item.size()) {
              *inserter = item;
            }
          } else {
            *inserter = str.substr(startPos);
          }
        }
        startPos = std::string::npos;
      }
    } while (pos != std::string::npos);

    if (startPos != std::string::npos && startPos < str.length()) {
      if (trim_results_) {
        item = Trim(str.substr(startPos));
        if ((!omit_empty_string_) || item.size()) {
          *inserter = item;
        }
      } else {
        *inserter = str.substr(startPos);
      }
    }
  }

  std::string Trim(const std::string& str) const;

 private:
  int fixed_length_;
  int limit_;
  bool omit_empty_string_;
  bool trim_results_;
  int8_t trim_char_table_[256];

  std::string separator_;
  std::string key_value_separator_;
};

} // end of namespace base
} // end of namespace gaea

#endif /* GAEA_BASE_SPLITTER_H_ */
