﻿//
//  system_util.h
//  Atlas
//
//  Created by 金喜 on 4/21/16.
//  Copyright © 2016 金喜. All rights reserved.
//

#ifndef GAEA_BASE_SYSTEM_UTIL_H_
#define GAEA_BASE_SYSTEM_UTIL_H_

#include <cstdint>
#include <string>
#include <vector>
#include <gaea/base/error_result.h>
#include <gaea/base/macros.h>

namespace gaea {
namespace base {

class SystemLoadAverage {
 public:
  SystemLoadAverage(): load1_(0), load5_(0), load15_(0) {};
  ~SystemLoadAverage() = default;
 public:
  double load1() const { return load1_; }
  double load5() const { return load5_; }
  double load15() const { return load15_; }

  void set_load1(double load1) { load1_ = load1; }
  void set_load5(double load5) { load5_ = load5; }
  void set_load15(double load15) { load15_ = load15; }
  
  std::string ToString() const;

 private:
  double load1_;
  double load5_;
  double load15_;
};

class SystemMemoryUsage {
 public:
  SystemMemoryUsage() : total_(0), available_(0), used_(0), free_(0) {}
  ~SystemMemoryUsage() = default;
 public:
  int64_t total() const { return total_; }
  int64_t available() const { return available_; }
  int64_t used() const { return used_; }
  int64_t free() const { return free_; }

  void set_total(int64_t total) { total_ = total; }
  void set_available(int64_t available) { available_ = available; }
  void set_used(int64_t used) { used_ = used; }
  void set_free(int64_t free) { free_ = free; }

  std::string ToString() const;

 private:
  int64_t total_; // 物理内存总大小
  int64_t available_;
  int64_t used_;
  int64_t free_;
};

class SystemDiskPartition {
 public:
  SystemDiskPartition() {}
  ~SystemDiskPartition() = default;

 public:
  const std::string& device() const { return device_; }
  const std::string& mount_point() const { return mount_point_; }
  const std::string& fstype() const { return fstype_; }
  const std::string& options() const { return options_; }

  void set_device(const std::string& device) { device_ = device;}
  void set_mount_point(const std::string& mount_point) { mount_point_ = mount_point;}
  void set_fstype(const std::string& fstype) { fstype_ = fstype;}
  void set_options(const std::string& options) { options_ = options;}

  std::string ToString() const;

 private:
  std::string device_;
  std::string mount_point_;
  std::string fstype_;
  std::string options_;
};

class SystemDiskUsage {
 public:
  SystemDiskUsage() : total_(0), used_(0), free_(0), percent_(0) {}
  ~SystemDiskUsage() = default;

 public:
  int64_t total() const { return total_; }
  int64_t used() const { return used_; }
  int64_t free() const { return free_; }
  double percent() const { return percent_; }

  void set_total(int64_t total) { total_ = total; }
  void set_used(int64_t used) { used_ = used; }
  void set_free(int64_t free) { free_ = free; }
  void set_percent(double percent) { percent_ = percent; }

  std::string ToString() const;

 private:
  int64_t total_;
  int64_t used_;
  int64_t free_;
  double percent_;
};

class SystemNetworkConnection {
 public:
  SystemNetworkConnection() : fd_(0), family_(0), type_(0), local_port_(0), remote_port_(0), status_(0), pid_(0) {}
  ~SystemNetworkConnection() = default;

  int fd() const { return fd_; }
  int family() const { return family_; }
  int type() const { return type_; }
  const std::string& local_ip() const { return local_ip_; }
  int local_port() const { return local_port_; }
  const std::string& remote_ip() const { return remote_ip_; }
  int remote_port() const { return remote_port_; }
  int status() const { return status_; }
  int pid() const { return pid_; }

  void set_fd(int fd) { fd_ = fd; }
  void set_family(int family) { family_ = family; }
  void set_type(int type) { type_ = type; }
  void set_local_ip(const std::string& local_ip) { local_ip_ = local_ip; }
  void set_local_port(int local_port) { local_port_ = local_port; }
  void set_remote_ip(const std::string& remote_ip) { remote_ip_ = remote_ip; }
  void set_remote_port(int remote_port) { remote_port_ = remote_port; }
  void set_status(int status) { status_ = status; }
  void set_pid(int pid) { pid_ = pid; }

  std::string ToString() const;

 private:
  int fd_;
  int family_;
  int type_;
  std::string local_ip_;
  int local_port_;
  std::string remote_ip_;
  int remote_port_;
  int status_;
  int pid_;
};

class SystemNetworkInterface {
 public:
  SystemNetworkInterface() : family_(0) {};
  ~SystemNetworkInterface() = default;

 public:
  const std::string& name() const { return name_; }
  int family() const { return family_; }
  const std::string& address() const { return address_; }
  const std::string& netmask() const { return netmask_; }
  const std::string& broadcast() const { return broadcast_; }
  const std::string& ptp() const { return ptp_; }

  void set_name(const std::string& name) { name_ = name; }
  void set_family(int family) { family_ = family; }
  void set_address(const std::string& address) { address_ = address; }
  void set_netmask(const std::string& netmask) { netmask_ = netmask; }
  void set_broadcast(const std::string& broadcast) { broadcast_ = broadcast; }
  void set_ptp(const std::string& ptp) { ptp_ = ptp; }

  std::string ToString() const;

 private:
  std::string name_;
  int family_;
  std::string address_;
  std::string netmask_;
  std::string broadcast_;
  std::string ptp_;
};

class SystemSensorsBattery {
 public:
  enum { kPowerTimeUnlimited = -1, kPowerTimeUnknown = -2, };
  enum PowerStatus { kPowerConnected, kPowerDisconnected, kPowerCannotDetermined, };
 public:
  SystemSensorsBattery() : percent_(0), left_seconds_(kPowerTimeUnknown), power_plugged_(kPowerCannotDetermined) {}
  ~SystemSensorsBattery() = default;

 public:
  double percent() const { return percent_; }
  int64_t left_seconds() const { return left_seconds_; }
  PowerStatus power_plugged() const { return power_plugged_; }

  void set_percent(double percent) { percent_ = percent; }
  void set_left_seconds(int64_t left_seconds) { left_seconds_ = left_seconds; }
  void set_power_plugged(PowerStatus power_plugged) { power_plugged_ = power_plugged; }

  std::string ToString() const;

 private:
  double percent_; // [0-100]
  // kPowerTimeUnlimited on AC power cable connected
  // kPowerTimeUnknown cannot be determined
  int64_t left_seconds_;
  PowerStatus power_plugged_;
};

class SystemUser {
 public:
  enum { kStartedTimeNone = -1, };
  enum { kPidNone = -1, };

 public:
  SystemUser() : started_(kStartedTimeNone), pid_(kPidNone) {}
  ~SystemUser() = default;

 public:
  const std::string& name() const { return name_; }
  const std::string& terminal() const { return terminal_; }
  const std::string& host() const { return host_; }
  int64_t started() const { return started_; }
  int pid() const { return pid_; }

  void set_name(const std::string& name) { name_ = name; }
  void set_terminal(const std::string& terminal) { terminal_ = terminal; }
  void set_host(const std::string& host) { host_ = host; }
  void set_started(int64_t started) { started_ = started; }
  void set_pid(int pid) { pid_ = pid; }

  std::string ToString() const;

 private:
  std::string name_;
  std::string terminal_;
  std::string host_;
  int64_t started_;
  int pid_;
};

class SystemUtil {
 public:
  static std::string GetHostname(bool fqdn = false);

  static void SetThreadName(const std::string& thread_name);
  static std::string GetThreadName();
  
  /**
   * @function RandomInt
   * @brief 返回 [0, bound] 闭包区间的随机数
   *        如果 bound <= 0 返回 0
   * @param min 最小闭包值
   * @param max 最大闭包值
   * @return 返回对应的随机数
   */
  static int RandomInt(int min, int max);

  static bool GetLogicalCpuCount(int* count, ErrorResult* error = nullptr);
  static bool GetPhysicalCpuCount(int* count, ErrorResult* error = nullptr);
  static bool GetLoadAverage(SystemLoadAverage* load_average, ErrorResult* error = nullptr);
  static bool GetMemoryUsage(SystemMemoryUsage* usage, ErrorResult* error = nullptr);
  static bool GetDiskPartitions(std::vector<SystemDiskPartition>* partitions, ErrorResult* error = nullptr);
  static bool GetDiskUsage(const std::string& mount_point, SystemDiskUsage* usage, ErrorResult* error = nullptr);
  static bool GetNetworkConnections(std::vector<SystemNetworkConnection>* connections, ErrorResult* error = nullptr);
  static bool GetNetworkInterfaces(std::vector<SystemNetworkInterface>* interfaces, ErrorResult* error = nullptr);
  static bool GetSensorsBattery(SystemSensorsBattery* battery, ErrorResult* error = nullptr);
  static bool GetBootTimestamp(int64_t* boot_timestamp, ErrorResult* error = nullptr);
  static bool GetConnectedUsers(std::vector<SystemUser>* users, ErrorResult* error = nullptr);
};

} // end of namespace base
} // end of namespace gaea

#endif /* GAEA_BASE_UTIL_SYSTEM_UTIL_H_ */
