/*
 * =====================================================================================
 *
 *       Filename:  module_json_helper.h
 *
 *    Description:  An auxiliary class that converts the model to json.
 *                  It is not needed outside. It is only used for template compilation.
 *        Version:  1.0
 *        Created:  2019-05-05
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  jack.kj@alibaba-inc.com
 *
 * =====================================================================================
 */
#ifndef GAEA_IDL_MODULE_JSON_HELPER_H_
#define GAEA_IDL_MODULE_JSON_HELPER_H_

#include <functional>
#include <map>
#include <string>
#include <vector>

#ifndef GAEA_IDL_DISABLE_JSON
namespace gaea {
namespace idl {

class BaseModel;
struct JsonSerializeContext;
struct JsonDeSerializeContext;

template <typename Type>
struct StringConverter {
  static std::string ToString(Type value) { return std::to_string(value); }
};

template <>
struct StringConverter<std::string> {
  static const std::string& ToString(const std::string& value) { return value; }
};

class ModelJsonHelper {
public:
  template <typename T>
  static bool ToJson(const std::string &key, const std::vector<T> &value, JsonSerializeContext *context) {
    SerializeList(key, context, [&value](JsonSerializeContext *ctx) {
      for (size_t i = 0; i < value.size(); ++i) {
        PushToArray(value[i], ctx);
      }
    });
    return true;
  }

  template <typename T>
  static bool FromJson(const JsonDeSerializeContext* context, std::vector<T>* output) {
    output->reserve(GetSize(context));
    DeserializeListElement list = [&output](const JsonDeSerializeContext& key) -> bool {
      T t;
      if (!FromJson(&key, &t)) {
        return false;
      }
      output->emplace_back(std::move(t));
      return true;
    };

    return IterateValue(context, list);
  }

  template <typename Key, typename Value>
  static bool ToJson(const std::string &key, const std::map<Key, Value> &value, JsonSerializeContext* context) {
    return SerializeMap(key, context, [&value](JsonSerializeContext* ctx)->bool {
      for (auto &it : value) {
        if (!ToJson(StringConverter<Key>::ToString(it.first), it.second, ctx)) {
          return false;
        }
      }
      return true;
    });
  }

  template <typename Key, typename Value>
  static bool FromJson(const JsonDeSerializeContext* context, std::map<Key, Value>* out) {

    DeserializeMapElement deserializeMapElementFunc = [&out](
      const JsonDeSerializeContext& key, const JsonDeSerializeContext&value) -> bool {
      Key mapKey;
      if (!FromJson(&key, &mapKey)) {
        return false;
      }

      Value mapValue;
      if (!FromJson(&value, &mapValue)) {
        return false;
      }

      out->emplace(std::make_pair(std::move(mapKey), std::move(mapValue)));
      return true;
    };

    return IterateValue(context, deserializeMapElementFunc);
  }

  static bool ToJsonBytes(const std::string &key, const std::string &val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, const std::string &val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, int64_t val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, int32_t val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, bool val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, char val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, float val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, double val, JsonSerializeContext* output);
  static bool ToJson(const std::string &key, const BaseModel& val, JsonSerializeContext* output);

  static bool FromJsonBytes(const JsonDeSerializeContext* context, std::string* out);
  static bool FromJson(const JsonDeSerializeContext* context, std::string* out);
  static bool FromJson(const JsonDeSerializeContext* context, char* out);
  static bool FromJson(const JsonDeSerializeContext* context, int32_t* out);
  static bool FromJson(const JsonDeSerializeContext* context, int64_t* out);
  static bool FromJson(const JsonDeSerializeContext* context, float* out);
  static bool FromJson(const JsonDeSerializeContext* context, double* out);
  static bool FromJson(const JsonDeSerializeContext* context, bool* out);
  static bool FromJson(const JsonDeSerializeContext* context, BaseModel* out);

private:
  static bool PushToArray(const BaseModel& model, JsonSerializeContext* output);
  static bool PushToArray(const std::string& str, JsonSerializeContext* output);
  static bool PushToArray(int32_t value, JsonSerializeContext* output);

  typedef std::function<void (JsonSerializeContext*)> SerializeListElement;
  static void SerializeList(const std::string& key, JsonSerializeContext* context,
          const SerializeListElement& serializeElement);
  typedef std::function<bool (JsonSerializeContext*)> SerializeMapElement;
  static bool SerializeMap(const std::string &key, JsonSerializeContext *context,
                            const SerializeMapElement &serializeElement);

  typedef std::function<bool (
      const JsonDeSerializeContext&, const JsonDeSerializeContext&)> DeserializeMapElement;
  static bool IterateValue(const JsonDeSerializeContext* context,
      const DeserializeMapElement& deserializeElement);

  typedef std::function<bool (const JsonDeSerializeContext&)> DeserializeListElement;
  static bool IterateValue(const JsonDeSerializeContext* context,
                           const DeserializeListElement& deserializeElement);

  static uint32_t GetSize(const JsonDeSerializeContext* context);
};
} // namespace idl
} // namespace gaea

#endif
#endif
