/*
 * =====================================================================================
 *
 *       Filename:  model_value.h
 *
 *    Description:  Encapsulates some basic types, models, and composite types of values.
 *
 *        Version:  1.0
 *        Created:  2019-04-29
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  jack.kj@alibaba-inc.com
 *
 * =====================================================================================
 */
#ifndef GAEA_IDL_MODEL_VALUE_H_
#define GAEA_IDL_MODEL_VALUE_H_

#include <algorithm>
#include <cstring>
#include <map>
#include <string>
#include "gaea/base/string_util.h"
#include "gaea/idl/base_model_value.h"

#ifndef GAEA_IDL_DISABLE_JSON
#include "gaea/idl/model_json_helper.h"
#endif

#include "gaea/idl/model_msgpack_helper.h"
#include "gaea/idl/msgpack_helper.h"

namespace gaea {
namespace idl {

template <typename Type, bool is_class = std::is_class<Type>::value>
struct Clearer {
};

template <typename Type>
struct Clearer<Type, false> {
  void operator() (Type& value) {
    memset(&value, 0, sizeof(Type));
  }
};

template <typename Type>
struct Clearer<Type, true> {
  void operator() (Type& value) {
    value.clear();
  }
};

template <typename Type, bool is_class = std::is_class<Type>::value>
struct Initializer {
};

template <typename Type>
struct Initializer<Type, false> {
  inline static void init(Type& type) {
    memset(&type, 0, sizeof(Type));
  }
};

template <typename Type>
struct Initializer<Type, true> {
  inline static void init(Type& type) {
  }
};

template <class T>
class ModelValue : public BaseModelValue {
public:
  ModelValue() {
    // 这里最好的方式是进行初始化列表，但是目前没有更优雅的方式，对于基本的POD类型来说这样多赋值性能也不会太大的差距
    Initializer<T>::init(value_);
  }
  ~ModelValue() override = default;

  void Clear() override {
    set_is_present(false);
    Clearer<T> clearer;
    clearer(value_);
  }

  inline void Set(const T &value) {
    value_ = value;
    set_is_present(true);
  }

  inline void Set(T &&value) {
    value_ = value;
    set_is_present(true);
  }

  inline T *Mutable() {
    set_is_present(true);
    return &value_;
  }

  inline const T &Get() const { return value_; }

  inline bool operator==(const T &value) const {
    if (!IsPresent()) {
      return false;
    }

    return value_ == value;
  }

  // 这里为了使用性的方便性，去掉了explicit，
  // 注意这里使用不当可能会有隐式转换出现的奇怪错误！！！
  operator const T& () const { return Get(); }

  inline ModelValue<T>& operator = (const T& value) {
    Set(value);
    return *this;
  }

  inline ModelValue<T>& operator = (T&& value) {
    Set(value);
    return *this;
  }

  void MergeFrom(const BaseModelValue& other) override {
    if (!other.IsPresent() || IsPresent()) {
      return;
    }

    auto other_ptr = dynamic_cast<ModelValue<T>*>(&const_cast<BaseModelValue&>(other));
    Set(other_ptr->Get());
  }

protected:
  bool Pack(cmp_ctx_t *cmp) override {
    return ModelMsgpackHelper::Pack(this->value_, cmp);
  }

  bool DoUnpack(cmp_ctx_t *context, bool *is_nil, bool *is_missing_fields) override {
    return ModelMsgpackHelper::Unpack(&value_, context, is_nil, is_missing_fields);
  }

#ifndef GAEA_IDL_DISABLE_JSON
    bool ToJson(const std::string &key, JsonSerializeContext* context) const override {
    return ModelJsonHelper::ToJson(key, value_, context);
  }

  bool DoFromJson(const JsonDeSerializeContext* context) override {
    return ModelJsonHelper::FromJson(context, Mutable());
  }
#endif

protected:
  T value_;
};

typedef ModelValue<int32_t > Int32ModelValue;
typedef ModelValue<int64_t > Int64ModelValue;
typedef ModelValue<double >  DoubleModelValue;
typedef ModelValue<float >   FloatModelValue;
typedef ModelValue<char >    CharModelValue;
typedef ModelValue<bool >    BooleanModelValue;
typedef ModelValue<std::string> StringModelValue;

template<typename T>
using ListModelValue = ModelValue< std::vector<T> >;

template <typename Key, typename Value>
using MapModelValue = ModelValue< std::map<Key, Value> >;

class ByteArrayModelValue : public StringModelValue {
public:
  ByteArrayModelValue();
  ~ByteArrayModelValue() override;

  inline ByteArrayModelValue & operator = (const std::string& value) {
    this->Set(value);
    return *this;
  }

  inline ByteArrayModelValue& operator = (std::string&& value) {
    this->Set(std::move(value));
    return *this;
  }

  inline void Assign(const char* bytes, int length) {
    value_.assign(bytes, length);
  }

protected:
#ifndef GAEA_IDL_DISABLE_JSON
  bool ToJson(const std::string &key, JsonSerializeContext* context) const final;
  bool DoFromJson(const JsonDeSerializeContext* context) final;
#endif
};

template<typename T>
class ObjectModelValue : public ModelValue<T> {
public:
  ObjectModelValue() = default;
  ~ObjectModelValue() override = default;

  ObjectModelValue(const ObjectModelValue<T>& value) {
    this->value_ = value.Get();
    this->set_is_present(value.IsPresent());
  }

  inline ObjectModelValue<T>& operator = (const T& value) {
    this->Set(value);
    return *this;
  }

  inline ObjectModelValue<T>& operator = (T&& value) {
    this->Set(std::move(value));
    return *this;
  }

protected:
  void MergeFrom(const BaseModelValue& other) override {
    if (!other.IsPresent()) {
      return;
    }
    auto other_ptr = dynamic_cast<ModelValue<T>*>(&const_cast<BaseModelValue&>(other));
    this->value_.MergeFrom(other_ptr->Get());
    this->set_is_present(true);
  }
};

} // namespace idl
} // namespace gaea

#endif
