//
//  request_handler.h
//  Atlas
//
//  Created by jinxi on 7/11/16.
//  Copyright © 2016 jinxi. All rights reserved.
//

#ifndef GAEA_LWP_REQUEST_HANDLER_H_
#define GAEA_LWP_REQUEST_HANDLER_H_

#include <functional>
#include <string.h>
#include <memory>

#include "gaea/lwp/gaea_define.h"
#include "gaea/idl/service_exception.h"
#include "gaea/lwp/request_context.h"
#include "gaea/lwp/response.h"
#include "gaea/lwp/error_result_helper.h"
#include "gaea/base/string_util.h"
#include "gaea/base/error_result.h"
#include "gaea/base/base64.h"

#include "gaea/lwp/gaea_define.h"

GAEA_LWP_NAMESPACE_BEGIN

gaea::base::ErrorResult RebuildIdlFailureError(ResponsePtr response);
  
template <class ResultType>
class RawRequestHandler : public RequestContext {
 public:
  typedef std::function<void(const ResultType& result,
                             RequestPtr request,
                             ResponsePtr response)> SuccessCallback;
  typedef std::function<void(const ResultType& result,
                             RequestPtr request,
                             ResponsePtr response)> PartialSuccessCallback;
  
  typedef std::function<void(const gaea::base::ErrorResult& error,
                             RequestPtr request,
                             ResponsePtr response)> FailureCallback;

  RawRequestHandler(SuccessCallback onSuccess, FailureCallback onFailure)
      : on_success_(onSuccess), on_partial_success_(nullptr), on_failure_(onFailure) {
  }
  RawRequestHandler(SuccessCallback onSuccess, PartialSuccessCallback on_partial_success, FailureCallback onFailure)
      : on_success_(onSuccess), on_partial_success_(on_partial_success), on_failure_(onFailure) {
  }
  ~RawRequestHandler() {}
  
  void SetBizCallback(SuccessCallback onSuccess, PartialSuccessCallback on_partial_success, FailureCallback onFailure) {
    on_success_ = onSuccess;
    on_partial_success_ = on_partial_success;
    on_failure_ = onFailure;
  }
  
  void OnPartialSuccess(RequestPtr request, ResponsePtr response) override {
    ResultType result;
    if (!Deserialize(request, response, &result)) {
      return;
    }
    if (on_partial_success_) {
      GAEA_LOG_DEBUG(logger_, "[idl] partial success, uri=" << request->request_line() << ", mid=" <<
      request->mid().Dumps() << ", siteId=" << request->site_id());
      on_partial_success_(result, request, response);
    }
  }

  void OnSuccess(RequestPtr request, ResponsePtr response) override {
    ResultType result;
    if (!Deserialize(request, response, &result)) {
      return;
    }
    if (on_success_ != nullptr) {
      GAEA_LOG_DEBUG(logger_, "[idl] success, uri=" << request->request_line() << ", mid=" <<
                    request->mid().Dumps() << ", siteId=" << request->site_id());
      on_success_(result, request, response);
    }
  }

  void OnFailure(RequestPtr request, ResponsePtr response) override {
    gaea::base::ErrorResult error = RebuildIdlFailureError(response);
    set_biz_error_result(error);
    if (on_failure_ != nullptr) {
      on_failure_(error, request, response);
      GAEA_LOG_INFO(logger_, "[idl] service_exception, uri=" << request->request_line()
                    << ", mid=" << request->mid().Dumps()
                    << ", siteId=" << request->site_id()
                    << ", error=" << error.ToString());
    }
  }
  
private:
  RawRequestHandler() {}
  bool Deserialize(RequestPtr request, ResponsePtr response, ResultType* result) {
    std::string body(response->body());
    BeforeUnpack();
    bool unpack_result = gaea::idl::ModelMsgpackHelper::Unpack(body, result, &is_missing_fields_);
    AfterUnpack();
    if (!allow_empty_body() && !unpack_result) {
      set_unpack_status(kIdlUnpackError);
      if (GAEA_LOG_DEBUG_ENABLE(logger_)) {
        std::string base64_str = gaea::base::Base64::Encode(body);
        GAEA_LOG_ERROR(logger_, "[idl] unpackb response.body failed"
                << ", uri=" << request->request_line()
                << ", body-base64=" << base64_str);
      } else {
        GAEA_LOG_ERROR(logger_, "[idl] unpackb response.body failed"
                << ", uri=" << request->request_line()
                << ", body.size=" << body.size());
      }
      gaea::base::ErrorResult error = gaea::lwp::ErrorResultHelper::BuildUnpackError();
      set_biz_error_result(error);
      if (on_failure_ != nullptr) {
        on_failure_(error, request, response);
        GAEA_LOG_INFO(logger_, "[idl] failure, unpackb_exception, uri=" << request->request_line() <<
                      ", mid=" << request->mid().Dumps() << ", siteId=" << request->site_id());
      }
      return false;
    }
    return true;
  }

private:
  SuccessCallback on_success_;
  PartialSuccessCallback on_partial_success_;
  FailureCallback on_failure_;
};

template <class ResultType>
class RequestHandler : public RequestContext {
public:
  typedef std::function<void(const ResultType& result)> SuccessCallback;
  typedef std::function<void(const gaea::base::ErrorResult& error)> FailureCallback;

public:
  RequestHandler(SuccessCallback onSuccess, FailureCallback onFailure) : on_success_(onSuccess), on_failure_(onFailure) {}
  ~RequestHandler() {}
  
  void SetBizCallback(SuccessCallback onSuccess, FailureCallback onFailure) {
    on_success_ = onSuccess;
    on_failure_ = onFailure;
  }

private:
  RequestHandler() {}
public:
  void OnSuccess(RequestPtr request, ResponsePtr response) {
    ResultType result;
    std::string body(response->body());
    BeforeUnpack();
    bool unpack_result = gaea::idl::ModelMsgpackHelper::Unpack(body, &result, &is_missing_fields_);
    AfterUnpack();
    if (!allow_empty_body() && !unpack_result) {
      set_unpack_status(kIdlUnpackError);
      if (GAEA_LOG_DEBUG_ENABLE(logger_)) {
        std::string base64_str = gaea::base::Base64::Encode(body);
        GAEA_LOG_ERROR(logger_, "[idl] unpackb response.body failed"
                       << ", uri=" << request->request_line()
                       << ", body-base64=" << base64_str);
      } else {
        GAEA_LOG_ERROR(logger_, "[idl] unpackb response.body failed"
                       << ", uri=" << request->request_line()
                       << ", body.size=" << body.size());
      }
      gaea::base::ErrorResult error = gaea::lwp::ErrorResultHelper::BuildUnpackError();
      set_biz_error_result(error);
      if (on_failure_ != nullptr) {
        on_failure_(error);
      }
      GAEA_LOG_INFO(logger_, "[idl] unpackb_exception, uri=" << request->request_line() <<
                    ", mid=" << request->mid().Dumps() << ", siteId=" << request->site_id());
      return;
    }

    if (on_success_ != nullptr) {
      on_success_(result);
      GAEA_LOG_DEBUG(logger_, "[idl] success, uri=" << request->request_line() << ", mid=" <<
                    request->mid().Dumps() << ", siteId=" << request->site_id() << ", code=" << response->code());
    }
  }
  void OnFailure(RequestPtr request, ResponsePtr response) {
    gaea::base::ErrorResult error = RebuildIdlFailureError(response);
    set_biz_error_result(error);
    if (on_failure_ != nullptr) {
      on_failure_(error);
    }
    GAEA_LOG_INFO(logger_, "[idl] service_exception, uri=" << request->request_line() << ", mid=" << request->mid().Dumps()
                  << ", siteId=" << request->site_id()
                  << ", error=" << error.ToString());
  }

private:
  SuccessCallback on_success_;
  FailureCallback on_failure_;
};

template <>
class RequestHandler<void> : public RequestContext {
public:
  typedef std::function<void(void)> SuccessCallback;
  typedef std::function<void(const gaea::base::ErrorResult& error)> FailureCallback;

public:
  RequestHandler(SuccessCallback onSuccess, FailureCallback onFailure) : on_success_(onSuccess), on_failure_(onFailure) {}
  ~RequestHandler() {}
  
  void SetBizCallback(SuccessCallback onSuccess, FailureCallback onFailure) {
    on_success_ = onSuccess;
    on_failure_ = onFailure;
  }

private:
  RequestHandler() {}

public:
  void OnSuccess(RequestPtr request, ResponsePtr response) {
    if (on_success_ != nullptr) {
      on_success_();
      GAEA_LOG_DEBUG(logger_, "[idl] success, uri=" << request->request_line()
                    << ", mid=" << request->mid().Dumps()
                    << ", siteId=" << request->site_id()
                    << ", code=" << response->code());
    }
  }
  void OnFailure(RequestPtr request, ResponsePtr response) {
    gaea::base::ErrorResult error = RebuildIdlFailureError(response);
    set_biz_error_result(error);
    if (on_failure_ != nullptr) {
      on_failure_(error);
      GAEA_LOG_INFO(logger_, "[idl] service_exception"
                    << ", uri=" << request->request_line()
                    << ", mid=" << request->mid().Dumps()
                    << ", siteId=" << request->site_id()
                    << ", error=" << error.ToString());
    }
  }

private:
  SuccessCallback on_success_;
  FailureCallback on_failure_;
};

template <class ResultType>
class RawDyeRequestHandler : public RequestContext {
 public:
  typedef std::function<void(const ResultType& result,
                             RequestPtr request,
                             ResponsePtr response,
                             const RequestContext& context)> SuccessCallback;
  
  typedef std::function<void(const gaea::base::ErrorResult& error,
                             RequestPtr request,
                             ResponsePtr response,
                             const RequestContext& context)> FailureCallback;

  RawDyeRequestHandler(SuccessCallback onSuccess, FailureCallback onFailure)
      : on_success_(onSuccess), on_failure_(onFailure) {
  }
  ~RawDyeRequestHandler() {}

  void SetBizCallback(SuccessCallback onSuccess, FailureCallback onFailure) {
    on_success_ = onSuccess;
    on_failure_ = onFailure;
  }
  
  void OnSuccess(RequestPtr request, ResponsePtr response) override {
    ResultType result;
    std::string body(response->body());
    set_trace_context(request->trace_context());
    BeforeUnpack();
    bool unpack_result = gaea::idl::ModelMsgpackHelper::Unpack(body, &result, &is_missing_fields_);
    AfterUnpack();
    if (!allow_empty_body() && !unpack_result) {
      set_unpack_status(kIdlUnpackError);
      GAEA_LOG_ERROR(logger_, "[idl] unpackb response.body failed, uri=" <<
                     request->request_line() << ", response.body.size=" << response->body().size());
      gaea::base::ErrorResult error = gaea::lwp::ErrorResultHelper::BuildUnpackError();
      set_biz_error_result(error);
      if (on_failure_ != nullptr) {
        on_failure_(error, request, response, *this);
        GAEA_LOG_INFO(logger_, "[idl] failure, unpackb_exception, uri=" << request->request_line() <<
                      ", mid=" << request->mid().Dumps() << ", siteId=" << request->site_id());
      }
      return;
    }
    if (on_success_ != nullptr) {
      GAEA_LOG_DEBUG(logger_, "[idl] success, uri=" << request->request_line() << ", mid=" <<
                    request->mid().Dumps() << ", siteId=" << request->site_id());
      on_success_(result, request, response, *this);
    }
  }
//  void OnPartialSuccess(RequestPtr request, ResponsePtr response) override {
//    set_trace_context(request->trace_context());
//    OnSuccess(request, response);
//  }

  void OnFailure(RequestPtr request, ResponsePtr response) override {
    gaea::base::ErrorResult error = RebuildIdlFailureError(response);
    set_trace_context(request->trace_context());
    set_biz_error_result(error);
    if (on_failure_ != nullptr) {
      on_failure_(error, request, response, *this);
      GAEA_LOG_INFO(logger_, "[idl] service_exception, uri=" << request->request_line()
                    << ", mid=" << request->mid().Dumps()
                    << ", siteId=" << request->site_id()
                    << ", error=" << error.ToString());
    }
  }
  
private:
  RawDyeRequestHandler() {}

private:
  SuccessCallback on_success_;
  FailureCallback on_failure_;
};
  
GAEA_LWP_NAMESPACE_END

#endif /* GAEA_LWP_REQUEST_HANDLER_H_ */
