//
//  abstract_connection.hpp
//  GaeaLwp
//
//  Created by Herb on 2019/4/17.
//  Copyright © 2019年 DingTalk. All rights reserved.
//

#ifndef Gaea_Lwp_Abstract_Connection_h
#define Gaea_Lwp_Abstract_Connection_h

#include <chrono>
#include <mutex>

#include "gaea/lwp/gaea_define.h"
#include "gaea/lwp/message.h"
#include "gaea/lwp/transfer_context.h"

GAEA_LWP_NAMESPACE_BEGIN

class AbstractConnection;
class AbstractConnectionDelegate {
public:
  AbstractConnectionDelegate() {}
  virtual ~AbstractConnectionDelegate() {}
  
public:
  virtual void DidRecvOnConnection(const std::string& conn_id,
                                   const std::string& buffer,
                                   TransferContextPtr transfer_context) = 0;
  virtual void DidSendOnConnection(const std::string& conn_id, int64_t len) = 0;
  virtual void ConnectedOnAbstractConnection(const std::string& conn_id) = 0;
  virtual void ConnectFailedOnAbstractConnection(const std::string& conn_id, const gaea::base::ErrorResult& result) = 0;
  virtual void DisconnectOnAbstractConnection(const std::string& conn_id) = 0;
};
typedef std::shared_ptr<AbstractConnectionDelegate> AbstractConnectionDelegatePtr;
  
enum AbstractConnStatus {
  kAcsUnset,
  kAcsConnecting,
  kAcsConnectFailed,
  kAcsConnected,
  kAcsBroken
};

class ConnectionProfile {
public:
  ConnectionProfile(bool is_connected) : error_code_(0) { is_connected_ = is_connected; }
  virtual ~ConnectionProfile() {}
  
  bool is_connected() const { return is_connected_; }
  void set_is_connected(bool is_connected) { is_connected_ = is_connected; }
  
  int error_code() const { return error_code_; }
  void set_error_code(int error_code) { error_code_ = error_code; }
  
  const std::string& error_type() const { return error_type_; }
  void set_error_type(const std::string& error_type) { error_type_ = error_type; }
  
  const std::string& raw_address() const { return raw_address_; }
  void set_raw_address(const std::string& raw_address) { raw_address_ = raw_address; }
  
private:
  bool is_connected_;
  std::string error_type_;
  int error_code_;
  std::string raw_address_;
};

class AbstractConnection {
public:
  AbstractConnection(EventLoopPtr session_eventloop, EventLoopPtr net_eventloop);
  virtual ~AbstractConnection();
  
public:
  void set_delegate(AbstractConnectionDelegate* delegate);
  
  const std::string conn_id() const { return conn_id_; }
  SiteType site_type() { return site_type_; }
  void set_site_type(SiteType site_type) { site_type_ = site_type; }
  
  virtual void Connect() = 0;
  virtual void Disconnect() = 0;
  virtual void Shutdown() = 0;
  /**
   * Check connection's network is available
   * For bifrost's connection, the physial netowrk if accessable;
   * For virsual connection,  accs network if accessable;
   */
  virtual bool CanConnect() { return true; }
//  virtual void NotifyConnect();
//  virtual void NotifyDisconnect();
//  virtual void NotifyDataTransferStable() {}
  
  virtual void SetServerList(const std::vector<std::string> &server_list) {};
  virtual const std::vector<std::string> ServerList() { return std::vector<std::string>(); };
  virtual void AddConnectionProfile(const ConnectionProfile& connection_profile) {}
  virtual const std::vector<ConnectionProfile> ConnectionProfiles() { return std::vector<ConnectionProfile>(); }
  /**
   * @brief 添加异步任务块
   */
  void AddTask(gaea::base::AsyncTaskPtr task);
  
  AbstractConnStatus status() const {return status_;}
  void set_status(AbstractConnStatus status) {status_ = status;}
  
  virtual int Send(const std::string& buffer, TransferContextPtr transfer_context) = 0;
  
//  int64_t idle_timeout() { return idle_timeout_; }
//  virtual void set_idle_timeout(int64_t timeout) {
//    idle_timeout_ = timeout;
//  };
//  
//  int64_t connect_timeout() const { return connect_timeout_; }
//  virtual void set_connect_timeout(int64_t timeout) {
//    connect_timeout_ = timeout;
//  };
  
  void set_unique_mark(const std::string& mark);
  
//  std::shared_ptr<Message> ParseMessage(std::string& buf, bool *except);
  static MessagePtr StaticParseMessage(const std::string& buf, bool *except);

  bool IsSessionThread();
  
  
private:
  std::string GenerateConnId();
  
protected:
  AbstractConnectionDelegate* delegate_;
  gaea::base::Logger logger_;
  EventLoopPtr session_eventloop_;
  EventLoopPtr net_eventloop_;
  
private:
  AbstractConnStatus status_;
  
  /** 唯一标识当前的实际物理连接的连接 ID */
  SiteType site_type_;
  std::string conn_id_;
  std::string unique_mark_;
};

typedef std::shared_ptr<AbstractConnection> AbstractConnectionPtr;

GAEA_LWP_NAMESPACE_END
#endif /* Gaea_Lwp_Abstract_Connection_h */
