//
//  base_transaction.hpp
//  gaea_lwp
//
//  Created by Herb on 2020/3/11.
//  Copyright © 2020 DingTalk. All rights reserved.
//

#ifndef GAEA_LWP_BASE_TRANSACTION_H
#define GAEA_LWP_BASE_TRANSACTION_H

#include "gaea/base/error_result.h"
#include "gaea/lwp/common/timer.h"
#include "gaea/lwp/gaea_define.h"

GAEA_LWP_NAMESPACE_BEGIN


typedef enum BaseTransactionStatus {
  kStatusInit,
  kStatusPending,
  kStatusPreProcess,
  kStatusProcessing,
  kStatusCompleted,
  kStatusEnd
} BaseTransactionStatus;


class BaseTransaction;
typedef std::shared_ptr<BaseTransaction> BaseTransactionPtr;
typedef std::weak_ptr<BaseTransaction> BaseTransactionWeakPtr;


class BaseTransaction : public std::enable_shared_from_this<BaseTransaction> {
 public:
  BaseTransaction();
  virtual ~BaseTransaction();
  
  // Check next status is valid
  virtual bool CheckIfNextStateIsValid(BaseTransactionStatus next_status);
  virtual bool CheckIfReadyProcess(gaea::base::ErrorResult* result) { return true; }
  
  virtual const std::string GetTransTypeDescript() = 0;
  
  const std::string transaction_id() { return transaction_id_; }
  void set_transaction_id(const std::string& transaction_id) { transaction_id_ = transaction_id; }
  
  const Timer::TimerId timer_id() const { return timer_id_; }
  void set_timer_id(const Timer::TimerId timer_id) { timer_id_ = timer_id; }
  
  static const std::string GenTransactionId();
  virtual void OnFinished(const gaea::base::ErrorResult& result);
  
  BaseTransactionStatus last_status() { return last_status_; }
  BaseTransactionStatus status() { return status_; }
  void set_status(BaseTransactionStatus status);
  static const std::string GetStatusDescript(BaseTransactionStatus status);
  
  const gaea::base::ErrorResult error_info() { return error_info_; }
  void set_error_info(const gaea::base::ErrorResult& result) { error_info_ = result; }
  
  int64_t HasLossTime();
  void UpdataLastRecvTime();
  
  void SetPendingBeginTime();
  void SetPreProcessBeginTime();
  void SetProcessingBeginTime();
  void SetCompletedBeginTime();
  void SetFinalEndTime();
   
  int64_t PendingCostTime();
  int64_t PreProcessCostTime();
  int64_t ProcessingCostTime();
  int64_t CompletedCostTime();
  int64_t TotalCostTime();
  
 protected:
  gaea::base::Logger logger_;
  std::string transaction_id_;
  BaseTransactionStatus status_;
  BaseTransactionStatus last_status_;
  Timer::TimerId timer_id_;
  bool enable_retry_;
  std::chrono::time_point<std::chrono::steady_clock> pending_begin_time_;
  std::chrono::time_point<std::chrono::steady_clock> preprocess_begin_time_;
  std::chrono::time_point<std::chrono::steady_clock> processing_begin_time_;
  std::chrono::time_point<std::chrono::steady_clock> completed_begin_time_;
  std::chrono::time_point<std::chrono::steady_clock> final_time_;
  std::chrono::time_point<std::chrono::steady_clock> last_recv_time_;
  gaea::base::ErrorResult error_info_;
};


GAEA_LWP_NAMESPACE_END

#endif /* GAEA_LWP_BASE_TRANSACTION_H */
