//
//  Timer.hpp
//  Atlas
//
//  Created by jinxi on 4/25/16.
//  Copyright © 2016 jinxi. All rights reserved.
//

#ifndef GAEA_LWP_COMMON_TIMER_H_
#define GAEA_LWP_COMMON_TIMER_H_

#include <chrono>
#include <map>
#include <mutex>
#include <memory>
#include <functional>

#include "gaea/lwp/gaea_define.h"

GAEA_LWP_NAMESPACE_BEGIN

class Timer {
public:
  typedef int64_t TimerId;
  typedef std::chrono::time_point<std::chrono::steady_clock> TimePoint;
  
  static TimerId kInvalidTimerId;
  
private:
  class TimerEvent {
  public:
    TimerEvent(TimerId tid, std::function<void()> cb) : timer_id_(tid), callback_(cb) {}
    virtual ~TimerEvent() {}
    
    TimerId timer_id() { return timer_id_; }
    void DoCallback() { callback_(); }
    
  private:
    TimerId timer_id_;
    std::function<void()> callback_;
  };
  typedef std::shared_ptr<TimerEvent> TimerEventPtr;

public:
  Timer();
  virtual ~Timer();

  /**
   * @function Add
   * @brief 添加定时器
   * @param interval 定时器时长
   * @param callback 定时器超时后触发的回调
   * @return 返回添加成功的定时器ID;
   */
  TimerId Add(std::chrono::milliseconds interval, std::function<void()> callback);
  
  /**
   * @function Delete
   * @param timer_id 删除指定 timer_id 的定时器
   * @brief 删除指定定时器
   * @return true 删除成功; false 删除失败;
   */
  bool Delete(TimerId timer_id);
  
  /**
   * @brief 清空定时器
   */
  void Clear();

  /**
   * @brief 遍历定时器，查找当前已经超时的事件对象
   * @return 返回接下来最近一个定时器超时时间间隔
   */
  std::chrono::milliseconds Poll();
  
  /**
   * @brief 当前挂载尚未触发的定时器的个数
   */
  size_t Size() { return timers_event_map_.size(); }
  
private:
  /**
   * @brief 返回当前可以使用的定时器 timer_id
   */
  TimerId nextId();

  /**
   * @brief 返回接下来最近一个定时器超时时间间隔
   */
  std::chrono::milliseconds LatestEventInterval() const;

private:
  gaea::base::Logger logger_;
  /**
   * 定时器ID 以及超时时间映射表: <定时器ID & 超时时间>
   */
  std::map<TimerId, TimePoint> timers_expire_time_map_;
  
  /**
   * 定时器映射表: <超时时间 & 超时事件>
   */
  std::multimap<TimePoint, TimerEventPtr> timers_event_map_;
  
  std::mutex mutex_;
  TimerId nextId_;
};

GAEA_LWP_NAMESPACE_END

#endif /* GAEA_LWP_COMMON_TIMER_H_ */
