//
//  user_agent.h
//  gaea
//
//  Created by guodi.ggd on 2019/1/7.
//  Copyright © 2019 DingTalk. All rights reserved.
//

#ifndef GAEA_LWP_USERAGENT_H_
#define GAEA_LWP_USERAGENT_H_

#include <map>
#include <mutex>
#include <memory>

#include "gaea/lwp/user_context.h"
#include "gaea/lwp/gaea_define.h"
#include "gaea/lwp/session_delegate.h"

#if defined(UTEST_PREPROCESS_MODULE)
  class UserAgentTest;
#endif

GAEA_LWP_NAMESPACE_BEGIN
  
class UserAgentDelegate {
 public:
  UserAgentDelegate() {}
  virtual ~UserAgentDelegate() {}
  
 public:
  /**
   * @brief UserAgent 进行注册时，需要携带上 token 进行注册
   */
  virtual std::string TokenRequiredOnUserAgent() { return ""; }
  virtual void AsyncTokenRequiredOnUserAgent(gaea::lwp::UserAgent* const user_agent, const gaea::base::ErrorResult& error) {};
  virtual std::map<std::string, std::string> GetRegHeaders() { std::map<std::string, std::string> headers; return headers; }
  virtual RpcRegModel GetRpcRegModelOnSessionForAutoSubscribe() {return RpcRegModel();}

  virtual void DidConnectionChanged(LwpConnectionStatus status) {}
  virtual void AuthedOnUserAgent() {}
  virtual void ConnectedOnUserAgent() {}
  virtual void ConnectFailedOnUserAgent(const gaea::base::ErrorResult& err_result) {}
  virtual void DisconnectOnUserAgent(const gaea::base::ErrorResult& err_result) {}
  virtual bool IsUserLogin() { return false; }
  virtual bool SetMainServerCookieOnSession(const std::string& cookie) { return true; }
};
typedef std::shared_ptr<UserAgentDelegate> UserAgentDelegatePtr;

class UserAgent : public SessionDelegate,
                  public std::enable_shared_from_this<UserAgent> {
#if defined(UTEST_PREPROCESS_MODULE)
  friend UserAgentTest;
#endif
 public:
  /**
   * @brief 通过静态方法构建对象
   */
  UserAgent(const std::string& ua_identifier = "");
  virtual ~UserAgent();
  static UserAgent* CreateUserAgent();
                    
  /**
   * @brief 释放 UserAgent 动态创建的资源空间
   */
//  static void Release(UserAgent** user_agent);
  static void Release(UserAgentPtr& user_agent);
  
  /********** RPC Section ***********/
  virtual void SendRequest(RequestPtr request, RequestContextPtr context);

  virtual void SendResponse(RequestPtr request, ResponsePtr response);
  
  /********** File Section ***********/
  virtual void SendRequest(
      FileUploadTaskPtr task,
      std::function<void()> onStart,
      std::function<void(const FileProgress&)> onProgress,
      std::function<void(const FileUploadResult&)> onSuccess,
      std::function<void(const gaea::base::ErrorResult&)> onFailure);
 
  virtual void SendRequest(
      FileDownloadTaskPtr task,
      std::function<void()> onStart,
      std::function<void(const FileProgress&)> onProgress,
      std::function<void(const FileDownloadResult&)> onSuccess,
      std::function<void(const gaea::base::ErrorResult&)> onFailure);

  virtual void CancelFileTask(const std::string& task_id);

  /********** Push Section ***********/
  /**
   * @function Subscribe
   * @brief 定义h指定 topic 的推送
   * @return true 完成插入; false 插入失败
   */
  virtual bool Subscribe(const std::string& topic, PushListenerBasePtr listener);

  /********** Setting Section ***********/
  /**
   * @function SetSetting 将外部配置信息，配置到 UserAgent 内部
   * @param setting 配置容器对象
   */
//  virtual void SetSetting(const gaea::lwp::Setting& setting);
  
  /**
   * @function setting 获取 UserAgent 内部配置信息
   * @return setting 配置容器对象
   */
//  virtual gaea::lwp::Setting& setting() { return setting_; }
  
  /**
   * @function start
   * @param auto_run default is true
   * @brief start UserAgent with two kinds of behavier. if auto_run is true, means start engine and begin auto connect. if set false, only init engine, but engine will not connect until you call this api again with auto_run.
   */
  virtual void Start(bool auto_run = true);
  virtual void StartAutoRun();
  /**
   * @function TriggerAuth
   * @brief 完成登陆请求拿到 access_token 后通过该接口触发 /reg 鉴权
   */
  virtual void TriggerAuth();

  /**
   * @function DisconnectOnce
   * @brief 断开所有连接一下，稍后会根据相关策略和设置来决定是否重连
   */
  virtual void DisconnectOnce();
                    
  virtual void SetEnableAutoAuth(bool enable_auto_auth);
  virtual bool IsEnableAutoAuth();
  
  /**
   * @function start
   * @brief 登出，完成token清理，退出auth
   */
  virtual void Logout(std::function<void()> callback);

  virtual void set_delegate(std::shared_ptr<UserAgentDelegate> delegate);
  virtual std::weak_ptr<UserAgentDelegate> delegate() { return delegate_; }

  virtual void set_trace_service(service::TraceInterfacePtr service_implement);
  virtual void set_setting_service(service::SettingInterfacePtr service_implement);
  
  gaea::service::AsrInterfacePtr translate_service() { return translate_service_; }
  void set_extern_user_agent(gaea::service::UserAgentInterface* ua_delegate) { extern_user_agent_ = ua_delegate; }
  /**
   * @function unique_mark
   * @brief 目的是为了获取ua的标记
   */
  virtual const std::string unique_mark();
  /**
   * @brief UserAgent 中存放的 CacheHeader 是当前账号私有属性
   * app级别公共的 CacheHeader 统一放到 Setting 中设置;
   */
  virtual void SetCacheHeader(const std::string& key, const std::string& value) {
    if (is_start_) {
      GAEA_LOG_ERROR(logger_, "SetCacheHeader should only be call before start ua");
      return ;
    }
    private_cache_header_[key] = value;
  }
  
 /**
  * @function AddRoutineFilter
  * @brief 向filter chain中添加filter
  * @Attention 必须在ua启动前添加filter
  */
  bool AddRoutineFilter(MessageFilterPtr filter);
  
  void set_get_token_handler(OnGetTokenHandler handler);
  void set_async_require_token_handler(OnAsyncGetTokenHandler handler);

  virtual const std::string GetCacheHeaderValueByName(const std::string& key) const;
  virtual const gaea::base::StringMap& GetCacheHeader() override { return private_cache_header_; }

  virtual void CheckAuth(CheckAuthCallback callback);
  virtual void CheckAuthWithExternInfo(SubscribeExternInfoPtr extern_info);

  virtual void EnterBackground();
  virtual void EnterForeground();
  
  virtual bool IsEnableAutoConnect();
  virtual void SetEnableAutoConnect(bool enable);
  
  virtual void SetCommitRelatedEventHandle(std::function<void(const std::string& dye_mid, const std::string& trace_id)> handler);

  virtual bool IsRunInCallbackThread();

  virtual const std::string did();
  virtual void set_did(const std::string &did);
  
 private:
  virtual void InnerCheckAuth(
      CheckAuthCallback callback,
      SubscribeExternInfoPtr extern_info);
  
  void RegisterNetworkListener();
  void RemoveNetworkListener();
                    
  void AddTask(EventLoopPtr event_loop, gaea::base::AsyncTaskPtr task);
  
  void InnerRelease();
  
//  void CheckInvalidSiteId(const SiteType site_id);
  
  /* delegate */
  virtual RpcRegModel GetRpcRegModelOnSessionForAutoSubscribe() override {
    auto strong_delegate = delegate_.lock();
    if (!strong_delegate) {
      return RpcRegModel();
    }
    return strong_delegate->GetRpcRegModelOnSessionForAutoSubscribe();
  }
  std::string TokenRequiredOnSession() override;
  virtual std::map<std::string, std::string> GetRegHeaders() override;

  virtual void AsyncTokenRequiredOnSession(const gaea::base::ErrorResult& error) override;
  // 底层完成建立连接后通知上层
  virtual void AuthedOnSession() override;
  virtual void ConnectedOnSession() override;
  virtual void ConnectFailedOnSession(const gaea::base::ErrorResult& err_result) override;
  virtual void DisconnectOnSession(const gaea::base::ErrorResult& err_result) override;
  virtual void DidConnectionChanged(LwpConnectionStatus status) override;
  virtual void OnIncomingRequest(RequestPtr request) override;
  virtual bool SetMainServerCookieOnSession(const std::string& cookie) override;
  virtual bool IsUserLogin() override;
                    
  void InnerSubscribe();
  void DidActiveWakeup();
  
private:
  UserContextPtr user_context_;

  gaea::base::Logger logger_;
  Session* session_;
  FileServiceInterfacePtr file_service_;
  gaea::service::AsrInterfacePtr translate_service_;
  EventLoopPtr session_eventloop_;
  EventLoopPtr file_eventloop_;
  EventLoopPtr net_eventloop_;
  EventLoopPtr callback_eventloop_;
  EventLoopPtr global_eventloop_;
  std::weak_ptr<UserAgentDelegate> delegate_;
  bool is_start_;
  /** 提供外部自定义配置 键-值 表对 */
  std::map<std::string, std::string> private_cache_header_;
  std::map<std::string, PushListenerBasePtr> push_listeners_;
  std::map<std::string, PushListenerBasePtr> inner_listeners_;
  std::mutex mutex_;
                    
  OnGetTokenHandler get_token_handler_;
  OnAsyncGetTokenHandler async_require_token_handler_;
  gaea::service::UserAgentInterface* extern_user_agent_;
};
typedef std::shared_ptr<UserAgent> UserAgentPtr;
typedef std::weak_ptr<UserAgent> UserAgentWeakPtr;

GAEA_LWP_NAMESPACE_END

#endif /* GAEA_LWP_USERAGENT_H_ */
