//
//  aladdin_service.h
//  gaea_lwp
//
//  Created by ninan on 2020/9/17.
//  Copyright © 2020 DingTalk. All rights reserved.
//

#ifndef GAEA_LWP_ALADDIN_SERVICE_H_
#define GAEA_LWP_ALADDIN_SERVICE_H_

#include <atomic>
#include <memory>
#include <mutex>
#include <string>
#include <vector>

#include "gaea/base/error_result.h"
#include "gaea/base/logging.h"
#include "gaea/lwp/gaea_define.h"


namespace gaea {
namespace service {
  class HttpInterface;
}
}


GAEA_LWP_NAMESPACE_BEGIN

class AladdinCache;
class NetworkPolicy;
class AladdinSuiteModel;


class ConnectResultContext {
 public:
  ConnectResultContext() : success_(false) {}
  virtual ~ConnectResultContext() {}
  
  bool success() const { return success_; }
  void set_success(bool success) { success_ = success; }

  SiteType site_type() const { return site_type_; }
  void set_site_type(SiteType site_type) { site_type_ = site_type; }
  
  const std::string server_url() const { return server_url_; }
  void set_server_url(const std::string& server_url) { server_url_ = server_url; }
  
 private:
  bool success_;
  std::string server_url_;
  SiteType site_type_;
};


class AladdinSetting {
public:
  AladdinSetting() : env_param_("online") {}
  virtual ~AladdinSetting() {};
  const std::string& aladdin_base_url() const {return aladdin_base_url_;}
  void set_aladdin_base_url(const std::string& base_url) { aladdin_base_url_ = base_url; }
  const std::string& env_param() const {return env_param_;}
  void set_env_param(const std::string& env_param) { env_param_ = env_param; }
  
  void set_default_long_addresses(const std::vector<std::string> &default_long_addresses) { default_long_addresses_ = default_long_addresses; }
  const std::vector<std::string>& default_long_addresses() { return default_long_addresses_; }
  void set_default_short_addresses(const std::vector<std::string> &default_short_addresses) { default_short_addresses_ = default_short_addresses; }
  const std::vector<std::string>& default_short_addresses() { return default_short_addresses_; }
  
  void set_debug_long_addresses(const std::vector<std::string> &debug_long_addresses) { debug_long_addresses_ = debug_long_addresses; }
  const std::vector<std::string>& debug_long_addresses() { return debug_long_addresses_; }
  void set_debug_short_addresses(const std::vector<std::string> &debug_short_addresses) { debug_short_addresses_ = debug_short_addresses; }
  const std::vector<std::string>& debug_short_addresses() { return debug_short_addresses_; }
  
private:
  std::vector<std::string> debug_long_addresses_;
  std::vector<std::string> debug_short_addresses_;
  
  std::vector<std::string> default_long_addresses_;
  std::vector<std::string> default_short_addresses_;
  std::string aladdin_base_url_;
  std::string env_param_;
};
typedef std::shared_ptr<AladdinSetting> AladdinSettingPtr;
typedef std::weak_ptr<AladdinSetting> AladdinSettingWeakPtr;


/* TODO: 改为interface,外部实现,清空default的值，写demo,参数检查 */
class AladdinServiceDelegate {
public:
  AladdinServiceDelegate() {}
  virtual ~AladdinServiceDelegate() {}
  
public:
  virtual const std::string GetOSOnAladdinService() = 0;
  virtual const std::string GetVHostOnAladdinService() = 0;
  
  virtual const std::string GetSDKVerOnAladdinService() = 0;
  virtual const std::string GetUIDOnAladdinService() { return ""; }
  virtual const std::string GetOrgIDOnAladdinService() { return ""; }
  virtual const std::string GetAppKeyOnAladdinService() = 0;
  virtual const std::string GetAppVerOnAladdinService() = 0;
  virtual const std::string GetEnvOnAladdinService() { return ""; }
  virtual const std::string GetNetOnAladdinService() { return ""; }
};
typedef std::shared_ptr<AladdinServiceDelegate> AladdinServiceDelegatePtr;
typedef std::weak_ptr<AladdinServiceDelegate> AladdinServiceDelegateWeakPtr;


class AladdinService : public std::enable_shared_from_this<AladdinService> {
 public:
  AladdinService();
  virtual ~AladdinService();
  
  void Init();

  void set_delegate(AladdinServiceDelegate* delegate);
  
  void ClearCache();
  
  /**
   * @brief Fetch aladdin server list by http request
   */
  void AsyncFetchServerLists(
      std::function<void()> on_success = nullptr,
      std::function<void(const gaea::base::ErrorResult& error)> on_failure = nullptr);
  
  /**
   * @brief Important: Get server list for connection
   */
  bool GetServerList(
      SiteType site_id,
      std::vector<std::string>* server_list,
      bool force_change = false);
  
  void UpdateConnectStatus(const std::vector<ConnectResultContext>& connect_results);
  
  /* TODO:读取不更新接口 */
//  bool GetSniSuit(const std::string& sni, gaea::lwp::AladdinSuiteModel* suit_model);
      
  AladdinSettingPtr mutable_aladdin_setting() { return aladdin_setting_; }

  bool has_initialized() { return has_initialized_.load(); }
  
 private:
  void ClearNetworkPolicy();
  
  /**
   * @brief Get server list from aladdin
   */
  bool InnerGetServerList(
      SiteType site_id,
      SecurityProtocol security_protocol,
      ServerGroup* server_group,
      std::vector<std::string>* server_list);
  
  /**
   * @brief Get server list from setting
   */
  bool GetDefaultServerList(SiteType site_id,
                            std::vector<std::string>* server_list);
                            
  void FetchServerLists(std::function<void()> on_success,
                        std::function<void(const gaea::base::ErrorResult& error)> on_failure);
  bool InitHttpService();
  bool AladdinRequestIsPerforming();
  void SetAladdinRequestIsPerforming(bool aladdin_request_is_performing);
  bool ProcessResponse(const std::string& response, gaea::base::ErrorResult* error_result);
  
  bool EnvIsOnLine() const;
  const std::string GetUrl() const;
  
 private:
  gaea::base::Logger logger_;
  EventLoopPtr aladdin_eventloop_;
  
  std::mutex mutex_;
  AladdinServiceDelegate* delegate_;
  
  std::atomic<bool> has_initialized_;
  std::atomic<bool> aladdin_request_is_performing_{false};
  std::atomic<bool> aladdin_model_is_updated_{false};
  
  int64_t last_query_start_timestamp_;

  std::shared_ptr<gaea::service::HttpInterface> http_client_;
  std::shared_ptr<AladdinCache> aladdin_cache_;
  std::shared_ptr<NetworkPolicy> network_policy_;
  
  AladdinSettingPtr aladdin_setting_;
};
typedef std::shared_ptr<AladdinService> AladdinServicePtr;
typedef std::weak_ptr<AladdinService> AladdinServiceWeakPtr;


GAEA_LWP_NAMESPACE_END

#endif /* GAEA_LWP_ALADDIN_SERVICE_H_ */
