//-----------------------------------------------------------------------------
//  Galv's Simple Crops
//-----------------------------------------------------------------------------
//  For: RPGMAKER MV
//  Galv_SimpleCrops.js
//-----------------------------------------------------------------------------
//  2017-10-25 - Version 1.4 - tweaked for compatibility
//  2017-10-12 - Version 1.3 - added watering info to documentation and also a
//                             'water all' script call.
//  2017-05-06 - Version 1.2 - added plugin setting for fixing the lower/front
//                             priority when activating CROP events only.
//  2016-12-14 - Version 1.1 - added priority types to growth stages of crops
//  2016-11-20 - Version 1.0 - release
//-----------------------------------------------------------------------------
// Terms can be found at:
// galvs-scripts.com
//-----------------------------------------------------------------------------

var Imported = Imported || {};
Imported.Galv_SimpleCrops = true;

var Galv = Galv || {};                  // Galv's main object
Galv.CROPS = Galv.CROPS || {};          // Galv's stuff

//-----------------------------------------------------------------------------
/*:
 * @plugindesc (v.1.4) A simple farming/growing crop system.
 * 
 * @author Galv - galvs-scripts.com
 *
 * @param Crop Charset Name
 * @desc 所有作物使用的行走图都叫这个，后跟一个imgId号码
 * @default !Crops
 *
 * @param Watered Time
 * @desc 土地湿润持续的秒数/变量天数，此后才能继续浇水
 * @default 60
 *
 * @param Watered Benefit
 * @desc 灌溉时作物生长时间减少的秒数/变量天数。
 * @default 5
 *
 * @param Crop Priority
 * @desc 成长阶段的优先度。0下方，1和玩家相同
 * 空,种子,发芽,半熟,成熟
 * @default 0,0,1,1,1
 *
 * @param Crop Under or Front Fix
 * @desc true or false - 进行互动时，应优先考虑前方的农作物，而不是下方的农作物。
 * @default true
 *
 * @help
 *   Galv's Simple Crops
 * ----------------------------------------------------------------------------
 * 插件允许玩家选择种子，在地图上种植作物。
 * 
 * 插件名为“简易”，不是设置简易，是游戏里玩家操作简易。
 * 
 *
 * 创建土地
 * --------------
 * 在事件备注中添加<crop>标签，可以令其作为土地，允许
 * 玩家在其上种植作物和进行灌溉。
 *
 * 创建种子
 * --------------
 * 为道具添加<seed>注释作为种子，标签包含作物成长的设置。
 * （详情见下方）
 * 
 *
 * 植物图像
 * -----------------
 * /img/characters/里的植物行走图使用的名称需要与插件设置'Crop Charset Name'
 * 里的一致，以及对应的imgId号。
 * 如果插件设置为： !Crops
 * 你的作物行走图就应该命名为：
 * !Crops0.png, !Crops1.png, !Crops2.png, 等等。(!Crops0.png左上角的角色
 * 将作为空地或刚种下种子的土地)
 *
 * 植物行走图必须按照规格进行设置。一个角色的行走图集合包含2个植物。
 * 每个植物有3个成长阶段（共6个）。它们与脸图的朝向呈以下关系。
 * 
 * for example:
 * Down  = 第一个植物（缺水）
 * Left  = 第一个植物（湿润）
 * Right = 第二个植物（缺水）
 * Up    = 第二个植物（湿润）
 *
 * 但，imgId 0里的第一个角色的第一个植物的图像将用在空地或刚种植植物的土地。
 * 不懂的可以看范例。
 * 
 *
 * 生长
 * -------
 * 植物的生长过程经过4个阶段。
 * 
 * 1. 种子一旦放置，就会显示种植的行走图。
 * 2. 经过20%的成长周期后，植物进入第一阶段
 * 3. 经过60%的成长周期后，植物进入第二阶段
 * 2. 经过100%或以上的成长周期后，植物进入第三阶段
 *    (此时可以进行收获)
 *
 * 成长可以基于游戏时间秒数或变量进行。这可以在您要种植种子的事件设置中指定
 * (详见下文)。
 *
 * 你还可以设置植物每个阶段的优先级情况，如有需要，也可以用标签设置每个作物
 * 种子的优先级。
 * 
 * 
 *
 * 如果作物根据时间增长，则要小心其通行度（优先级），这可能会导致玩家卡住。
 * 
 *
 * 作物
 * ---------
 * 作物是你种子设置里指定的道具，收获时会获得，并且土地会变为空地。
 * 
 * ----------------------------------------------------------------------------
 *
 * ----------------------------------------------------------------------------
 *  物品注释标签
 * ----------------------------------------------------------------------------
 * 
 *    <seed:imgId,charId,pId,growTime,itemId>
 *
 * EXPLANATION:
 * seed     = 必填，不能改
 * imgId    = 指定要使用的作物imgId号码
 * charId   = 行走图里哪个角色（1~8）
 * pId      = 玩家种植时的作物ID（1~2）
 * growTime = 作物从种下到能收获所需的秒数
 * itemId   = 收获时获得的道具ID，不能收获则设为0
 *            
 *
 *    <seedPrio:seed,sprouting,halfGrown,fullGrown>
 *
 * EXPLANATION:
 * 该注释将覆盖插件“优先级”的设置。
 * 这可以单独控制作物每阶段的优先级。0=低于玩家，1=和玩家一样
 * 
 * eg.
 * <seedPrio:0,1,1,1>
 *
 * ----------------------------------------------------------------------------
 *  脚本信息
 * ----------------------------------------------------------------------------
 * eId = 事件ID
 * iId = 道具ID
 * vId = 变量ID
 * 你可以使用this._event代替非并行处理事件中的事件ID，作为当前事件ID
 * 
 *
 * ----------------------------------------------------------------------------
 *  条件分歧脚本
 * ----------------------------------------------------------------------------
 *
 *    Galv.CROPS.isCrop(eId)     // C检查事件是否具有<crop>注释标签
 * 
 *    Galv.CROPS.hasSeed(eId)    // 检查事件是否已播种
 *
 *    Galv.CROPS.isReady(eId)    // 检查事件种子是否准备好收获
 *
 *    Galv.CROPS.isSeed(iId)     // 检查道具是否具有<seed>标签并且是种子
 *
 * ----------------------------------------------------------------------------
 *  脚本调用
 * ----------------------------------------------------------------------------
 *
 *    Galv.CROPS.harvest(eId,v);  // 收获事件作物。
 *                                // v指变量ID，会把收获的道具名称代入其中，
 *                                // 作为一个字符串，可以用\v[x]在文本里显
 *                                // 示出来。
 *
 *    Galv.CROPS.remove(eId);     // 将植物从事件中移除。
 *
 *    Galv.CROPS.plant(eId,iId,vId);  // 在事件中播下种子。
 *                                    // 将vId留空以使作物根据游戏时间的秒数增长。
 *                                    // 如果您使用vId，它将根据所选的变量ID增长。
 *                                    // 
 *                                    // 
 *
 * 给农作物浇水可以将生长时间减少一定的天数或时间（具体取决于您的插件设置）。
 * 湿润持续时间为你预设的时长，时间结束后才能继续灌溉。
 * 
 * 
 *
 *    Galv.CROPS.water(eId,vId);   // 灌溉作物。
 *                                 // vId留空以使用游戏时间，否则将使用变量
 *                                 // 代指的天数。
 *                                 // 
 *                                 // 
 *
 *    Galv.CROPS.waterAll(vId);  // 与上面相同，但适用于当前地图上的所有事件。
 *                               // 
 *
 * ----------------------------------------------------------------------------
 *  移动路线脚本调用
 * ----------------------------------------------------------------------------
 *
 *    this.updateCrop();   // 使用移动路线频率来控制玩家附近的作物生长。
 *                         // 这样一来，就不必同时运行多个计时器
 *                         // 
 *
 * ----------------------------------------------------------------------------
 *  SCRIPT call
 * ----------------------------------------------------------------------------
 *
 *    Galv.CROPS.update();   // 更新地图上的所有作物
 *
 * ----------------------------------------------------------------------------  
 */



//-----------------------------------------------------------------------------
//  CODE STUFFS
//-----------------------------------------------------------------------------

(function() {

Galv.CROPS.img = PluginManager.parameters('Galv_SimpleCrops')["Crop Charset Name"];
Galv.CROPS.wateredTime = Number(PluginManager.parameters('Galv_SimpleCrops')["Watered Time"]);
Galv.CROPS.wateredBenefit = Number(PluginManager.parameters('Galv_SimpleCrops')["Watered Benefit"]);

Galv.CROPS.useFix = PluginManager.parameters('Galv_SimpleCrops')["Crop Under or Front Fix"] === 'true' ? true : false;


var txt = PluginManager.parameters('Galv_SimpleCrops')["Crop Priority"].split(',');
Galv.CROPS.prio = [1,1,1,1,1];
for (var i = 0; i < txt.length; i++) {
	Galv.CROPS.prio[i] = Number(txt[i]);
}

Galv.CROPS.isCrop = function(eventId) {
	return $gameMap.event(eventId) && $gameMap.event(eventId).crop();
};

Galv.CROPS.hasSeed = function(eventId) {
	return Galv.CROPS.isCrop(eventId) && $gameMap.event(eventId).crop()._seed;
};

Galv.CROPS.isReady = function(eventId) {
	return Galv.CROPS.hasSeed(eventId) ? $gameMap.event(eventId).crop().isReady() : false;
};

Galv.CROPS.harvest = function(eventId,varId) {
	$gameVariables._data[varId] = 0;
	if (Galv.CROPS.isReady(eventId)) {
		var itemId = $gameMap.event(eventId).crop()._seed.itemId;
		var item = $dataItems[itemId];
		if (item) {
			$gameParty.gainItem(item,1);
			var name = item.name;
		} else {
			var name = '';
		}
		
		if (varId > 0 && name != '') {
			$gameMap.event(eventId).crop().resetPlot(true);
			$gameMap.event(eventId).updateCropGraphic();
			$gameVariables._data[varId] = name;
		}
	};
};

Galv.CROPS.remove = function(eventId,keepWatered) {
	if (Galv.CROPS.isCrop(eventId)) {
		$gameMap.event(eventId).crop().resetPlot(!keepWatered); // remove crop but keep watered
		$gameMap.event(eventId).updateCropGraphic();
	}
};

Galv.CROPS.water = function(eventId,daysVar) {
	if (Galv.CROPS.isCrop(eventId)) {
		$gameMap.event(eventId).crop().waterPlot(daysVar);
		$gameMap.event(eventId).updateCropGraphic();
	}
};

Galv.CROPS.waterAll = function(daysVar) {
	var events = $gameMap.events();
	for (var i = 0; i < events.length; i++) {
		if (events[i]) Galv.CROPS.water(events[i]._eventId,daysVar);
	}
};

Galv.CROPS.plant = function(eventId,itemId,daysVar) {
	$gameParty.loseItem($dataItems[itemId],1);
	$gameMap.event(eventId).plantCrop(itemId,daysVar);
};

Galv.CROPS.isSeed = function(itemId) {
	return $dataItems[itemId] && $dataItems[itemId].meta.seed;
};

Galv.CROPS.update = function(mapId) {
	var events = $gameMap.events();
	for (var e in events) {
		if (events[e].event().meta.crop) events[e].updateCrop();
	}
};


//-----------------------------------------------------------------------------
//  GAME SYSTEM
//-----------------------------------------------------------------------------

Galv.CROPS.Game_System_initialize = Game_System.prototype.initialize;
Game_System.prototype.initialize = function() {
	Galv.CROPS.Game_System_initialize.call(this);
	this._crops = {}; //mapId: {eventId: Game_Crop},
};

})();


//-----------------------------------------------------------------------------
//  GAME CROP
//-----------------------------------------------------------------------------

// Game crop timer needs to check what % is left.
// if 100-80 left, show seeds
// if 80-40 left, show pattern 0
// if 40-1 left, show pattern 1
// if 0 left, show pattern 2 (ready to harvest)


function Game_Crop() {
    this.initialize.apply(this, arguments);
}

Game_Crop.prototype.initialize = function(eventId,type) {
	this._type = Number(type) || 0;
	this._eventId = eventId;
	this._daysVar = 0; // dont reset this. Keep last var used. // if want to use a variable to controls days instead of seconds.
	this.resetPlot();
};

Game_Crop.prototype.event = function() {
	return $gameMap.events(this._eventId);
};

Game_Crop.prototype.hasPlant = function() {
	return this._plantTime > 0;
};

Game_Crop.prototype.resetPlot = function(keepWatered) {
	this._plantTime = 0;   // game time when planted
	this._growTime = 0;    // target game time when fully grown
	
	if (!keepWatered) {
		this._wateredTime = 0; // game time when watered
		this._dryTime = 0;     // target game time when watered status ends
	}

	this._seed = null;
};

Game_Crop.prototype.getSeedSettings = function(seedId) {
	// get string from meta <seed> tag
	if ($dataItems[seedId] && $dataItems[seedId].meta.seed) {
		return $dataItems[seedId].meta.seed;
	} else {
		this.resetPlot();
		return null;	
	}
};

Game_Crop.prototype.growRate = function() {
	if (this._seed) {
		var full = this._seed.growTime;
		if (this._daysVar) { // use variable and days
			//var full = this._seed.growTime;
			var isAt = this._growTime - $gameVariables.value(this._daysVar);
			
		} else { // used seconds
			//var full = this._seed.growTime;
			var isAt = this._growTime - $gameSystem.playtime();
		}
		return Math.max(isAt / full,0);
	} else {
		return 0;
	}
};

Game_Crop.prototype.isWatered = function() {
	var playtime = this._daysVar ? $gameVariables.value(this._daysVar) : $gameSystem.playtime();
	return playtime < this._dryTime;
};

Game_Crop.prototype.isReady = function() {
	return this.growRate() <= 0;
};

Game_Crop.prototype.plantCrop = function(seedId,daysVar) {
	this._daysVar = daysVar || 0;
	var s = this.getSeedSettings(seedId);
	if (s) {
		var s = s.split(',');
		
		// create seed settings <seed:imgId,charId,pId,growTime,itemId>
		this._seed = {
			seedId: seedId,
			imgId: Number(s[0]),
			charId: Number(s[1]),
			pId: Number(s[2]),
			growTime: Number(s[3]),
			itemId: Number(s[4]),
			prio: Galv.CROPS.prio.slice(1)
		}
		
		// create prio settings <seedPrio:empty,seed,sprouting,halfGrown,fullGrown>
		var s2 = $dataItems[seedId].meta.seedPrio;
		if (s2) {
			s2 = s2.split(',');
			for (var i = 0; i < s2.length; i++) {
				this._seed.prio[i] = Number(s2[i]);
			}
		}
		
		this._plantTime = this._daysVar ? $gameVariables.value(this._daysVar) : $gameSystem.playtime();
		this._growTime = this._plantTime + this._seed.growTime;
	}
};

Game_Crop.prototype.waterPlot = function(daysVar) {
	if (this.isWatered()) return; // don't allow multiple waterings

	if (daysVar) this._daysVar = daysVar;
	this._wateredTime = this._daysVar ? $gameVariables.value(this._daysVar) : $gameSystem.playtime();
	if (this._seed) this._growTime -= Galv.CROPS.wateredBenefit;
	this._dryTime = this._wateredTime + Galv.CROPS.wateredTime;
};

//-----------------------------------------------------------------------------
//  SCENE LOAD
//-----------------------------------------------------------------------------

Galv.CROPS.Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
Scene_Load.prototype.onLoadSuccess = function() {
	Galv.CROPS.reSetupCropEvents = true; // set crops to reload graphics on load
	Galv.CROPS.Scene_Load_onLoadSuccess.call(this);
};


//-----------------------------------------------------------------------------
//  SCENE MAP
//-----------------------------------------------------------------------------

Galv.CROPS.Scene_Map_start = Scene_Map.prototype.start;
Scene_Map.prototype.start = function() {
    Galv.CROPS.Scene_Map_start.call(this);
	this.setupCropEvents();
};

Scene_Map.prototype.setupCropEvents = function() {
	if (Galv.CROPS.reSetupCropEvents) {
		var events = $gameMap.events();
		for (var e in events) {
			if (events[e].setupCropEvent) events[e].setupCropEvent($gameMap._mapId,e._eventId);
		}
	}
	Galv.CROPS.reSetupCropEvents = false;
};


//-----------------------------------------------------------------------------
//  GAME EVENT
//-----------------------------------------------------------------------------

Galv.CROPS.Game_Event_initialize = Game_Event.prototype.initialize;
Game_Event.prototype.initialize = function(mapId, eventId) {
	Galv.CROPS.Game_Event_initialize.call(this,mapId,eventId);
	this.setupCropEvent(mapId,eventId);
};


Game_Event.prototype.setupCropEvent = function(mapId,eventId) {
	var e = this.event().note.match(/<crop(.*)>/i);
	if (e) {
		var type = e[1];
		this._isCrop = true;
		this._cropType = type || 0;
		// create crop data if not created already
		$gameSystem._crops[mapId][eventId] = $gameSystem._crops[mapId][eventId] || new Game_Crop(eventId,type);
		this.updatePattern = function() {}; // remove pattern update from crop event
		this.updateCropGraphic();
	}
	
};

Game_Event.prototype.crop = function() {
	return $gameSystem._crops[this._mapId][this._eventId];
};

Game_Event.prototype.updateCrop = function() {
	this.updateCropGraphic();
	this.resetStopCount();
};

Game_Event.prototype.getCropDir = function(crop) {
	// change crop event direction
	if (!crop._seed) return crop.isWatered() ? 4 : 2;
	if (crop._seed.pId == 1) {
		return crop.isWatered() ? 4 : 2;
	} else {
		return crop.isWatered() ? 8 : 6;
	}
};

Game_Event.prototype.updateCropGraphic = function() {
	var crop = this.crop();
	if (!crop) return;

	// Update Graphic
	var rate = crop.growRate();
	
	if (crop._seed) {
		var charset = crop._seed.imgId;
		var ind = crop._seed.charId - 1;
		var dir = this.getCropDir(crop);

		if (rate <= 0) {
			// pattern 2 (full grown)
			var pat = 2;
			this._priorityType = crop._seed.prio[3];
		} else if (rate <= 0.4) {
			// pattern 1 (half grown)
			var pat = 1;
			this._priorityType = crop._seed.prio[2];
		} else if (rate <= 0.8) {
			// pattern 0 (sprouting)
			var pat = 0;
			this._priorityType = crop._seed.prio[1];
		} else {
			// charset and pattern for seed
			this._priorityType = crop._seed.prio[0];
			var charset = 0;
			var pat = 2;
			var ind = 0;
			var dir = crop.isWatered() ? 4 : 2;
		}
	} else {
		// No seed planted
		this._priorityType = Galv.CROPS.prio[0];
		var charset = 0;
		var pat = 1;
		var ind = 0;
		var dir = crop.isWatered() ? 4 : 2;
	}

	this._direction = dir;
	this._pattern = pat;
	this._characterName = Galv.CROPS.img + charset;
	this._characterIndex = ind;
	
};

Game_Event.prototype.plantCrop = function(id,daysVar) {
	this.crop().plantCrop(id,daysVar);
	this.updateCropGraphic();
};


//-----------------------------------------------------------------------------
//  GAME MAP
//-----------------------------------------------------------------------------

Galv.CROPS.Game_Map_setup = Game_Map.prototype.setup;
Game_Map.prototype.setup = function(mapId) {
	$gameSystem._crops[mapId] = $gameSystem._crops[mapId] || {};
	Galv.CROPS.Game_Map_setup.call(this,mapId);
};



if (Galv.CROPS.useFix) {

Galv.CROPS.Game_Player_checkEventTriggerHere = Game_Player.prototype.checkEventTriggerHere;
Game_Player.prototype.checkEventTriggerHere = function(triggers) {
	if (Input.isTriggered('ok')) this.checkCropEventThere(triggers);
	if ($gameMap.setupStartingEvent()) return true;
	Galv.CROPS.Game_Player_checkEventTriggerHere.call(this,triggers);
};

Game_Player.prototype.checkCropEventThere = function(triggers) {
	if (this.canStartLocalEvents()) {
		var direction = this.direction();
		var x = $gameMap.roundXWithDirection(this.x, direction);
		var y = $gameMap.roundYWithDirection(this.y, direction);
	
		if (!$gameMap.isEventRunning()) {
			$gameMap.eventsXy(x, y).forEach(function(event) {
				if (event.crop()) {
					event.start();
				}
			});
		}
	}
};

}; // end Galv.CROPS.useFix
