//=============================================================================
// QPopup
//=============================================================================

var Imported = Imported || {};

if (!Imported.QPlus || !QPlus.versionCheck(Imported.QPlus, '1.2.2')) {
  alert('Error: QPopup requires QPlus 1.2.2 or newer to work.');
  throw new Error('Error: QPopup requires QPlus 1.2.2 or newer to work.');
}

Imported.QPopup = '1.1.3';

//=============================================================================
/*:
 * @plugindesc v1.13 QABS-弹出窗口
 * Lets you create popups in the map or on screen
 * @version 1.1.3
 * @author Quxios  | Version 1.1.3
 * @site https://quxios.github.io/
 * @updateurl https://quxios.github.io/data/pluginsMin.json
 *
 * @requires QPlus
 *
 * @param Presets
 * @desc 弹出预设列表
 * @type Struct<PopupPreset>[]
 * @default []
 *
 * @help
 * ============================================================================
 * ## 关于
 * ============================================================================
 * 此插件可让您在事件中以固定的时间间隔播放随机弹出窗口。
 * 还可以创建自己的弹出窗口，并将其放置在所需的位置。
 * ============================================================================
 * ## 弹出预设列表
 * ============================================================================
 * 可以在插件参数中或使用插件命令创建预设。
 * 预设具有用于弹出样式和过渡的预定义规则。
 * ============================================================================
 * ## 事件弹出窗口
 * ============================================================================
 * 此功能将在每X帧的事件中创建一个弹出窗口。
 *
 * 要使用此功能，您需要将它们配置为事件页面中的注释：
 * ----------------------------------------------------------------------------
 * **弹出设置**
 * ----------------------------------------------------------------------------
 * REQUIRED
 * ~~~
 *  <qPopupSettings>
 *  OPTIONS
 *  </qPopupSettings>
 * ~~~
 * *每个option单独占一行*
 * - Possible Options:
 *  - "interval: X": 将X设置为弹出窗口之间的时间间隔。
 *  - "preset: X": 将X设置为要使用的预设ID。预设是使用插件命令创建的。
 * *需要在此事件页面开始之前配置预设，否则将不应用预设。
 *  - "ox: X": 将X设置为此弹出窗口相对于x位置的偏移量。可以为负。
 *  - "oy: Y": 将Y设置为此弹出窗口相对于y位置的偏移量。可以为负。
 *  - "duration: X": 将X设置为弹出窗口的持续时间。
 *
 * 弹出设置示例
 * ~~~
 *  <qPopupSettings>
 *  interval: 120
 *  duration: 60
 *  oy: -48
 *  </qPopupSettings>
 * ~~~
  * ----------------------------------------------------------------------------
 * **弹出样式**
  * ----------------------------------------------------------------------------
 * OPTIONAL
 * ~~~
 *  <qPopupStyle>
 *  OPTIONS
 *  </qPopupStyle>
 * ~~~
 * *每个option单独占一行*
 * - Possible Options:
 *  - "fontFace: X": 将X设置为字体名称。
 *  - "fontSize: X":将X设置为字体大小。
 *  - "color: X": 将X设置为字体颜色。
 *  - "padding: X": 将X设置为字体间距。
 *  - "windowed: X": 将X设置为true或false。
 *
 * 弹出样式示例
 * ~~~
 *  <qPopupStyle>
 *  fontSize: 18
 *  windowed: true
 *  color: #FF0000
 *  </qPopupStyle>
 * ~~~
 * ----------------------------------------------------------------------------
 * **弹出过渡**
 * ----------------------------------------------------------------------------
 * OPTIONAL
 * ~~~
 *  <qPopupTransitions>
 *  TRANSITIONS
 *  </qPopupTransitions>
 * ~~~
 * *每个transition单独占一行*
 * - 有关详细信息，请参见过渡部分。
 * ----------------------------------------------------------------------------
 * **qPopup**
 * ----------------------------------------------------------------------------
 * REQUIRED
 * ~~~
 * <qPopup>
 * STRING
 * </qPopup>
 * ~~~
 * - STRING: 在弹出窗口中使用的文本。您可以在消息框中使用一些转义码。
 *
 * 您可以根据需要添加很多的<qPopups></ qPopups>。将在每个间隔中随机选择一个。
 * ============================================================================
 * ## 过渡效果
 * ============================================================================
 * 过渡效果格式:
 * ~~~
 *  STARTFRAME DURATION TYPE PARAM1
 * ~~~
 * - STARTFRAME: 多少帧之后开始过渡。
 * - DURATION: 过渡效果持续多少帧
 * - TYPE: 过渡类型可以是：slideUp，slideDown，fadeIn，fadeOut。
 * - PARAM1: 如果type为slideUp或slideDown，则为滑动像素的距离。
 * ============================================================================
 * ## 插件命令
 * ============================================================================
 * **开始**
 * ----------------------------------------------------------------------------
 * ~~~
 *  qPopup start "STRING" [OPTIONS]
 * ~~~
 * - STRING: 弹出窗口文本。您可以在消息框中使用一些转义码。换行请使用\n。
 * - List of Options:
 *   - idX: 将X设置为此弹出窗口的ID。
 *   - presetX: 将X设置为要使用的样式预设ID。使用其他插件命令配置预设。
 *   - xX: 将X设置为此弹出窗口的x位置。如果使用bindToX，则将其忽略。
 *   - yX: 将X设置为此弹出窗口的y位置。如果使用bindToX，则将其忽略。
 *   - oxX: 将X设置为此弹出窗口相对于x位置的偏移量。可以为负。
 *   - oyY: 将Y设置为此弹出窗口相对于y位置的偏移量。可以为负。
 *   - bindToCHARAID: 其中CHARAID是弹出窗口要绑定的CHARAID。
 *     * CHARAID - CHARA标识符。
 *       - For player: 0, p, player
 *       - For events: EVENTID, eEVENTID or this
 *   - durationX: 将X设置为弹出窗口的持续时间。
 * ----------------------------------------------------------------------------
 * **配置样式**
 * ----------------------------------------------------------------------------
 * ~~~
 *  qPopup configStyle ID [OPTIONS]
 * ~~~
 * - ID: 要对其应用的预设ID。用于启动命令中presetX的option。
 * - List of Options:
 *   - fontX: 将X设置为字体名称。如果字体名称包含空格，则将选项包装在""中。
 *   - sizeX: 将X设置为字体大小。
 *   - colorX: 将X设置为字体颜色。
 *   - paddingX: 将设置X为字体间距。
 *   - windowed: 表示使用窗口皮肤。
 *
 * 如果该ID是先前设置的，它将被新的配置替换。
 * ----------------------------------------------------------------------------
 * **配置过渡**
 * ----------------------------------------------------------------------------
 * ~~~
 *  qPopup configTransition ID [TRANSITION]
 * ~~~
 * - ID: 要对其应用的预设ID。用于启动命令中presetX的option。
 * - TRANSITION: 要添加到此预设的过渡。 有关信息，请参见过渡部分。
 *
 * 如果该ID是先前设置的，则过渡将添加到其中。
 * ============================================================================
 * ## Links
 * ============================================================================
 * Formated Help:
 *
 *  https://quxios.github.io/plugins/QPopup
 *
 * RPGMakerWebs:
 *
 *  http://forums.rpgmakerweb.com/index.php?threads/qplugins.73023/
 *
 * Terms of use:
 *
 *  https://github.com/quxios/QMV-Master-Demo/blob/master/readme.md
 *
 * Like my plugins? Support me on Patreon!
 *
 *  https://www.patreon.com/quxios
 *
 * @tags map, popup
 */
/*~struct~PopupPreset:
* @param ID
* @desc 预设的唯一ID
* @default
*
* @param Style
*
* @param Font Face
* @parent Style
* @desc 字体名称
* @default GameFont
*
* @param Font Size
* @parent Style
* @desc 字体大小
* @default 28
*
* @param Font Color
* @parent Style
* @desc 字体颜色
* @default #ffffff
*
* @param Padding
* @parent Style
* @desc 字体间距
* @type Number
* @min 0
* @default 0
*
* @param Windowed
* @parent Style
* @desc 是否使用窗口皮肤
* @type Boolean
* @default false
*
* @param Transitions
* @desc 过渡效果列表
* 有关信息，请参见帮助中的“过渡”部分。
* @type String[]
* @default []
*/
//=============================================================================

//=============================================================================
// QPopup Static Class

function QPopup() {
  throw new Error('This is a static class');
}

//=============================================================================
// New Classes

function Window_QPopup() {
  this.initialize.apply(this, arguments);
}

//=============================================================================
// QPopup

(function() {
  var _PARAMS = QPlus.getParams('<QPopup>', {
    'Presets': []
  });
  var _PRESETS = {};
  _PARAMS.Presets.forEach(function(preset) {
    _PRESETS[preset.ID] = {
      style: {
        fontFace: preset['Font Face'],
        fontSize: preset['Font Size'],
        color: preset['Font Color'],
        padding: preset['Padding'],
        windowed: preset['Windowed']
      },
      transitions: preset.Transitions
    }
  });

  //-----------------------------------------------------------------------------
  // QPopup

  QPopup._popups = {};
  QPopup._mapId = -1;
  QPopup._defaultOptions = {
    id: '*',
    string: '',
    x: 0, y: 0, ox: 0, oy: 0,
    bindTo: null,
    duration: 120,
    isNotification: false
  }
  QPopup._defaultTransitions = [];
  QPopup._defaultStyle = {
    fontFace: 'GameFont',
    fontSize: 28,
    color: '#ffffff',
    padding: 18,
    windowed: false
  }

  QPopup.start = function(options) {
    options = Object.assign({}, this._defaultOptions, options);
    options.transitions = options.transitions || QPopup._defaultTransitions;
    options.style = Object.assign({}, this._defaultStyle, options.style);
    if (options.id === '*') {
      options.id = this.getUniqueQPopupId();
    }
    var popup = new Window_QPopup(options);
    this.add(popup);
    return popup;
  };

  QPopup.getUniqueQPopupId = function() {
    var id = '*0';
    var counter = 0;
    var newId = false;
    while (!newId) {
      id = '*' + counter;
      var newId = true;
      for (var ids in this._popups) {
        if (!this._popups.hasOwnProperty(ids)) continue;
        if (ids === id) {
          newId = false;
          break;
        }
      }
      counter++;
    }
    return id;
  };

  QPopup.add = function(popup) {
    if (popup.constructor !== Window_QPopup) return;
    if (this._popups[popup.id]) this.remove(this._popups[popup.id]);
    this._popups[popup.id] = popup;
    var scene = SceneManager._scene;
    if (scene.constructor !== Scene_Map) {
      scene.addChild(popup);
    } else {
      scene._spriteset._tilemap.addChild(popup);
    }
    popup.update();
  };

  QPopup.remove = function(popup) {
    if (typeof popup === 'string') {
      popup = this._popups[popup];
      if (popup === null) return;
    }
    if (popup.constructor !== Window_QPopup) return;
    var scene = SceneManager._scene;
    delete this._popups[popup.id];
    if (scene.constructor !== Scene_Map) {
      scene.removeChild(popup);
    } else {
      scene._spriteset._tilemap.removeChild(popup);
    }
  };

  //-----------------------------------------------------------------------------
  // Game_System

  var Alias_Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    Alias_Game_System_initialize.call(this);
    this._qPopupPresets = JSON.parse(JSON.stringify(_PRESETS));
  };

  Game_System.prototype.qPopupPreset = function(id) {
    if (!this._qPopupPresets[id]) {
      this._qPopupPresets[id] = {
        style: Object.assign({}, QPopup._defaultStyle),
        transitions: null
      }
    }
    return this._qPopupPresets[id];
  };

  //-----------------------------------------------------------------------------
  // Game_Interpreter

  var Alias_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    if (command.toLowerCase() === 'qpopup') {
      return this.qPopupCommand(QPlus.makeArgs(args));
    }
    Alias_Game_Interpreter_pluginCommand.call(this, command, args);
  };

  Game_Interpreter.prototype.qPopupCommand = function(args) {
    var cmd = args.shift().toLowerCase();
    switch (cmd) {
      case 'configstyle': {
        var id = args.shift();
        var fontFace = QPlus.getArg(args, /^font(.+)/i);
        var fontSize = Number(QPlus.getArg(args, /^size(\d+)/i));
        var color = QPlus.getArg(args, /^color(.+)/i);
        var padding = QPlus.getArg(args, /^padding(\d+)/i);
        var windowed = !!QPlus.getArg(args, /^windowed$/i);
        var style = {};
        if (fontFace) style.fontFace = fontFace;
        if (fontSize) style.fontSize = fontSize;
        if (color) style.color = color;
        if (padding) style.padding = Number(padding);
        style.windowed = windowed;
        var presetStyle = $gameSystem.qPopupPreset(id).style;
        Object.assign(presetStyle, style);
        break;
      }
      case 'configtransition': {
        var id = args.shift();
        if (!$gameSystem.qPopupPreset(id).transitions) {
          $gameSystem.qPopupPreset(id).transitions = [];
        }
        $gameSystem.qPopupPreset(id).transitions.push(args.join(' '));
      }
      case 'start': {
        var str = args.shift();
        var id = QPlus.getArg(args, /^id(.+)/i);
        var preset = QPlus.getArg(args, /^preset(.+)/i);
        var x = QPlus.getArg(args, /^x(-?\d+)/i);
        var y = QPlus.getArg(args, /^y(-?\d+)/i);
        var ox = QPlus.getArg(args, /^ox(-?\d+)/i);
        var oy = QPlus.getArg(args, /^oy(-?\d+)/i);
        var bindTo = QPlus.getArg(args, /^bindTo(.+)/i);
        if (bindTo && bindTo.toLowerCase() === 'this') {
          bindTo = this.character(0).charaId();
        }
        var duration = QPlus.getArg(args, /^duration(\d+)/i);
        if (preset) preset = $gameSystem.qPopupPreset(preset);
        var options = Object.assign({}, preset);
        if (id !== null) options.id = id;
        if (x !== null) options.x = Number(x);
        if (y !== null) options.y = Number(y);
        if (ox !== null) options.ox = Number(ox);
        if (oy !== null) options.oy = Number(oy);
        if (bindTo !== null) options.bindTo = bindTo;
        if (duration !== null) options.duration = Number(duration);
        options.string = str.replace(/\\n/g, '\n');
        QPopup.start(options);
      }
    }
  };

  //-----------------------------------------------------------------------------
  // Game_Map

  var Alias_Game_Map_setup = Game_Map.prototype.setup;
  Game_Map.prototype.setup = function(mapId) {
    Alias_Game_Map_setup.call(this, mapId);
    if (mapId !== QPopup._mapId) {
      QPopup._mapId = mapId;
      QPopup._popups = [];
    }
  };

  //-----------------------------------------------------------------------------
  // Game_Event

  var Alias_Game_Event_setupPage = Game_Event.prototype.setupPage;
  Game_Event.prototype.setupPage = function() {
    this._qPopups = null;
    Alias_Game_Event_setupPage.call(this);
  };

  var Alias_Game_Event_setupPageSettings = Game_Event.prototype.setupPageSettings;
  Game_Event.prototype.setupPageSettings = function() {
    Alias_Game_Event_setupPageSettings.call(this);
    this.setupQPopups();
  };

  Game_Event.prototype.setupQPopups = function() {
    var notes = this.notes(true);
    var settings = /<qPopupSettings>([\s\S]*)<\/qPopupSettings>/i.exec(notes);
    if (!settings) return;
    var style = /<qPopupStyle>([\s\S]*)<\/qPopupStyle>/i.exec(notes);
    var transitions = /<qPopupTransitions>([\s\S]*)<\/qPopupTransitions>/i.exec(notes);
    var popupsRegex = /<qPopup>([\s\S]*?)<\/qPopup>/ig;
    var popups = [];
    while (true) {
      var match = popupsRegex.exec(notes);
      if (match) {
        popups.push(match[1].trim());
      } else {
        break;
      }
    }
    settings = QPlus.stringToObj(settings[1]);
    settings.bindTo = this.charaId();
    style = style ? QPlus.stringToObj(style[1]) : null;
    transitions = transitions ? transitions.split('\n') : null;
    if (settings.preset) {
      var preset = $gameSystem.qPopupPreset(settings.preset);
      style = Object.assign(style || {}, preset.style);
      transitions = (transitions || []).concat(preset.transitions || []);
    }
    this._qPopups = {
      settings: settings,
      transitions: transitions,
      style: style,
      popups: popups,
      ticker: Math.randomIntBetween(0, settings.interval)
    }
  };

  var Alias_Game_Event_update = Game_Event.prototype.update;
  Game_Event.prototype.update = function() {
    Alias_Game_Event_update.call(this);
    if (this._qPopups && !this._locked) {
      this.updateQPopups();
    }
  };

  Game_Event.prototype.updateQPopups = function() {
    if (this._qPopups.ticker >= this._qPopups.settings.interval) {
      var i = Math.randomInt(this._qPopups.popups.length);
      var popup = QPopup.start(Object.assign({},
        this._qPopups.settings,
        {
          string: this._qPopups.popups[i],
          style: this._qPopups.style,
          transitions: this._qPopups.transitions
        }
      ))
      this._qPopups.ticker = -popup._duration || 0;
    } else {
      this._qPopups.ticker++;
    }
  };

  //-----------------------------------------------------------------------------
  // Window_QPopup

  Window_QPopup.prototype = Object.create(Window_Base.prototype);
  Window_QPopup.prototype.constructor = Window_QPopup;

  Window_QPopup.prototype.initialize = function(options) {
    this.id = options.id;
    this._style = options.style;
    this._transitions = options.transitions;
    Window_Base.prototype.initialize.call(this, 0, 0, 0, 0);
    this.realX = options.x;
    this.realY = options.y;
    this.z = 9;
    this._ox = options.ox;
    this._oy = options.oy;
    this._timeline = {};
    this._bindTo = options.bindTo;
    this._duration = options.duration;
    this._tick = 0;
    this._isNotification = options.isNotification;
    this.x = this.realX;
    this.y = this.realY;
    this.formatString(options.string);
    this.formatTransitions();
  };

  Object.defineProperty(Window_QPopup.prototype, 'realX', {
    get: function() {
      return this._realX + this._ox;
    },
    set: function(value) {
      this._realX = value;
    },
    configurable: true
  });

  Object.defineProperty(Window_QPopup.prototype, 'realY', {
    get: function() {
      return this._realY + this._oy;
    },
    set: function(value) {
      this._realY = value;
    },
    configurable: true
  });

  Window_QPopup.prototype.normalColor = function() {
    return this._style.color;
  };

  Window_QPopup.prototype.standardFontFace = function() {
    return this._style.fontFace;
  };

  Window_QPopup.prototype.standardFontSize = function() {
    return this._style.fontSize;
  };

  Window_QPopup.prototype.standardPadding = function() {
    return this._style.padding;
  };

  Window_QPopup.prototype.formatString = function(string) {
    this.resetFontSettings();
    this.updatePadding();
    if (!this._style.windowed) {
      this.opacity = 0;
    }
    var lines = string.split(/\n|\r/);
    var largestW = 0;
    for (var i = 0; i < lines.length; i++) {
      var line = this.convertEscapeCharacters(lines[i]);
      line = line.replace(/\x1b.+?\[.+?\]/, '');
      var w = this.contents.measureTextWidth(line);
      var icons = /\\I\[(\d+?)\]/g;
      while (true) {
        var match = icons.exec(lines[i]);
        if (match) {
          w += Window_Base._iconWidth + 4;
        } else {
          break;
        }
      }
      if (w > largestW) largestW = w;
    }
    this.width = largestW + this.standardPadding() * 2;
    this.height = this.calcTextHeight({
      text: string,
      index: 0
    }, true) + this.standardPadding() * 2;
    this.createContents();
    this.drawTextEx(string, 0, 0);
    this._ox -= this.width / 2;
    this._oy -= this.height;
  };

  Window_QPopup.prototype.formatTransitions = function() {
    var transitions = this._transitions;
    for (var i = 0; i < transitions.length; i++) {
      var params = transitions[i].split(' ');
      var transition = params[2] + ' ' + params[1] + ' ';
      var j;
      for (j = 3; j < params.length; j++) {
        transition += params[j];
        if (j !== params.length - 1) {
          transition += ' ';
        }
      }
      var startTime = Number(params[0]);
      var totalTime = Number(params[1]);
      for (j = 0; j < totalTime; j++) {
        if (!this._timeline[startTime + j]) {
          this._timeline[startTime + j] = [];
        }
        this._timeline[startTime + j].push(transition);
      }
    }
  };

  Window_QPopup.prototype.update = function() {
    Window_Base.prototype.update.call(this);
    if (this._bindTo !== null) {
      var chara = QPlus.getCharacter(this._bindTo);
      if (chara) {
        if (Imported.QMovement) {
          var x = chara.cx();
          var y = chara.cy();
        } else {
          var x = chara.x * $gameMap.tileWidth() + $gameMap.tileWidth() / 2;
          var y = chara.y * $gameMap.tileHeight();
        }
        this.realX = x;
        this.realY = y;
      }
    }
    if (!this._isNotification) {
      this.x = this.realX;
      this.x -= $gameMap.displayX() * $gameMap.tileWidth();
      this.y = this.realY;
      this.y -= $gameMap.displayY() * $gameMap.tileHeight();
    }
    this.updateTransition();
    if (this.isPlaying()) {
      this._tick++;
    } else {
      QPopup.remove(this);
    }
  };

  Window_QPopup.prototype.updateTransition = function() {
    var currentFrame = this._timeline[this._tick];
    if (currentFrame) {
      for (var i = 0; i < currentFrame.length; i++) {
        this.processAction(currentFrame[i].split(' '));
      }
    }
  };

  Window_QPopup.prototype.processAction = function(action) {
    switch (action[0].toLowerCase()) {
      case 'slideup':
        this.slideup(action);
        break;
      case 'slidedown':
        this.slidedown(action);
        break;
      case 'fadein':
        this.fadein(action);
        break;
      case 'fadeout':
        this.fadeout(action);
        break;
    }
  };

  Window_QPopup.prototype.slideup = function(action) {
    var duration = Number(action[1]);
    var distance = Number(action[2]);
    var speed = distance / duration;
    this.pivot.y += speed;
  };

  Window_QPopup.prototype.slidedown = function(action) {
    var duration = Number(action[1]);
    var distance = Number(action[2]);
    var speed = distance / duration;
    this.pivot.y -= speed;
  };

  Window_QPopup.prototype.fadeout = function(action) {
    var duration = Number(action[1]);
    var speed = 1 / duration;
    if (this.alpha > 0) {
      this.alpha -= speed;
    }
  };

  Window_QPopup.prototype.fadein = function(action) {
    var duration = Number(action[1]);
    var speed = 1 / duration;
    if (this.alpha < 1) {
      this.alpha += speed;
    }
  };

  Window_QPopup.prototype.isPlaying = function() {
    if (this._duration === -1) return true;
    return this._tick < this._duration;
  };

  Window_QPopup.prototype.isMouseInside = function() {
    // return false, this window needs to act more like a sprite
    return false;
  };
})()
