//=============================================================================
// Ryo_CommandChain.js
//=============================================================================

/*:
 * @plugindesc 类似QTE的一类插件。
 * @author RyoSonic
 *
 * @param Switch ID
 * @desc 默认成功后打开开关.
 * @default 20
 *
 * @param Max Time
 * @desc 默认QTE允许的时间限制.
 * @default 60
 *
 * @param Max Step
 * @desc 默认需要多少步骤完成QTE.
 * @default 1
 *
 * @param Layout X
 * @desc Layout装饰X轴.
 * @default 300
 *
 * @param Layout Y
 * @desc Layout装饰Y轴.
 * @default 200
 *
 * @param Key X
 * @desc 按键提示所在X轴.
 * @default 650
 *
 * @param Key Y
 * @desc 按键提示所在Y轴.
 * @default 200
 *
 * @param Meter X
 * @desc 时间提示所在X轴.
 * @default 46
 *
 * @param Meter Y
 * @desc 时间提示所在Y轴.
 * @default 25
 * @help 
 * 类似QTE的一类插件。
 * 使用方法插件命令：chain_command a b c abc均为大于0的数字。
 * a为成功后打开的开关，b为限制时间，c为按键次数。
 * 如果不设置则使用默认的设置。
 * 包括上下左右、确定、取消、pageup与pagedown。
 * 将time、meter与command扔到img/qte下。
 */

//=============================================================================
// ** PLUGIN PARAMETERS
//=============================================================================
var Imported = Imported || {};
Imported.Ryo_CommandChain = true;
var Ryo = Ryo || {}; 

Ryo.parameters = PluginManager.parameters('Ryo_CommandChain');
Ryo.qte_switchID = Number(Ryo.parameters['Switch ID'] || 30);
Ryo.qte_maxtime = Number(Ryo.parameters['Max Time'] || 60);
Ryo.qte_maxstep = Number(Ryo.parameters['Max Step'] || 1);
Ryo.qte_layoutx = Number(Ryo.parameters['Layout X']);
Ryo.qte_layouty = Number(Ryo.parameters['Layout Y']);
Ryo.qte_keyx = Number(Ryo.parameters['Key X']);
Ryo.qte_keyy = Number(Ryo.parameters['Key Y']);
Ryo.qte_meterx = Number(Ryo.parameters['Meter X']);
Ryo.qte_metery = Number(Ryo.parameters['Meter Y']);

//=============================================================================
// ** ImageManager
//=============================================================================

//==============================
// * QTE
//==============================
ImageManager.loadQTE = function(filename) {
    return this.loadBitmap('img/qte/', filename, 0, true);
};

//=============================================================================
// ** Game_Interpreter
//=============================================================================	

//==============================
// * PluginCommand
//==============================
var _ryo_qte_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _ryo_qte_pluginCommand.call(this,command, args)
    if (command === "chain_command") {
        this.rewriteQTE(args);
        SceneManager.push(Scene_Chain);
    }
}

//==============================
// * ReWrite QTE
//==============================
Game_Interpreter.prototype.rewriteQTE = function(args) {
    if (!args) return;
    var switchID = args[0] || 0;
    switchID = parseInt(switchID);
    var maxtime = args[1] || 0;
    maxtime = parseInt(maxtime);
    var maxstep = args[2] || 0;
    maxstep = parseInt(maxstep);
    if (switchID > 0 && maxtime > 0 && maxstep > 0) {
        $gameTemp._com = [switchID, maxtime, maxstep];
    } else {
        $gameTemp._com = [Ryo.qte_switchID, Ryo.qte_maxtime, Ryo.qte_maxstep];
    }
};

//=============================================================================
// ** Game_Temp
//=============================================================================	
var _ryo_qte_initialize = Game_Temp.prototype.initialize;
Game_Temp.prototype.initialize = function() {
    _ryo_qte_initialize.call(this);
    this._com = [Ryo.qte_switchID, Ryo.qte_maxtime, Ryo.qte_maxstep];
};

//=============================================================================
// ** Scene Chain
//=============================================================================
function Scene_Chain() {
    this.initialize.apply(this, arguments);
};

Scene_Chain.prototype = Object.create(Scene_MenuBase.prototype);
Scene_Chain.prototype.constructor = Scene_Chain;

//==============================
// * Initialize
//==============================
Scene_Chain.prototype.initialize = function() {
    Scene_MenuBase.prototype.initialize.call(this);	
    this.z = 25;
    this._data = [-1,-1,false];
    this._keyIndex = 0;
    this._duration = 0;
    this._meterflow = 0
};

//==============================
// * Create
//==============================
Scene_Chain.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
    this.loadImages();
    this.createLayout();
    this.createMeter();	
    this.createKeys();
    this.refresh();
}

//==============================
// * Load Images
//==============================
Scene_Chain.prototype.loadImages = function() {
    this._keysImg = ImageManager.loadQTE("Command"); 
    this._layoutImg = ImageManager.loadQTE("Time");
    this._meterImg = ImageManager.loadQTE("Meter");
};

//==============================
// * getData
//==============================
Scene_Chain.prototype.getData = function() {	
    this._data[0] = Math.floor(this._keysImg.width / 8);
    this._data[1] = this._keysImg.height;
};

//==============================
// * create Layout
//==============================
Scene_Chain.prototype.createLayout = function() {	
    this._layout = new Sprite(this._layoutImg);
    this._layout.x = Ryo.qte_layoutx;
    this._layout.y = Ryo.qte_layouty;
    this.addChild(this._layout);
};

//==============================
// * create Keys
//==============================
Scene_Chain.prototype.createKeys = function() {
    this._keys = new Sprite(this._keysImg);
    this._keys.x = Ryo.qte_keyx;
    this._keys.y = Ryo.qte_keyy;	
    this.addChild(this._keys);
};

//==============================
// * create Meter
//==============================
Scene_Chain.prototype.createMeter = function() {
    this._meter = new Sprite(this._meterImg);
    this._meter.x = Ryo.qte_meterx + Ryo.qte_layoutx;
    this._meter.y = Ryo.qte_metery + Ryo.qte_layouty;	
    this._meter.opacity = 0;
    this.addChild(this._meter);	
};

//==============================
// * Need Refresh
//==============================
Scene_Chain.prototype.needRefresh = function() {
    if (!this._data[2]) {return false};
    return true;
};

//==============================
// * Refresh
//==============================
Scene_Chain.prototype.refresh = function() {
    this._data[2] = true;	
    this._duration = $gameTemp._com[1];
    this._keys.opacity = 255;
    this._keys.scale.x = 2.0;
    this._keys.scale.y = 2.0;
    this._keys.setFrame(this._data[0] * this._keyIndex,0,this._data[0],this._data[1]);
};

//==============================
// * Update Action
//==============================
Scene_Chain.prototype.updateAction = function() {
    this.updateCommands();
    if (this._duration > 0) {this.updateTime()};
};

//==============================
// * Update Action
//==============================
Scene_Chain.prototype.updateTime = function() {
    this._duration --
    this.refresh_meter(this._meter, this._duration, $gameTemp._com[1], this._meterflow);
    if (this._meter.opacity != 255) this._meter.opacity = 255;
    this._meterflow += 1.5;
    if (this._meterflow >= this._meter.bitmap.width * 2 /3) this._meterflow = 0;
    if (this._duration <= 0) {this.setWrong();}; 
};

//==============================
// * Update Meter
//==============================
Scene_Chain.prototype.refresh_meter = function(sprite,value,value_max,flow) {
    var cw = sprite.bitmap.width / 3
    var ch = sprite.bitmap.height;
    var meter_rate = cw * value / value_max;
    sprite.setFrame(flow, 0, meter_rate, ch);
};
//==============================
// * Update Action
//==============================
Scene_Chain.prototype.check_key = function(value) {
    if (value == this._keyIndex) {		
	this.nextKey();
    } else {
        this.setWrong();
    };
};

//==============================
// * set Wrong
//==============================
Scene_Chain.prototype.setWrong = function(value) {
    SoundManager.playBuzzer();
    this.clearCommands();
    SceneManager.pop();
};

//==============================
// * Next Key
//==============================
Scene_Chain.prototype.nextKey = function(value) {
    if ($gameTemp._com[2] <= 0) {
        this.enableAction();
        return
    };	
    SoundManager.playCursor();
    this._keyIndex = Math.randomInt(8);
    $gameTemp._com[2] -= 1;
    this.refresh();
};

//==============================
// * enable Action
//==============================
Scene_Chain.prototype.enableAction = function(value) {
    SoundManager.playUseSkill();
    this.clearCommands();
    $gameSwitches.setValue($gameTemp._com[0], true);
    SceneManager.pop();
};

//==============================
// * clear Commands
//==============================
Scene_Chain.prototype.clearCommands = function() {	
    this._duration = 0;
    this._data[2] = false;
};

//==============================
// * Update Commands
//==============================
Scene_Chain.prototype.updateCommands = function() {
    if (Input.isTriggered("right")) {this.check_key(0)}
    else if (Input.isTriggered("left")) {this.check_key(1)}
    else if (Input.isTriggered("down")) {this.check_key(2)}
    else if (Input.isTriggered("up")) {this.check_key(3)}
    else if (Input.isTriggered("ok")) {this.check_key(4)}
    else if (Input.isTriggered("cancel")) {this.check_key(5)}
    else if (Input.isTriggered("pageup")) {this.check_key(6)}
    else if (Input.isTriggered("pagedown")) {this.check_key(7)}
};

//==============================
// * Update
//==============================
Scene_Chain.prototype.update = function() {
    Sprite.prototype.update.call(this);	
    if (this._data[0] === -1) {
        if (this._keysImg.isReady() && this._meterImg.isReady()) {this.getData()};
        return;
    };
    if (this.needRefresh()) {this.nextKey()};
    this.updateAction();
    this._data[2] = false;
    if (this._keys.scale.x > 1.00) {this._keys.scale.x -= 0.1};
    this._keys.scale.y = this._keys.scale.x
};