#extension GL_OES_EGL_image_external : require
precision mediump float;

uniform samplerExternalOES sTexture;
uniform int textureWidth;
uniform int textureHeight;

varying vec2 texCoord;


///////////////////////////////
/////////// Helpers ///////////
///////////////////////////////


highp float lum(lowp vec3 c) {
    return dot(c, vec3(0.3, 0.59, 0.11));
}

lowp vec3 clipcolor(lowp vec3 c) {
    highp float l = lum(c);
    lowp float n = min(min(c.r, c.g), c.b);
    lowp float x = max(max(c.r, c.g), c.b);

    if (n < 0.0) {
        c.r = l + ((c.r - l) * l) / (l - n);
        c.g = l + ((c.g - l) * l) / (l - n);
        c.b = l + ((c.b - l) * l) / (l - n);
    }
    if (x > 1.0) {
        c.r = l + ((c.r - l) * (1.0 - l)) / (x - l);
        c.g = l + ((c.g - l) * (1.0 - l)) / (x - l);
        c.b = l + ((c.b - l) * (1.0 - l)) / (x - l);
    }

    return c;
}

lowp vec3 setlum(lowp vec3 c, highp float l) {
    highp float d = l - lum(c);
    c = c + vec3(d);
    return clipcolor(c);
}

highp float sat(lowp vec3 c) {
    lowp float n = min(min(c.r, c.g), c.b);
    lowp float x = max(max(c.r, c.g), c.b);
    return x - n;
}

lowp float mid(lowp float cmin, lowp float cmid, lowp float cmax, highp float s) {
    return ((cmid - cmin) * s) / (cmax - cmin);
}

lowp vec3 setsat(lowp vec3 c, highp float s) {
    if (c.r > c.g) {
        if (c.r > c.b) {
            if (c.g > c.b) {
                /* g is mid, b is min */
                c.g = mid(c.b, c.g, c.r, s);
                c.b = 0.0;
            } else {
                /* b is mid, g is min */
                c.b = mid(c.g, c.b, c.r, s);
                c.g = 0.0;
            }
            c.r = s;
        } else {
            /* b is max, r is mid, g is min */
            c.r = mid(c.g, c.r, c.b, s);
            c.b = s;
            c.r = 0.0;
        }
    } else if (c.r > c.b) {
        /* g is max, r is mid, b is min */
        c.r = mid(c.b, c.r, c.g, s);
        c.g = s;
        c.b = 0.0;
    } else if (c.g > c.b) {
        /* g is max, b is mid, r is min */
        c.b = mid(c.r, c.b, c.g, s);
        c.g = s;
        c.r = 0.0;
    } else if (c.b > c.g) {
        /* b is max, g is mid, r is min */
        c.g = mid(c.r, c.g, c.b, s);
        c.b = s;
        c.r = 0.0;
    } else {
        c = vec3(0.0);
    }
    return c;
}

// coef - Aspect ratio. normalization coefficients for x and y. Without this param elipse will be drawn on rectangled displays
// circleCenter - (0.5, 0.5) screen center. Range [0.0, 1.0]
// point - position of processed pixel/texel
// colorN - color in format rgba
// stopN - stop points of gradient in percents [0.0 - 1.0]
vec4 radialGradient2(vec2 coef, vec2 circleCenter, vec2 point,
    vec4 color1, float stop1,
    vec4 color2, float stop2)
{
    point /= coef; // normalize texel posiition
    circleCenter /= coef; // normalize center of circle position

    float pct = distance(point, circleCenter); // calculate distance from texel to circle center

    // texel is located before first stop point
    if (pct <= stop1 ) {
        return color1;
    }

    // texel is located between first and second stop points
    if (pct > stop1 && pct <= stop2) {
        float diff = pct - stop1;
        float gradientLength = stop2 - stop1;
        vec4 color = mix(color1, color2, diff / gradientLength);

        return color;
    }

    // texel is located after third stop point
    return color2;
}


// coef - normalization coefficients for x and y. Without this param elipse will be drawn on rectangled displays
// circleCenter - (0.5, 0.5) screen center. Range [0.0, 1.0]
// point - position of processed pixel/texel
// colorN - color in format rgba
// stopN - stop points of gradient in percents [0.0 - 1.0]
vec4 radialGradient3(vec2 coef, vec2 circleCenter, vec2 point,
    vec4 color1, float stop1,
    vec4 color2, float stop2,
    vec4 color3, float stop3)
{
    point /= coef; // normalize texel posiition
    circleCenter /= coef; // normalize center of circle position

    float pct = distance(point, circleCenter); // calculate distance from texel to circle center

    // texel is located before first stop point
    if (pct <= stop1 ) {
        return color1;
    }

    // texel is located between first and second stop points
    if (pct > stop1 && pct <= stop2) {
        float diff = pct - stop1;
        float gradientLength = stop2 - stop1;
        vec4 color = mix(color1, color2, diff / gradientLength);

        return color;
    }

    // texel is located between second and third stop points
    if (pct > stop2 && pct <= stop3) {
        float diff = pct - stop2;
        float gradientLength = stop3 - stop2;
        vec4 color = mix(color2, color3, diff / gradientLength);

        return color;
    }

    // texel is located after third stop point
    return color3;
}

vec2 getAspectRatioCoef(int width, int height) {
    float f_width = float(width);
    float f_height = float(height);
    float ratio = min(f_width, f_height) / max(f_width, f_height);

    bool isPortraitMode = width < height;
    vec2 coef = vec2(isPortraitMode ? 1.0 : ratio, isPortraitMode ? ratio : 1.0);

    return coef;
}


///////////////////////////////
//////// End of helpers ///////
///////////////////////////////



///////////////////////////////
///////Color processing////////
///////////////////////////////


vec3 kLuminanceWeighting = vec3(0.2125, 0.7154, 0.0721);  // Values from "Graphics Shaders: Theory and Practice" by Bailey and Cunningham

// brightness: 0.0 - 1.0
// 0.0 - everything is black, 1.0 - original color, > 1.0 - brighten color
// Range of brightness value the same as in webkit
// https://developer.mozilla.org/en-US/docs/Web/CSS/filter-function/brightness
vec4 BrightnessCorrection(vec4 in_texel, float brightness) {
    return vec4(in_texel.rgb + (brightness - 1.0), in_texel.a);
}

// contrast: 0.0 - 4.0
vec4 ContrastCorrection(vec4 in_texel, float contrast) {
    return vec4(((in_texel.rgb - vec3(0.5)) * contrast + vec3(0.5)), in_texel.a);
}

//angle: 0.0 - 2.0
vec4 SaturationCorrection(vec4 in_texel, float satuartion) {
    float luminance = dot(in_texel.rgb, kLuminanceWeighting);
    return vec4(mix(vec3(luminance), in_texel.rgb, satuartion), in_texel.a);
}

//angle: 0.0 - 2.0
vec4 VibranceCorrection(vec4 in_texel, float vibrance) {
    vec3 color  = in_texel.rgb;
    float average = (color.r + color.g + color.b) / 3.0;
    float mx = max(color.r, max(color.g, color.b));
    float amt = (mx - average) * (-vibrance * 3.0);
    color.rgb = mix(color.rgb, vec3(mx), amt);
    return vec4(color.rgb,1.0);
}

// intensity: 0.0h - 1.0
vec4 SepiaEffect(vec4 in_texel, float intensity) {
    mat3 color_matrix = mat3(vec3(0.393, 0.769, 0.189),
                                   vec3(0.349, 0.686, 0.168),
                                   vec3(0.272, 0.534, 0.131));
    vec3 in_color = in_texel.rgb;
    vec3 out_color = in_color * color_matrix;
    return vec4(intensity * out_color  + (1.0 - intensity) * in_color, in_texel.a);
}


vec4 GrayscaleEffect(vec4 texel, float intensity) {
    float luminance = dot(texel.rgb, kLuminanceWeighting);
    vec4 res = vec4(texel.rgb * (1.0 - intensity) + (luminance * intensity), 1.0);
    return res;
}

vec4 rotateHue(vec4 color, float deg) {
    float rad = 0.55; //radians(deg);
    float cosA = cos(rad);
    float sinA = sin(rad);

    vec3 neo = vec3((cosA + (1. - cosA) / 3.),
    ((1. - cosA) / 3. - .57735 * sinA),
    ((1. - cosA) / 3. + .57735 * sinA)
    );

    vec3 result = vec3(color.r * neo.r + color.g * neo.g + color.b * neo.b,
    color.r * neo.b + color.g * neo.r + color.b * neo.g,
    color.r * neo.g + color.g * neo.b + color.b * neo.r);

    return vec4(result, color.a);
}


///////////////////////////////
////End of color processing////
///////////////////////////////




///////////////////////////////
/////////// Blends ////////////
///////////////////////////////


vec4 DarkenBlend(vec4 base, vec4 overlay) {
    return vec4(min(overlay.rgb * base.a, base.rgb * overlay.a) +
                 overlay.rgb * (1.0 - base.a) +
                 base.rgb * (1.0 - overlay.a), 1.0);
}


vec4 LightenBlend(vec4 texel1, vec4 texel2) {
    return max(texel1, texel2);
}


vec3 ScreenBlend(vec3 base, vec3 blend) {
    return 1.0-((1.0-base)*(1.0-blend));
}

vec3 ScreenBlend(vec3 base, vec3 blend, float opacity) {
    return (ScreenBlend(base, blend) * opacity + base * (1.0 - opacity));
}

vec4 ScreenBlend(vec4 texel1, vec4 texel2) {
    return vec4(ScreenBlend(texel1.rgb, texel2.rgb, texel2.a), 1.0);
}



vec4 OverlayBlend(vec4 base, vec4 overlay) {
    float ra;
    if (2.0 * base.r < base.a) {
        ra = 2.0 * overlay.r * base.r + overlay.r * (1.0 - base.a) + base.r * (1.0 - overlay.a);
    } else {
        ra = overlay.a * base.a - 2.0 * (base.a - base.r) * (overlay.a - overlay.r) + overlay.r * (1.0 - base.a) + base.r * (1.0 - overlay.a);
    }

    float ga;
    if (2.0 * base.g < base.a) {
        ga = 2.0 * overlay.g * base.g + overlay.g * (1.0 - base.a) + base.g * (1.0 - overlay.a);
    } else {
        ga = overlay.a * base.a - 2.0 * (base.a - base.g) * (overlay.a - overlay.g) + overlay.g * (1.0 - base.a) + base.g * (1.0 - overlay.a);
    }

    float ba;
    if (2.0 * base.b < base.a) {
        ba = 2.0 * overlay.b * base.b + overlay.b * (1.0 - base.a) + base.b * (1.0 - overlay.a);
    } else {
        ba = overlay.a * base.a - 2.0 * (base.a - base.b) * (overlay.a - overlay.b) + overlay.b * (1.0 - base.a) + base.b * (1.0 - overlay.a);
    }

    return vec4(ra, ga, ba, 1.0);
}

vec4 ColorDodgeBlend(vec4 base, vec4 overlay) {
    vec3 baseOverlayAlphaProduct = vec3(overlay.a * base.a);
    vec3 rightHandProduct = overlay.rgb * (1.0 - base.a) + base.rgb * (1.0 - overlay.a);

    vec3 firstBlendColor = baseOverlayAlphaProduct + rightHandProduct;
    vec3 overlayRGB = clamp((overlay.rgb / clamp(overlay.a, 0.01, 1.0)) * step(0.0, overlay.a), 0.0, 0.99);

    vec3 secondBlendColor = (base.rgb * overlay.a) / (1.0 - overlayRGB) + rightHandProduct;

    vec3 colorChoice = step((overlay.rgb * base.a + base.rgb * overlay.a), baseOverlayAlphaProduct);

    return vec4(mix(firstBlendColor, secondBlendColor, colorChoice), 1.0);
}

vec4 SoftLightBlend(vec4 base, vec4 overlay) {
    float alphaDivisor = base.a + step(base.a, 0.0); // Protect against a divide-by-zero blacking out things in the output
    return base * (overlay.a * (base / alphaDivisor) + (2.0 * overlay * (1.0 - (base / alphaDivisor)))) + overlay * (1.0 - base.a) + base * (1.0 - overlay.a);
}

vec4 ExclusionBlend(vec4 base, vec4 overlay) {
    return vec4((overlay.rgb * base.a + base.rgb * overlay.a - 2.0 * overlay.rgb * base.rgb) + overlay.rgb * (1.0 - base.a) + base.rgb * (1.0 - overlay.a), base.a);
}

vec4 ColorBlend(highp vec4 baseColor, highp vec4 overlayColor) {
    vec4 color = vec4(baseColor.rgb * (1.0 - overlayColor.a) + setlum(overlayColor.rgb, lum(baseColor.rgb)) * overlayColor.a, baseColor.a);
    return color;
}

vec4 ColorBurnBlend(vec4 base, vec4 overlay) {
    mediump vec4 whiteColor = vec4(1.0);
    vec4 color = whiteColor - (whiteColor - base) / overlay;

    return color;
}

vec4 MultiplyBlend(vec4 base, vec4 overlay) {
    return vec4(overlay * base + overlay * (1.0 - base.a) + base * (1.0 - overlay.a));
}

vec4 HueBlend(vec4 base, vec4 overlay) {
    vec4 color = vec4(base.rgb * (1.0 - overlay.a) + setlum(setsat(overlay.rgb, sat(base.rgb)), lum(base.rgb)) * overlay.a, base.a);
    return color;
}

///////////////////////////////
//////// End of Blends ////////
///////////////////////////////





///////////////////////////////
////////// Filters ////////////
///////////////////////////////

vec4 FilterNone(vec4 texel) {
    return texel;
}

vec4 Filter1977(vec4 texel) {
    texel = ContrastCorrection(texel, 1.1);
    texel = BrightnessCorrection(texel, 1.1);
    texel = SaturationCorrection(texel, 1.3);

    vec4 after = vec4(0.952, 0.415, 0.737, 0.3);
    texel = ScreenBlend(texel, after);

    return texel;
}

vec4 FilterAden(vec4 texel) {
    texel = rotateHue(texel, -20.);

    texel = ContrastCorrection(texel, 0.9);
    texel = SaturationCorrection(texel, 0.85);
    texel = BrightnessCorrection(texel, 1.2);

    vec4 start = vec4(0.258, 0.039, 0.054, 0.2);
    vec4 stop = vec4(1.0, 1.0, 1.0, 0.0);
    vec4 color = mix(start, stop, texCoord.x / 1.0); // linear interpolation (gradient)

    texel = DarkenBlend(texel, color);

    return texel;
}

vec4 FilterAmaro(vec4 texel) {
    texel = rotateHue(texel, -10.0);
    texel = ContrastCorrection(texel, 0.9);
    texel = BrightnessCorrection(texel, 1.1);
    texel = SaturationCorrection(texel, 1.5);

    // TODO: https://github.com/una/CSSgram/blob/master/source/css/amaro.css - there is no color for blending
    //    texel = ScreenBlend(texel);

    return texel;
}

vec4 FilterBrannan(vec4 texel) {
    texel = SepiaEffect(texel, 0.5);
    texel = ContrastCorrection(texel, 1.4);

    vec4 after = vec4(0.631, 0.172, 0.780, 0.31);
    texel = LightenBlend(texel, after);
    return texel;
}

vec4 FilterBrooklyn(vec4 texel) {
    texel = ContrastCorrection(texel, 0.9);
    texel = BrightnessCorrection(texel, 1.1);

    vec4 colorA = vec4(0.658, 0.874, 0.756, 0.4); // alpha: 1.0 looks better
    vec4 colorB = vec4(0.768, 0.717, 0.784, 1.0);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.7, colorB, 1.0);
    texel = OverlayBlend(texel, after);

    return texel;
}

vec4 FilterClarendon(vec4 texel) {
    vec4 before = vec4(0.498, 0.733, 0.89, 0.2);
    texel = OverlayBlend(texel, before);

    texel = ContrastCorrection(texel, 1.2);
    texel = SaturationCorrection(texel, 1.35);

    return texel;
}

vec4 FilterEarlybird(vec4 texel) {

    texel = ContrastCorrection(texel, 0.9);
    texel = SepiaEffect(texel, 0.2);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec2 centerOffset = vec2(0.5);

    vec4 colorA = vec4(0.815, 0.729, 0.556, 1.0);
    vec4 colorB = vec4(0.211, 0.011, 0.035, 1.0);
    vec4 colorC = vec4(0.113, 0.007, 0.062, 1.0);

    vec4 after = radialGradient3(coef, centerOffset, texCoord, colorA, 0.2, colorB, 0.85, colorC, 1.0);
    texel = OverlayBlend(texel, after);

    return texel;
}

vec4 FilterGingham(vec4 texel) {
    texel = BrightnessCorrection(texel, 1.05);
    texel = rotateHue(texel, -10.0);

    vec4 lavender = vec4(0.901, 0.901, 0.980, 1.0);
    texel = SoftLightBlend(texel, lavender);

    return texel;
}

vec4 FilterHudson(vec4 texel) {
    texel = BrightnessCorrection(texel, 1.2);
    texel = ContrastCorrection(texel, 0.9);
    texel = SaturationCorrection(texel, 1.1);

    vec4 colorA = vec4(0.65, 0.694, 1.0, 1.0);
    vec4 colorB = vec4(0.203, 0.129, 0.203, 1.0);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.5, colorB, 1.0);
    texel = MultiplyBlend(texel, after);
//    texel = MultiplyBlend(texel, vec4(after.rgb, 0.5));

    return texel;
}

vec4 FilterInkwell(vec4 texel) {
    vec4 texel_orig = texel;

    texel = SepiaEffect(texel, 0.3);
    texel = ContrastCorrection(texel, 1.1);
    texel = BrightnessCorrection(texel, 1.1);
    texel = GrayscaleEffect(texel_orig, 1.0);

    return texel;
}

vec4 FilterKelvin(vec4 texel) {
    vec4 before = vec4(0.219, 0.172, 0.203, 1.0);
    texel = ColorDodgeBlend(texel, before);
    vec4 after = vec4(0.717, 0.490, 0.129, 1.0);
    texel = OverlayBlend(texel, after);
    return texel;
}

vec4 FilterLark(vec4 texel) {
    vec4 before = vec4(0.133, 0.145, 0.247, 1.0);
    texel = ColorDodgeBlend(texel, before);

    texel = ContrastCorrection(texel, 0.9);

    vec4 after = vec4(0.949, 0.949, 0.949, 0.8);
    texel = DarkenBlend(texel, after);

    return texel;
}


vec4 FilterLofi(vec4 texel) {
    texel = SaturationCorrection(texel, 1.1);
    texel = ContrastCorrection(texel, 1.5);

    vec4 colorA = vec4(vec3(1.0), 0.0); // transparent
    vec4 colorB = vec4(vec3(0.133), 1.0);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.7, colorB, 1.5);

    texel = MultiplyBlend(texel, after);

    return texel;
}

vec4 FilterMaven(vec4 texel) {
    texel = SepiaEffect(texel, 0.25);
    texel = BrightnessCorrection(texel, 0.95);
    texel = ContrastCorrection(texel, 0.95);
    texel = SaturationCorrection(texel, 1.5);

    vec4 after = vec4(0.011, 0.901, 0.101, 0.2);

    texel = HueBlend(texel, after);

    return texel;
}

vec4 FilterMayfair(vec4 texel) {
    texel = ContrastCorrection(texel, 1.1);
    texel = SaturationCorrection(texel, 1.1);

    vec4 colorA = vec4(1.0, 1.0, 1.0, 0.8);
    vec4 colorB = vec4(1.0, 0.784, 0.784, 0.6);
    vec4 colorC = vec4(1.0, 1.0, 1.0, 1.0);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient3(coef, vec2(0.4), texCoord, colorA, 0.0, colorB, 0.3, colorC, 0.6);

    texel = OverlayBlend(texel, vec4(after.grb, 0.4));

    return texel;
}


vec4 FilterMoon(vec4 texel) {
    vec4 before = vec4(vec3(0.627), 1.0);
    texel = SoftLightBlend(texel, before);

    texel = GrayscaleEffect(texel, 1.0);
    texel = ContrastCorrection(texel, 1.1);
    texel = BrightnessCorrection(texel, 1.1);

    vec4 after = vec4(vec3(0.219), 1.0);
    texel = LightenBlend(texel, after);

    return texel;
}

vec4 FilterNashville(vec4 texel) {
    vec4 before = vec4(0.968, 0.690, 0.600, 0.56);
    texel = DarkenBlend(texel, before);
    texel = SepiaEffect(texel, 0.2);
    texel = ContrastCorrection(texel, 1.1);
    texel = BrightnessCorrection(texel, 1.05);
    texel = SaturationCorrection(texel, 1.2);
    vec4 after = vec4(0.0, 0.274, 0.588, 0.4);
    texel = LightenBlend(texel, after);
    return texel;
}

vec4 FilterPerpetua(vec4 texel) {
    // linear gradient to bottom
    vec3 color1 = vec3(0.0, 0.356, 0.603);
    vec3 color2 = vec3(0.901, 0.756, 0.239);
    vec3 after = mix(color1, color2, texCoord.y / 1.0);

    texel = SoftLightBlend(texel, vec4(after, 0.5));
    return texel;
}

vec4 FilterReyes(vec4 texel) {
    texel = SepiaEffect(texel, 0.22);
    texel = BrightnessCorrection(texel, 1.1);
    texel = ContrastCorrection(texel, 0.85);
    texel = SaturationCorrection(texel, 0.75);

    vec4 after = vec4(0.937, 0.803, 0.678, 1.0);
    texel = SoftLightBlend(texel, after);

    return texel;
}

// TODO: double check final OverlayBlend (CSS has an opacity property after blend)
vec4 FilterRise(vec4 texel) {
    vec4 beforeColorA = vec4(0.925, 0.803, 0.662, 0.15);
    vec4 beforeColorB = vec4(0.196, 0.117, 0.027, 0.4);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 before = radialGradient2(coef, vec2(0.5), texCoord, beforeColorA, 0.55, beforeColorB, 1.0);

    texel = MultiplyBlend(texel, before);

    texel = BrightnessCorrection(texel, 1.05);
    texel = SepiaEffect(texel, 0.2);
    texel = ContrastCorrection(texel, 0.9);
    texel = SaturationCorrection(texel, 0.9);

    vec4 colorA = vec4(0.909, 0.772, 0.596, 0.8);
    vec4 colorB = vec4(0.909, 0.772, 0.596, 0.0);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.0, colorB, 0.9);

    texel = OverlayBlend(texel, vec4(after.rgb, 0.4));

    return texel;
}


vec4 FilterSepiaTone(vec4 texel) {
    return SepiaEffect(texel, 1.0);
}

vec4 FilterSlumber(vec4 texel) {
    vec4 before = vec4(0.27, 0.15, 0.047, 0.4);
    texel = LightenBlend(texel, before);

    texel = SaturationCorrection(texel, 0.66);
    texel = BrightnessCorrection(texel, 1.05);

    vec4 after = vec4(0.49, 0.411, 0.094, 0.5);
    texel = SoftLightBlend(texel, after);

    return texel;
}

vec4 FilterStinson(vec4 texel) {
    vec4 before = vec4(0.941, 0.584, 0.501, 0.2);
    texel = SoftLightBlend(texel, before);

    texel = ContrastCorrection(texel, 0.75);
    texel = SaturationCorrection(texel, 0.85);
    texel = BrightnessCorrection(texel, 1.15);

    return texel;
}

vec4 FilterToaster(vec4 texel) {
    texel = ContrastCorrection(texel, 1.5);
    texel = BrightnessCorrection(texel, 0.9);

    vec4 colorA = vec4(0.501, 0.305, 0.058, 1.0);
    vec4 colorB = vec4(0.231, 0.0, 0.231, 1.0);
    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.0, colorB, 1.0);

    texel = ScreenBlend(texel, after);
    return texel;
}

vec4 FilterValencia(vec4 texel) {
    texel = ContrastCorrection(texel, 1.08);
    texel = BrightnessCorrection(texel, 1.08);
    texel = SepiaEffect(texel, 0.08);
    vec4 after = vec4(0.227, 0.111, 0.223, 0.5);
    texel = ExclusionBlend(texel, after);

    return texel;
}

vec4 FilterWalden(vec4 texel) {
    texel = BrightnessCorrection(texel, 1.1);

    texel = rotateHue(texel, -10.);

    texel = SepiaEffect(texel, 0.3);
    texel = SaturationCorrection(texel, 1.6);
    vec4 after = vec4(0.0, 0.172, 0.8, 0.3);
    texel = ScreenBlend(texel, after);

    return texel;
}

vec4 FilterWillow(vec4 texel) {
    vec4 colorA = vec4(0.803, 0.67, 0.686, 1.0);
    vec4 colorB = vec4(vec3(0.0), 1.0);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 before = radialGradient2(coef, vec2(0.4), texCoord, colorA, 0.55, colorB, 1.5);

    texel = OverlayBlend(texel, before);
    texel = GrayscaleEffect(texel, 0.5);
    texel = ContrastCorrection(texel, 0.95);
    texel = BrightnessCorrection(texel, 0.9);

    vec4 after = vec4(0.847, 0.803, 0.796, 1.0);
    texel = ColorBlend(texel, after);

    return texel;
}

vec4 FilterXpro2(vec4 texel) {
    vec4 colorA = vec4(0.901, 0.905, 0.878, 1.0);
    vec4 colorB = vec4(0.168, 0.164, 0.631, 0.6);

    vec2 coef = getAspectRatioCoef(textureWidth, textureHeight);
    vec4 after = radialGradient2(coef, vec2(0.5), texCoord, colorA, 0.4, colorB, 1.1);

    texel = SepiaEffect(texel, 0.3);
    texel = ColorBurnBlend(texel, after);

    return texel;
}

///////////////////////////////
//////// End of Filters ///////
///////////////////////////////



// TODO: OpenGL doesn't support preprocessor's "#include"
// Here's a possible solution to split up this file (color processing and blends could be moved to different files)
// https://github.com/tntmeijs/GLSL-Shader-Includes
void main() {
    vec4 texel = texture2D(sTexture, texCoord);

//    textureSize(sTexture);

//    texel = Filter1977(texel);
//    texel = FilterAden(texel);
//    texel = FilterAmaro(texel);
//    texel = FilterBrannan(texel);
//    texel = FilterBrooklyn(texel);

//    texel = FilterClarendon(texel);
//    texel = FilterEarlybird(texel);
//    texel = FilterGingham(texel);
//    texel = FilterHudson(texel);
//    texel = FilterInkwell(texel);

//    texel = FilterKelvin(texel);
//    texel = FilterLark(texel);
//    texel = FilterLofi(texel);
//    texel = FilterMaven(texel);
//    texel = FilterMayfair(texel);

//    texel = FilterMoon(texel);
//    texel = FilterNashville(texel);
//    texel = FilterPerpetua(texel);
//    texel = FilterReyes(texel);
//    texel = FilterRise(texel);

//    texel = FilterSepiaTone(texel);
//    texel = FilterSlumber(texel);
//    texel = FilterStinson(texel);
//    texel = FilterToaster(texel);
//    texel = FilterValencia(texel);

//    texel = FilterWalden(texel);
//    texel = FilterWillow(texel);
//    texel = FilterXpro2(texel);


    //texel = FilterNone(texel);


    gl_FragColor = texel;
}
