#extension GL_OES_EGL_image_external : require
precision mediump float;

uniform samplerExternalOES sTexture;
varying vec2 texCoord;

vec3 kLuminanceWeighting = vec3(0.2125, 0.7154, 0.0721);// Values from "Graphics Shaders: Theory and Practice" by Bailey and Cunningham

//angle: 0.0 - 2.0
vec4 VibranceCorrection(vec4 in_texel, float vibrance) {
    vec3 color  = in_texel.rgb;
    float average = (color.r + color.g + color.b) / 3.0;
    float mx = max(color.r, max(color.g, color.b));
    float amt = (mx - average) * (-vibrance * 3.0);
    color.rgb = mix(color.rgb, vec3(mx), amt);
    return vec4(color.rgb,1.0);
}

//angle: 0.0 - 2.0
vec4 SaturationCorrection(vec4 in_texel, float satuartion) {
    float luminance = dot(in_texel.rgb, kLuminanceWeighting);
    return vec4(mix(vec3(luminance), in_texel.rgb, satuartion), in_texel.a);
}

vec4 SoftLightBlend(vec4 base, vec4 overlay) {
    float alphaDivisor = base.a + step(base.a, 0.0);// Protect against a divide-by-zero blacking out things in the output
    return base * (overlay.a * (base / alphaDivisor) + (2.0 * overlay * (1.0 - (base / alphaDivisor)))) + overlay * (1.0 - base.a) + base * (1.0 - overlay.a);
}

vec4 WhiteBalanceCorrection(vec4 in_texel, float temperature, float tint) {

    // Values from Dan
    vec3 warmFilter = vec3(0.93, 0.54, 0.0);
    mat3 RGBtoYIQ = mat3(vec3(0.299, 0.587, 0.114),
    vec3(0.596, -0.274, -0.322),
    vec3(0.212, -0.523, 0.311));
    mat3 YIQtoRGB = mat3(vec3(1.0, 0.956, 0.621),
    vec3(1.0, -0.272, -0.647),
    vec3(1.0, -1.105, 1.702));
    vec3 yiq = RGBtoYIQ * in_texel.rgb;//adjusting tint
    yiq.b = clamp(yiq.b + tint * 0.5226 * 0.1, -0.5226, 0.5226);
    vec3 rgb = YIQtoRGB * yiq;
    vec3 processed = vec3(
    (rgb.r < 0.5 ? (2.0 * rgb.r * warmFilter.r) : (1.0 - 2.0 * (1.0 - rgb.r) * (1.0 - warmFilter.r))), //adjusting temperature
    (rgb.g < 0.5 ? (2.0 * rgb.g * warmFilter.g) : (1.0 - 2.0 * (1.0 - rgb.g) * (1.0 - warmFilter.g))),
    (rgb.b < 0.5 ? (2.0 * rgb.b * warmFilter.b) : (1.0 - 2.0 * (1.0 - rgb.b) * (1.0 - warmFilter.b))));
    return vec4(mix(rgb, processed, temperature), in_texel.a);
}

// contrast: 0.0 - 4.0
vec4 ContrastCorrection(vec4 in_texel, float contrast) {
    return vec4(((in_texel.rgb - vec3(0.5)) * contrast + vec3(0.5)), in_texel.a);
}

// brightness: 0.0 - 1.0
// 0.0 - everything is black, 1.0 - original color, > 1.0 - brighten color
// Range of brightness value the same as in webkit
// https://developer.mozilla.org/en-US/docs/Web/CSS/filter-function/brightness
vec4 BrightnessCorrection(vec4 in_texel, float brightness) {
    return vec4(in_texel.rgb * brightness, in_texel.a);
}

vec4 rotateHue(vec4 color, float deg) {
    float rad = radians(deg);
    float cosA = cos(rad);
    float sinA = sin(rad);

    vec3 neo = vec3((cosA + (1. - cosA) / 3.),
    ((1. - cosA) / 3. - .57735 * sinA),
    ((1. - cosA) / 3. + .57735 * sinA)
    );

    vec3 result = vec3(color.r * neo.r + color.g * neo.g + color.b * neo.b,
    color.r * neo.b + color.g * neo.r + color.b * neo.g,
    color.r * neo.g + color.g * neo.b + color.b * neo.r);

    return vec4(result, color.a);
}

//----------- OverlayBlend ----------

float OverlayBlend(float base, float blend) {
    return base<0.5?(2.0*base*blend):(1.0-2.0*(1.0-base)*(1.0-blend));
}

vec3 OverlayBlend(vec3 base, vec3 blend) {
    return vec3(OverlayBlend(base.r, blend.r), OverlayBlend(base.g, blend.g), OverlayBlend(base.b, blend.b));
}

vec3 OverlayBlend(vec3 base, vec3 blend, float opacity) {
    return (OverlayBlend(base, blend) * opacity + base * (1.0 - opacity));
}

vec4 OverlayBlend(vec4 base, vec4 blend) {
    vec3 res = OverlayBlend(base.rgb, blend.rgb, blend.a);
    return vec4(res, 1.0);
}

vec4 ExposureCorrection(vec4 in_texel, float exposure) {
    return vec4((in_texel.rgb * pow(2.0, exposure)), in_texel.a);
}

//----------- END OF OverlayBlend----------

// ---------- LightenBlend --------------
float LightenBlend(float base, float blend) {
    return max(blend, base);
}

vec3 LightenBlend(vec3 base, vec3 blend) {
    return vec3(LightenBlend(base.r, blend.r), LightenBlend(base.g, blend.g), LightenBlend(base.b, blend.b));
}

vec3 LightenBlend(vec3 base, vec3 blend, float opacity) {
    return (LightenBlend(base, blend) * opacity + base * (1.0 - opacity));
}

vec4 LightenBlend(vec4 base, vec4 blend) {
    vec3 res = LightenBlend(base.rgb, blend.rgb, blend.a);
    return vec4(res, base.a);
}

// -------- END OF LightenBlend ---------


