(function (root) {
  var exports = undefined,
      module = undefined,
      require = undefined;
  var define = undefined;
  var self = root,
      window = root,
      global = root,
      globalThis = root;
  (function () {
    (function (global, factory) {
      (global["dcodeIO"] = global["dcodeIO"] || {})["ProtoBuf"] = factory(global["dcodeIO"]["ByteBuffer"]);
    })(window, function (ByteBuffer, isCommonJS) {
      var ProtoBuf = {};
      ProtoBuf.ByteBuffer = ByteBuffer;
      ProtoBuf.Long = ByteBuffer.Long || null;
      ProtoBuf.VERSION = "5.0.1";
      ProtoBuf.WIRE_TYPES = {};
      ProtoBuf.WIRE_TYPES.VARINT = 0;
      ProtoBuf.WIRE_TYPES.BITS64 = 1;
      ProtoBuf.WIRE_TYPES.LDELIM = 2;
      ProtoBuf.WIRE_TYPES.STARTGROUP = 3;
      ProtoBuf.WIRE_TYPES.ENDGROUP = 4;
      ProtoBuf.WIRE_TYPES.BITS32 = 5;
      ProtoBuf.PACKABLE_WIRE_TYPES = [ProtoBuf.WIRE_TYPES.VARINT, ProtoBuf.WIRE_TYPES.BITS64, ProtoBuf.WIRE_TYPES.BITS32];
      ProtoBuf.TYPES = {
        "int32": {
          name: "int32",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: 0
        },
        "uint32": {
          name: "uint32",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: 0
        },
        "sint32": {
          name: "sint32",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: 0
        },
        "int64": {
          name: "int64",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: ProtoBuf.Long ? ProtoBuf.Long.ZERO : undefined
        },
        "uint64": {
          name: "uint64",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: ProtoBuf.Long ? ProtoBuf.Long.UZERO : undefined
        },
        "sint64": {
          name: "sint64",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: ProtoBuf.Long ? ProtoBuf.Long.ZERO : undefined
        },
        "bool": {
          name: "bool",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: false
        },
        "double": {
          name: "double",
          wireType: ProtoBuf.WIRE_TYPES.BITS64,
          defaultValue: 0
        },
        "string": {
          name: "string",
          wireType: ProtoBuf.WIRE_TYPES.LDELIM,
          defaultValue: ""
        },
        "bytes": {
          name: "bytes",
          wireType: ProtoBuf.WIRE_TYPES.LDELIM,
          defaultValue: null
        },
        "fixed32": {
          name: "fixed32",
          wireType: ProtoBuf.WIRE_TYPES.BITS32,
          defaultValue: 0
        },
        "sfixed32": {
          name: "sfixed32",
          wireType: ProtoBuf.WIRE_TYPES.BITS32,
          defaultValue: 0
        },
        "fixed64": {
          name: "fixed64",
          wireType: ProtoBuf.WIRE_TYPES.BITS64,
          defaultValue: ProtoBuf.Long ? ProtoBuf.Long.UZERO : undefined
        },
        "sfixed64": {
          name: "sfixed64",
          wireType: ProtoBuf.WIRE_TYPES.BITS64,
          defaultValue: ProtoBuf.Long ? ProtoBuf.Long.ZERO : undefined
        },
        "float": {
          name: "float",
          wireType: ProtoBuf.WIRE_TYPES.BITS32,
          defaultValue: 0
        },
        "enum": {
          name: "enum",
          wireType: ProtoBuf.WIRE_TYPES.VARINT,
          defaultValue: 0
        },
        "message": {
          name: "message",
          wireType: ProtoBuf.WIRE_TYPES.LDELIM,
          defaultValue: null
        },
        "group": {
          name: "group",
          wireType: ProtoBuf.WIRE_TYPES.STARTGROUP,
          defaultValue: null
        }
      };
      ProtoBuf.MAP_KEY_TYPES = [ProtoBuf.TYPES["int32"], ProtoBuf.TYPES["sint32"], ProtoBuf.TYPES["sfixed32"], ProtoBuf.TYPES["uint32"], ProtoBuf.TYPES["fixed32"], ProtoBuf.TYPES["int64"], ProtoBuf.TYPES["sint64"], ProtoBuf.TYPES["sfixed64"], ProtoBuf.TYPES["uint64"], ProtoBuf.TYPES["fixed64"], ProtoBuf.TYPES["bool"], ProtoBuf.TYPES["string"], ProtoBuf.TYPES["bytes"]];
      ProtoBuf.ID_MIN = 1;
      ProtoBuf.ID_MAX = 536870911;
      ProtoBuf.convertFieldsToCamelCase = false;
      ProtoBuf.populateAccessors = true;
      ProtoBuf.populateDefaults = true;

      ProtoBuf.Util = function () {
        var Util = {};
        Util.IS_NODE = !!(typeof process === "object" && process + "" === "[object process]" && !process["browser"]);

        Util.XHR = function () {
          var XMLHttpFactories = [function () {
            return new XMLHttpRequest();
          }, function () {
            return new ActiveXObject("Msxml2.XMLHTTP");
          }, function () {
            return new ActiveXObject("Msxml3.XMLHTTP");
          }, function () {
            return new ActiveXObject("Microsoft.XMLHTTP");
          }];
          var xhr = null;

          for (var i = 0; i < XMLHttpFactories.length; i++) {
            try {
              xhr = XMLHttpFactories[i]();
            } catch (e) {
              continue;
            }

            break;
          }

          if (!xhr) {
            throw Error("XMLHttpRequest is not supported");
          }

          return xhr;
        };

        Util.fetch = function (path, callback) {
          if (callback && typeof callback != "function") {
            callback = null;
          }

          if (Util.IS_NODE) {
            var fs = require("fs");

            if (callback) {
              fs.readFile(path, function (err, data) {
                if (err) {
                  callback(null);
                } else {
                  callback("" + data);
                }
              });
            } else {
              try {
                return fs.readFileSync(path);
              } catch (e) {
                return null;
              }
            }
          } else {
            var xhr = Util.XHR();
            xhr.open("GET", path, callback ? true : false);
            xhr.setRequestHeader("Accept", "text/plain");

            if (typeof xhr.overrideMimeType === "function") {
              xhr.overrideMimeType("text/plain");
            }

            if (callback) {
              xhr.onreadystatechange = function () {
                if (xhr.readyState != 4) {
                  return;
                }

                if (xhr.status == 200 || xhr.status == 0 && typeof xhr.responseText === "string") {
                  callback(xhr.responseText);
                } else {
                  callback(null);
                }
              };

              if (xhr.readyState == 4) {
                return;
              }

              xhr.send(null);
            } else {
              xhr.send(null);

              if (xhr.status == 200 || xhr.status == 0 && typeof xhr.responseText === "string") {
                return xhr.responseText;
              }

              return null;
            }
          }
        };

        Util.toCamelCase = function (str) {
          return str.replace(/_([a-zA-Z])/g, function ($0, $1) {
            return $1.toUpperCase();
          });
        };

        return Util;
      }();

      ProtoBuf.Lang = {
        DELIM: /[\s\{\}=;:\[\],'"\(\)<>]/g,
        RULE: /^(?:required|optional|repeated|map)$/,
        TYPE: /^(?:double|float|int32|uint32|sint32|int64|uint64|sint64|fixed32|sfixed32|fixed64|sfixed64|bool|string|bytes)$/,
        NAME: /^[a-zA-Z_][a-zA-Z_0-9]*$/,
        TYPEDEF: /^[a-zA-Z][a-zA-Z_0-9]*$/,
        TYPEREF: /^(?:\.?[a-zA-Z_][a-zA-Z_0-9]*)+$/,
        FQTYPEREF: /^(?:\.[a-zA-Z][a-zA-Z_0-9]*)+$/,
        NUMBER: /^-?(?:[1-9][0-9]*|0|0[xX][0-9a-fA-F]+|0[0-7]+|([0-9]*(\.[0-9]*)?([Ee][+-]?[0-9]+)?)|inf|nan)$/,
        NUMBER_DEC: /^(?:[1-9][0-9]*|0)$/,
        NUMBER_HEX: /^0[xX][0-9a-fA-F]+$/,
        NUMBER_OCT: /^0[0-7]+$/,
        NUMBER_FLT: /^([0-9]*(\.[0-9]*)?([Ee][+-]?[0-9]+)?|inf|nan)$/,
        BOOL: /^(?:true|false)$/i,
        ID: /^(?:[1-9][0-9]*|0|0[xX][0-9a-fA-F]+|0[0-7]+)$/,
        NEGID: /^\-?(?:[1-9][0-9]*|0|0[xX][0-9a-fA-F]+|0[0-7]+)$/,
        WHITESPACE: /\s/,
        STRING: /(?:"([^"\\]*(?:\\.[^"\\]*)*)")|(?:'([^'\\]*(?:\\.[^'\\]*)*)')/g,
        STRING_DQ: /(?:"([^"\\]*(?:\\.[^"\\]*)*)")/g,
        STRING_SQ: /(?:'([^'\\]*(?:\\.[^'\\]*)*)')/g
      };

      ProtoBuf.DotProto = function (ProtoBuf, Lang) {
        var DotProto = {};

        var Tokenizer = function (proto) {
          this.source = proto + "";
          this.index = 0;
          this.line = 1;
          this.stack = [];
          this._stringOpen = null;
        };

        var TokenizerPrototype = Tokenizer.prototype;

        TokenizerPrototype._readString = function () {
          var re = this._stringOpen === '"' ? Lang.STRING_DQ : Lang.STRING_SQ;
          re.lastIndex = this.index - 1;
          var match = re.exec(this.source);

          if (!match) {
            throw Error("unterminated string");
          }

          this.index = re.lastIndex;
          this.stack.push(this._stringOpen);
          this._stringOpen = null;
          return match[1];
        };

        TokenizerPrototype.next = function () {
          if (this.stack.length > 0) {
            return this.stack.shift();
          }

          if (this.index >= this.source.length) {
            return null;
          }

          if (this._stringOpen !== null) {
            return this._readString();
          }

          var repeat, prev, next;

          do {
            repeat = false;

            while (Lang.WHITESPACE.test(next = this.source.charAt(this.index))) {
              if (next === "\n") {
                ++this.line;
              }

              if (++this.index === this.source.length) {
                return null;
              }
            }

            if (this.source.charAt(this.index) === "/") {
              ++this.index;

              if (this.source.charAt(this.index) === "/") {
                while (this.source.charAt(++this.index) !== "\n") {
                  if (this.index == this.source.length) {
                    return null;
                  }
                }

                ++this.index;
                ++this.line;
                repeat = true;
              } else {
                if ((next = this.source.charAt(this.index)) === "*") {
                  do {
                    if (next === "\n") {
                      ++this.line;
                    }

                    if (++this.index === this.source.length) {
                      return null;
                    }

                    prev = next;
                    next = this.source.charAt(this.index);
                  } while (prev !== "*" || next !== "/");

                  ++this.index;
                  repeat = true;
                } else {
                  return "/";
                }
              }
            }
          } while (repeat);

          if (this.index === this.source.length) {
            return null;
          }

          var end = this.index;
          Lang.DELIM.lastIndex = 0;
          var delim = Lang.DELIM.test(this.source.charAt(end++));

          if (!delim) {
            while (end < this.source.length && !Lang.DELIM.test(this.source.charAt(end))) {
              ++end;
            }
          }

          var token = this.source.substring(this.index, this.index = end);

          if (token === '"' || token === "'") {
            this._stringOpen = token;
          }

          return token;
        };

        TokenizerPrototype.peek = function () {
          if (this.stack.length === 0) {
            var token = this.next();

            if (token === null) {
              return null;
            }

            this.stack.push(token);
          }

          return this.stack[0];
        };

        TokenizerPrototype.skip = function (expected) {
          var actual = this.next();

          if (actual !== expected) {
            throw Error("illegal '" + actual + "', '" + expected + "' expected");
          }
        };

        TokenizerPrototype.omit = function (expected) {
          if (this.peek() === expected) {
            this.next();
            return true;
          }

          return false;
        };

        TokenizerPrototype.toString = function () {
          return "Tokenizer (" + this.index + "/" + this.source.length + " at line " + this.line + ")";
        };

        DotProto.Tokenizer = Tokenizer;

        var Parser = function (source) {
          this.tn = new Tokenizer(source);
          this.proto3 = false;
        };

        var ParserPrototype = Parser.prototype;

        ParserPrototype.parse = function () {
          var topLevel = {
            "name": "[ROOT]",
            "package": null,
            "messages": [],
            "enums": [],
            "imports": [],
            "options": {},
            "services": []
          };
          var token,
              head = true,
              weak;

          try {
            while (token = this.tn.next()) {
              switch (token) {
                case "package":
                  if (!head || topLevel["package"] !== null) {
                    throw Error("unexpected 'package'");
                  }

                  token = this.tn.next();

                  if (!Lang.TYPEREF.test(token)) {
                    throw Error("illegal package name: " + token);
                  }

                  this.tn.skip(";");
                  topLevel["package"] = token;
                  break;

                case "import":
                  if (!head) {
                    throw Error("unexpected 'import'");
                  }

                  token = this.tn.peek();

                  if (token === "public" || (weak = token === "weak")) {
                    this.tn.next();
                  }

                  token = this._readString();
                  this.tn.skip(";");

                  if (!weak) {
                    topLevel["imports"].push(token);
                  }

                  break;

                case "syntax":
                  if (!head) {
                    throw Error("unexpected 'syntax'");
                  }

                  this.tn.skip("=");

                  if ((topLevel["syntax"] = this._readString()) === "proto3") {
                    this.proto3 = true;
                  }

                  this.tn.skip(";");
                  break;

                case "message":
                  this._parseMessage(topLevel, null);

                  head = false;
                  break;

                case "enum":
                  this._parseEnum(topLevel);

                  head = false;
                  break;

                case "option":
                  this._parseOption(topLevel);

                  break;

                case "service":
                  this._parseService(topLevel);

                  break;

                case "extend":
                  this._parseExtend(topLevel);

                  break;

                default:
                  throw Error("unexpected '" + token + "'");
              }
            }
          } catch (e) {
            e.message = "Parse error at line " + this.tn.line + ": " + this.tn.source + e.message;
            throw e;
          }

          delete topLevel["name"];
          return topLevel;
        };

        Parser.parse = function (source) {
          return new Parser(source).parse();
        };

        function mkId(value, mayBeNegative) {
          var id = -1,
              sign = 1;

          if (value.charAt(0) == "-") {
            sign = -1;
            value = value.substring(1);
          }

          if (Lang.NUMBER_DEC.test(value)) {
            id = parseInt(value);
          } else {
            if (Lang.NUMBER_HEX.test(value)) {
              id = parseInt(value.substring(2), 16);
            } else {
              if (Lang.NUMBER_OCT.test(value)) {
                id = parseInt(value.substring(1), 8);
              } else {
                throw Error("illegal id value: " + (sign < 0 ? "-" : "") + value);
              }
            }
          }

          id = sign * id | 0;

          if (!mayBeNegative && id < 0) {
            throw Error("illegal id value: " + (sign < 0 ? "-" : "") + value);
          }

          return id;
        }

        function mkNumber(val) {
          var sign = 1;

          if (val.charAt(0) == "-") {
            sign = -1;
            val = val.substring(1);
          }

          if (Lang.NUMBER_DEC.test(val)) {
            return sign * parseInt(val, 10);
          } else {
            if (Lang.NUMBER_HEX.test(val)) {
              return sign * parseInt(val.substring(2), 16);
            } else {
              if (Lang.NUMBER_OCT.test(val)) {
                return sign * parseInt(val.substring(1), 8);
              } else {
                if (val === "inf") {
                  return sign * Infinity;
                } else {
                  if (val === "nan") {
                    return NaN;
                  } else {
                    if (Lang.NUMBER_FLT.test(val)) {
                      return sign * parseFloat(val);
                    }
                  }
                }
              }
            }
          }

          throw Error("illegal number value: " + (sign < 0 ? "-" : "") + val);
        }

        ParserPrototype._readString = function () {
          var value = "",
              token,
              delim;

          do {
            delim = this.tn.next();

            if (delim !== "'" && delim !== '"') {
              throw Error("illegal string delimiter: " + delim);
            }

            value += this.tn.next();
            this.tn.skip(delim);
            token = this.tn.peek();
          } while (token === '"' || token === '"');

          return value;
        };

        ParserPrototype._readValue = function (mayBeTypeRef) {
          var token = this.tn.peek(),
              value;

          if (token === '"' || token === "'") {
            return this._readString();
          }

          this.tn.next();

          if (Lang.NUMBER.test(token)) {
            return mkNumber(token);
          }

          if (Lang.BOOL.test(token)) {
            return token.toLowerCase() === "true";
          }

          if (mayBeTypeRef && Lang.TYPEREF.test(token)) {
            return token;
          }

          throw Error("illegal value: " + token);
        };

        ParserPrototype._parseOption = function (parent, isList) {
          var token = this.tn.next(),
              custom = false;

          if (token === "(") {
            custom = true;
            token = this.tn.next();
          }

          if (!Lang.TYPEREF.test(token)) {
            throw Error("illegal option name: " + token);
          }

          var name = token;

          if (custom) {
            this.tn.skip(")");
            name = "(" + name + ")";
            token = this.tn.peek();

            if (Lang.FQTYPEREF.test(token)) {
              name += token;
              this.tn.next();
            }
          }

          this.tn.skip("=");

          this._parseOptionValue(parent, name);

          if (!isList) {
            this.tn.skip(";");
          }
        };

        function setOption(options, name, value) {
          if (typeof options[name] === "undefined") {
            options[name] = value;
          } else {
            if (!Array.isArray(options[name])) {
              options[name] = [options[name]];
            }

            options[name].push(value);
          }
        }

        ParserPrototype._parseOptionValue = function (parent, name) {
          var token = this.tn.peek();

          if (token !== "{") {
            setOption(parent["options"], name, this._readValue(true));
          } else {
            this.tn.skip("{");

            while ((token = this.tn.next()) !== "}") {
              if (!Lang.NAME.test(token)) {
                throw Error("illegal option name: " + name + "." + token);
              }

              if (this.tn.omit(":")) {
                setOption(parent["options"], name + "." + token, this._readValue(true));
              } else {
                this._parseOptionValue(parent, name + "." + token);
              }
            }
          }
        };

        ParserPrototype._parseService = function (parent) {
          var token = this.tn.next();

          if (!Lang.NAME.test(token)) {
            throw Error("illegal service name at line " + this.tn.line + ": " + token);
          }

          var name = token;
          var svc = {
            "name": name,
            "rpc": {},
            "options": {}
          };
          this.tn.skip("{");

          while ((token = this.tn.next()) !== "}") {
            if (token === "option") {
              this._parseOption(svc);
            } else {
              if (token === "rpc") {
                this._parseServiceRPC(svc);
              } else {
                throw Error("illegal service token: " + token);
              }
            }
          }

          this.tn.omit(";");
          parent["services"].push(svc);
        };

        ParserPrototype._parseServiceRPC = function (svc) {
          var type = "rpc",
              token = this.tn.next();

          if (!Lang.NAME.test(token)) {
            throw Error("illegal rpc service method name: " + token);
          }

          var name = token;
          var method = {
            "request": null,
            "response": null,
            "request_stream": false,
            "response_stream": false,
            "options": {}
          };
          this.tn.skip("(");
          token = this.tn.next();

          if (token.toLowerCase() === "stream") {
            method["request_stream"] = true;
            token = this.tn.next();
          }

          if (!Lang.TYPEREF.test(token)) {
            throw Error("illegal rpc service request type: " + token);
          }

          method["request"] = token;
          this.tn.skip(")");
          token = this.tn.next();

          if (token.toLowerCase() !== "returns") {
            throw Error("illegal rpc service request type delimiter: " + token);
          }

          this.tn.skip("(");
          token = this.tn.next();

          if (token.toLowerCase() === "stream") {
            method["response_stream"] = true;
            token = this.tn.next();
          }

          method["response"] = token;
          this.tn.skip(")");
          token = this.tn.peek();

          if (token === "{") {
            this.tn.next();

            while ((token = this.tn.next()) !== "}") {
              if (token === "option") {
                this._parseOption(method);
              } else {
                throw Error("illegal rpc service token: " + token);
              }
            }

            this.tn.omit(";");
          } else {
            this.tn.skip(";");
          }

          if (typeof svc[type] === "undefined") {
            svc[type] = {};
          }

          svc[type][name] = method;
        };

        ParserPrototype._parseMessage = function (parent, fld) {
          var isGroup = !!fld,
              token = this.tn.next();
          var msg = {
            "name": "",
            "fields": [],
            "enums": [],
            "messages": [],
            "options": {},
            "services": [],
            "oneofs": {}
          };

          if (!Lang.NAME.test(token)) {
            throw Error("illegal " + (isGroup ? "group" : "message") + " name: " + token);
          }

          msg["name"] = token;

          if (isGroup) {
            this.tn.skip("=");
            fld["id"] = mkId(this.tn.next());
            msg["isGroup"] = true;
          }

          token = this.tn.peek();

          if (token === "[" && fld) {
            this._parseFieldOptions(fld);
          }

          this.tn.skip("{");

          while ((token = this.tn.next()) !== "}") {
            if (Lang.RULE.test(token)) {
              this._parseMessageField(msg, token);
            } else {
              if (token === "oneof") {
                this._parseMessageOneOf(msg);
              } else {
                if (token === "enum") {
                  this._parseEnum(msg);
                } else {
                  if (token === "message") {
                    this._parseMessage(msg);
                  } else {
                    if (token === "option") {
                      this._parseOption(msg);
                    } else {
                      if (token === "service") {
                        this._parseService(msg);
                      } else {
                        if (token === "extensions") {
                          if (msg.hasOwnProperty("extensions")) {
                            msg["extensions"] = msg["extensions"].concat(this._parseExtensionRanges());
                          } else {
                            msg["extensions"] = this._parseExtensionRanges();
                          }
                        } else {
                          if (token === "reserved") {
                            this._parseIgnored();
                          } else {
                            if (token === "extend") {
                              this._parseExtend(msg);
                            } else {
                              if (Lang.TYPEREF.test(token)) {
                                if (!this.proto3) {
                                  throw Error("illegal field rule: " + token);
                                }

                                this._parseMessageField(msg, "optional", token);
                              } else {
                                throw Error("illegal message token: " + token);
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }

          this.tn.omit(";");
          parent["messages"].push(msg);
          return msg;
        };

        ParserPrototype._parseIgnored = function () {
          while (this.tn.peek() !== ";") {
            this.tn.next();
          }

          this.tn.skip(";");
        };

        ParserPrototype._parseMessageField = function (msg, rule, type) {
          if (!Lang.RULE.test(rule)) {
            throw Error("illegal message field rule: " + rule);
          }

          var fld = {
            "rule": rule,
            "type": "",
            "name": "",
            "options": {},
            "id": 0
          };
          var token;

          if (rule === "map") {
            if (type) {
              throw Error("illegal type: " + type);
            }

            this.tn.skip("<");
            token = this.tn.next();

            if (!Lang.TYPE.test(token) && !Lang.TYPEREF.test(token)) {
              throw Error("illegal message field type: " + token);
            }

            fld["keytype"] = token;
            this.tn.skip(",");
            token = this.tn.next();

            if (!Lang.TYPE.test(token) && !Lang.TYPEREF.test(token)) {
              throw Error("illegal message field: " + token);
            }

            fld["type"] = token;
            this.tn.skip(">");
            token = this.tn.next();

            if (!Lang.NAME.test(token)) {
              throw Error("illegal message field name: " + token);
            }

            fld["name"] = token;
            this.tn.skip("=");
            fld["id"] = mkId(this.tn.next());
            token = this.tn.peek();

            if (token === "[") {
              this._parseFieldOptions(fld);
            }

            this.tn.skip(";");
          } else {
            type = typeof type !== "undefined" ? type : this.tn.next();

            if (type === "group") {
              var grp = this._parseMessage(msg, fld);

              if (!/^[A-Z]/.test(grp["name"])) {
                throw Error("illegal group name: " + grp["name"]);
              }

              fld["type"] = grp["name"];
              fld["name"] = grp["name"].toLowerCase();
              this.tn.omit(";");
            } else {
              if (!Lang.TYPE.test(type) && !Lang.TYPEREF.test(type)) {
                throw Error("illegal message field type: " + type);
              }

              fld["type"] = type;
              token = this.tn.next();

              if (!Lang.NAME.test(token)) {
                throw Error("illegal message field name: " + token);
              }

              fld["name"] = token;
              this.tn.skip("=");
              fld["id"] = mkId(this.tn.next());
              token = this.tn.peek();

              if (token === "[") {
                this._parseFieldOptions(fld);
              }

              this.tn.skip(";");
            }
          }

          msg["fields"].push(fld);
          return fld;
        };

        ParserPrototype._parseMessageOneOf = function (msg) {
          var token = this.tn.next();

          if (!Lang.NAME.test(token)) {
            throw Error("illegal oneof name: " + token);
          }

          var name = token,
              fld;
          var fields = [];
          this.tn.skip("{");

          while ((token = this.tn.next()) !== "}") {
            fld = this._parseMessageField(msg, "optional", token);
            fld["oneof"] = name;
            fields.push(fld["id"]);
          }

          this.tn.omit(";");
          msg["oneofs"][name] = fields;
        };

        ParserPrototype._parseFieldOptions = function (fld) {
          this.tn.skip("[");
          var token,
              first = true;

          while ((token = this.tn.peek()) !== "]") {
            if (!first) {
              this.tn.skip(",");
            }

            this._parseOption(fld, true);

            first = false;
          }

          this.tn.next();
        };

        ParserPrototype._parseEnum = function (msg) {
          var enm = {
            "name": "",
            "values": [],
            "options": {}
          };
          var token = this.tn.next();

          if (!Lang.NAME.test(token)) {
            throw Error("illegal name: " + token);
          }

          enm["name"] = token;
          this.tn.skip("{");

          while ((token = this.tn.next()) !== "}") {
            if (token === "option") {
              this._parseOption(enm);
            } else {
              if (!Lang.NAME.test(token)) {
                throw Error("illegal name: " + token);
              }

              this.tn.skip("=");
              var val = {
                "name": token,
                "id": mkId(this.tn.next(), true)
              };
              token = this.tn.peek();

              if (token === "[") {
                this._parseFieldOptions({
                  "options": {}
                });
              }

              this.tn.skip(";");
              enm["values"].push(val);
            }
          }

          this.tn.omit(";");
          msg["enums"].push(enm);
        };

        ParserPrototype._parseExtensionRanges = function () {
          var ranges = [];
          var token, range, value;

          do {
            range = [];

            while (true) {
              token = this.tn.next();

              switch (token) {
                case "min":
                  value = ProtoBuf.ID_MIN;
                  break;

                case "max":
                  value = ProtoBuf.ID_MAX;
                  break;

                default:
                  value = mkNumber(token);
                  break;
              }

              range.push(value);

              if (range.length === 2) {
                break;
              }

              if (this.tn.peek() !== "to") {
                range.push(value);
                break;
              }

              this.tn.next();
            }

            ranges.push(range);
          } while (this.tn.omit(","));

          this.tn.skip(";");
          return ranges;
        };

        ParserPrototype._parseExtend = function (parent) {
          var token = this.tn.next();

          if (!Lang.TYPEREF.test(token)) {
            throw Error("illegal extend reference: " + token);
          }

          var ext = {
            "ref": token,
            "fields": []
          };
          this.tn.skip("{");

          while ((token = this.tn.next()) !== "}") {
            if (Lang.RULE.test(token)) {
              this._parseMessageField(ext, token);
            } else {
              if (Lang.TYPEREF.test(token)) {
                if (!this.proto3) {
                  throw Error("illegal field rule: " + token);
                }

                this._parseMessageField(ext, "optional", token);
              } else {
                throw Error("illegal extend token: " + token);
              }
            }
          }

          this.tn.omit(";");
          parent["messages"].push(ext);
          return ext;
        };

        ParserPrototype.toString = function () {
          return "Parser at line " + this.tn.line;
        };

        DotProto.Parser = Parser;
        return DotProto;
      }(ProtoBuf, ProtoBuf.Lang);

      ProtoBuf.Reflect = function (ProtoBuf) {
        var Reflect = {};

        var T = function (builder, parent, name) {
          this.builder = builder;
          this.parent = parent;
          this.name = name;
          this.className;
        };

        var TPrototype = T.prototype;

        TPrototype.fqn = function () {
          var name = this.name,
              ptr = this;

          do {
            ptr = ptr.parent;

            if (ptr == null) {
              break;
            }

            name = ptr.name + "." + name;
          } while (true);

          return name;
        };

        TPrototype.toString = function (includeClass) {
          return (includeClass ? this.className + " " : "") + this.fqn();
        };

        TPrototype.build = function () {
          throw Error(this.toString(true) + " cannot be built directly");
        };

        Reflect.T = T;

        var Namespace = function (builder, parent, name, options, syntax) {
          T.call(this, builder, parent, name);
          this.className = "Namespace";
          this.children = [];
          this.options = options || {};
          this.syntax = syntax || "proto2";
        };

        var NamespacePrototype = Namespace.prototype = Object.create(T.prototype);

        NamespacePrototype.getChildren = function (type) {
          type = type || null;

          if (type == null) {
            return this.children.slice();
          }

          var children = [];

          for (var i = 0, k = this.children.length; i < k; ++i) {
            if (this.children[i] instanceof type) {
              children.push(this.children[i]);
            }
          }

          return children;
        };

        NamespacePrototype.addChild = function (child) {
          var other;

          if (other = this.getChild(child.name)) {
            if (other instanceof Message.Field && other.name !== other.originalName && this.getChild(other.originalName) === null) {
              other.name = other.originalName;
            } else {
              if (child instanceof Message.Field && child.name !== child.originalName && this.getChild(child.originalName) === null) {
                child.name = child.originalName;
              } else {
                throw Error("Duplicate name in namespace " + this.toString(true) + ": " + child.name);
              }
            }
          }

          this.children.push(child);
        };

        NamespacePrototype.getChild = function (nameOrId) {
          var key = typeof nameOrId === "number" ? "id" : "name";

          for (var i = 0, k = this.children.length; i < k; ++i) {
            if (this.children[i][key] === nameOrId) {
              return this.children[i];
            }
          }

          return null;
        };

        NamespacePrototype.resolve = function (qn, excludeNonNamespace) {
          var part = typeof qn === "string" ? qn.split(".") : qn,
              ptr = this,
              i = 0;

          if (part[i] === "") {
            while (ptr.parent !== null) {
              ptr = ptr.parent;
            }

            i++;
          }

          var child;

          do {
            do {
              if (!(ptr instanceof Reflect.Namespace)) {
                ptr = null;
                break;
              }

              child = ptr.getChild(part[i]);

              if (!child || !(child instanceof Reflect.T) || excludeNonNamespace && !(child instanceof Reflect.Namespace)) {
                ptr = null;
                break;
              }

              ptr = child;
              i++;
            } while (i < part.length);

            if (ptr != null) {
              break;
            }

            if (this.parent !== null) {
              return this.parent.resolve(qn, excludeNonNamespace);
            }
          } while (ptr != null);

          return ptr;
        };

        NamespacePrototype.qn = function (t) {
          var part = [],
              ptr = t;

          do {
            part.unshift(ptr.name);
            ptr = ptr.parent;
          } while (ptr !== null);

          for (var len = 1; len <= part.length; len++) {
            var qn = part.slice(part.length - len);

            if (t === this.resolve(qn, t instanceof Reflect.Namespace)) {
              return qn.join(".");
            }
          }

          return t.fqn();
        };

        NamespacePrototype.build = function () {
          var ns = {};
          var children = this.children;

          for (var i = 0, k = children.length, child; i < k; ++i) {
            child = children[i];

            if (child instanceof Namespace) {
              ns[child.name] = child.build();
            }
          }

          if (Object.defineProperty) {
            Object.defineProperty(ns, "$options", {
              "value": this.buildOpt()
            });
          }

          return ns;
        };

        NamespacePrototype.buildOpt = function () {
          var opt = {},
              keys = Object.keys(this.options);

          for (var i = 0, k = keys.length; i < k; ++i) {
            var key = keys[i],
                val = this.options[keys[i]];
            opt[key] = val;
          }

          return opt;
        };

        NamespacePrototype.getOption = function (name) {
          if (typeof name === "undefined") {
            return this.options;
          }

          return typeof this.options[name] !== "undefined" ? this.options[name] : null;
        };

        Reflect.Namespace = Namespace;

        var Element = function (type, resolvedType, isMapKey, syntax, name) {
          this.type = type;
          this.resolvedType = resolvedType;
          this.isMapKey = isMapKey;
          this.syntax = syntax;
          this.name = name;

          if (isMapKey && ProtoBuf.MAP_KEY_TYPES.indexOf(type) < 0) {
            throw Error("Invalid map key type: " + type.name);
          }
        };

        var ElementPrototype = Element.prototype;

        function mkDefault(type) {
          if (typeof type === "string") {
            type = ProtoBuf.TYPES[type];
          }

          if (typeof type.defaultValue === "undefined") {
            throw Error("default value for type " + type.name + " is not supported");
          }

          if (type == ProtoBuf.TYPES["bytes"]) {
            return new ByteBuffer(0);
          }

          return type.defaultValue;
        }

        Element.defaultFieldValue = mkDefault;

        function mkLong(value, unsigned) {
          if (value && typeof value.low === "number" && typeof value.high === "number" && typeof value.unsigned === "boolean" && value.low === value.low && value.high === value.high) {
            return new ProtoBuf.Long(value.low, value.high, typeof unsigned === "undefined" ? value.unsigned : unsigned);
          }

          if (typeof value === "string") {
            return ProtoBuf.Long.fromString(value, unsigned || false, 10);
          }

          if (typeof value === "number") {
            return ProtoBuf.Long.fromNumber(value, unsigned || false);
          }

          throw Error("not convertible to Long");
        }

        ElementPrototype.toString = function () {
          return (this.name || "") + (this.isMapKey ? "map" : "value") + " element";
        };

        ElementPrototype.verifyValue = function (value) {
          var self = this;

          function fail(val, msg) {
            throw Error("Illegal value for " + self.toString(true) + " of type " + self.type.name + ": " + val + " (" + msg + ")");
          }

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
            case ProtoBuf.TYPES["sint32"]:
            case ProtoBuf.TYPES["sfixed32"]:
              if (typeof value !== "number" || value === value && value % 1 !== 0) {
                fail(typeof value, "not an integer");
              }

              return value > 4294967295 ? value | 0 : value;

            case ProtoBuf.TYPES["uint32"]:
            case ProtoBuf.TYPES["fixed32"]:
              if (typeof value !== "number" || value === value && value % 1 !== 0) {
                fail(typeof value, "not an integer");
              }

              return value < 0 ? value >>> 0 : value;

            case ProtoBuf.TYPES["int64"]:
            case ProtoBuf.TYPES["sint64"]:
            case ProtoBuf.TYPES["sfixed64"]:
              if (ProtoBuf.Long) {
                try {
                  return mkLong(value, false);
                } catch (e) {
                  fail(typeof value, e.message);
                }
              } else {
                fail(typeof value, "requires Long.js");
              }

            case ProtoBuf.TYPES["uint64"]:
            case ProtoBuf.TYPES["fixed64"]:
              if (ProtoBuf.Long) {
                try {
                  return mkLong(value, true);
                } catch (e) {
                  fail(typeof value, e.message);
                }
              } else {
                fail(typeof value, "requires Long.js");
              }

            case ProtoBuf.TYPES["bool"]:
              if (typeof value !== "boolean") {
                fail(typeof value, "not a boolean");
              }

              return value;

            case ProtoBuf.TYPES["float"]:
            case ProtoBuf.TYPES["double"]:
              if (typeof value !== "number") {
                fail(typeof value, "not a number");
              }

              return value;

            case ProtoBuf.TYPES["string"]:
              if (typeof value !== "string" && !(value && value instanceof String)) {
                fail(typeof value, "not a string");
              }

              return "" + value;

            case ProtoBuf.TYPES["bytes"]:
              if (ByteBuffer.isByteBuffer(value)) {
                return value;
              }

              return ByteBuffer.wrap(value, "base64");

            case ProtoBuf.TYPES["enum"]:
              var values = this.resolvedType.getChildren(ProtoBuf.Reflect.Enum.Value);

              for (i = 0; i < values.length; i++) {
                if (values[i].name == value) {
                  return values[i].id;
                } else {
                  if (values[i].id == value) {
                    return values[i].id;
                  }
                }
              }

              if (this.syntax === "proto3") {
                if (typeof value !== "number" || value === value && value % 1 !== 0) {
                  fail(typeof value, "not an integer");
                }

                if (value > 4294967295 || value < 0) {
                  fail(typeof value, "not in range for uint32");
                }

                return value;
              } else {
                fail(value, "not a valid enum value");
              }

            case ProtoBuf.TYPES["group"]:
            case ProtoBuf.TYPES["message"]:
              if (!value || typeof value !== "object") {
                fail(typeof value, "object expected");
              }

              if (value instanceof this.resolvedType.clazz) {
                return value;
              }

              if (value instanceof ProtoBuf.Builder.Message) {
                var obj = {};

                for (var i in value) {
                  if (value.hasOwnProperty(i)) {
                    obj[i] = value[i];
                  }
                }

                value = obj;
              }

              return new this.resolvedType.clazz(value);
          }

          throw Error("[INTERNAL] Illegal value for " + this.toString(true) + ": " + value + " (undefined type " + this.type + ")");
        };

        ElementPrototype.calculateLength = function (id, value) {
          if (value === null) {
            return 0;
          }

          var n;

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
              return value < 0 ? ByteBuffer.calculateVarint64(value) : ByteBuffer.calculateVarint32(value);

            case ProtoBuf.TYPES["uint32"]:
              return ByteBuffer.calculateVarint32(value);

            case ProtoBuf.TYPES["sint32"]:
              return ByteBuffer.calculateVarint32(ByteBuffer.zigZagEncode32(value));

            case ProtoBuf.TYPES["fixed32"]:
            case ProtoBuf.TYPES["sfixed32"]:
            case ProtoBuf.TYPES["float"]:
              return 4;

            case ProtoBuf.TYPES["int64"]:
            case ProtoBuf.TYPES["uint64"]:
              return ByteBuffer.calculateVarint64(value);

            case ProtoBuf.TYPES["sint64"]:
              return ByteBuffer.calculateVarint64(ByteBuffer.zigZagEncode64(value));

            case ProtoBuf.TYPES["fixed64"]:
            case ProtoBuf.TYPES["sfixed64"]:
              return 8;

            case ProtoBuf.TYPES["bool"]:
              return 1;

            case ProtoBuf.TYPES["enum"]:
              return ByteBuffer.calculateVarint32(value);

            case ProtoBuf.TYPES["double"]:
              return 8;

            case ProtoBuf.TYPES["string"]:
              n = ByteBuffer.calculateUTF8Bytes(value);
              return ByteBuffer.calculateVarint32(n) + n;

            case ProtoBuf.TYPES["bytes"]:
              if (value.remaining() < 0) {
                throw Error("Illegal value for " + this.toString(true) + ": " + value.remaining() + " bytes remaining");
              }

              return ByteBuffer.calculateVarint32(value.remaining()) + value.remaining();

            case ProtoBuf.TYPES["message"]:
              n = this.resolvedType.calculate(value);
              return ByteBuffer.calculateVarint32(n) + n;

            case ProtoBuf.TYPES["group"]:
              n = this.resolvedType.calculate(value);
              return n + ByteBuffer.calculateVarint32(id << 3 | ProtoBuf.WIRE_TYPES.ENDGROUP);
          }

          throw Error("[INTERNAL] Illegal value to encode in " + this.toString(true) + ": " + value + " (unknown type)");
        };

        ElementPrototype.encodeValue = function (id, value, buffer) {
          if (value === null) {
            return buffer;
          }

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
              if (value < 0) {
                buffer.writeVarint64(value);
              } else {
                buffer.writeVarint32(value);
              }

              break;

            case ProtoBuf.TYPES["uint32"]:
              buffer.writeVarint32(value);
              break;

            case ProtoBuf.TYPES["sint32"]:
              buffer.writeVarint32ZigZag(value);
              break;

            case ProtoBuf.TYPES["fixed32"]:
              buffer.writeUint32(value);
              break;

            case ProtoBuf.TYPES["sfixed32"]:
              buffer.writeInt32(value);
              break;

            case ProtoBuf.TYPES["int64"]:
            case ProtoBuf.TYPES["uint64"]:
              buffer.writeVarint64(value);
              break;

            case ProtoBuf.TYPES["sint64"]:
              buffer.writeVarint64ZigZag(value);
              break;

            case ProtoBuf.TYPES["fixed64"]:
              buffer.writeUint64(value);
              break;

            case ProtoBuf.TYPES["sfixed64"]:
              buffer.writeInt64(value);
              break;

            case ProtoBuf.TYPES["bool"]:
              if (typeof value === "string") {
                buffer.writeVarint32(value.toLowerCase() === "false" ? 0 : !!value);
              } else {
                buffer.writeVarint32(value ? 1 : 0);
              }

              break;

            case ProtoBuf.TYPES["enum"]:
              buffer.writeVarint32(value);
              break;

            case ProtoBuf.TYPES["float"]:
              buffer.writeFloat32(value);
              break;

            case ProtoBuf.TYPES["double"]:
              buffer.writeFloat64(value);
              break;

            case ProtoBuf.TYPES["string"]:
              buffer.writeVString(value);
              break;

            case ProtoBuf.TYPES["bytes"]:
              if (value.remaining() < 0) {
                throw Error("Illegal value for " + this.toString(true) + ": " + value.remaining() + " bytes remaining");
              }

              var prevOffset = value.offset;
              buffer.writeVarint32(value.remaining());
              buffer.append(value);
              value.offset = prevOffset;
              break;

            case ProtoBuf.TYPES["message"]:
              var bb = new ByteBuffer().LE();
              this.resolvedType.encode(value, bb);
              buffer.writeVarint32(bb.offset);
              buffer.append(bb.flip());
              break;

            case ProtoBuf.TYPES["group"]:
              this.resolvedType.encode(value, buffer);
              buffer.writeVarint32(id << 3 | ProtoBuf.WIRE_TYPES.ENDGROUP);
              break;

            default:
              throw Error("[INTERNAL] Illegal value to encode in " + this.toString(true) + ": " + value + " (unknown type)");
          }

          return buffer;
        };

        ElementPrototype.decode = function (buffer, wireType, id) {
          if (wireType != this.type.wireType) {
            throw Error("Unexpected wire type for element");
          }

          var value, nBytes;

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
              return buffer.readVarint32() | 0;

            case ProtoBuf.TYPES["uint32"]:
              return buffer.readVarint32() >>> 0;

            case ProtoBuf.TYPES["sint32"]:
              return buffer.readVarint32ZigZag() | 0;

            case ProtoBuf.TYPES["fixed32"]:
              return buffer.readUint32() >>> 0;

            case ProtoBuf.TYPES["sfixed32"]:
              return buffer.readInt32() | 0;

            case ProtoBuf.TYPES["int64"]:
              return buffer.readVarint64();

            case ProtoBuf.TYPES["uint64"]:
              return buffer.readVarint64().toUnsigned();

            case ProtoBuf.TYPES["sint64"]:
              return buffer.readVarint64ZigZag().toNumber();

            case ProtoBuf.TYPES["fixed64"]:
              return buffer.readUint64();

            case ProtoBuf.TYPES["sfixed64"]:
              return buffer.readInt64();

            case ProtoBuf.TYPES["bool"]:
              return !!buffer.readVarint32();

            case ProtoBuf.TYPES["enum"]:
              return buffer.readVarint32();

            case ProtoBuf.TYPES["float"]:
              return buffer.readFloat();

            case ProtoBuf.TYPES["double"]:
              return buffer.readDouble();

            case ProtoBuf.TYPES["string"]:
              return buffer.readVString();

            case ProtoBuf.TYPES["bytes"]:
              nBytes = buffer.readVarint32();

              if (buffer.remaining() < nBytes) {
                throw Error("Illegal number of bytes for " + this.toString(true) + ": " + nBytes + " required but got only " + buffer.remaining());
              }

              value = buffer.clone();
              value.limit = value.offset + nBytes;
              buffer.offset += nBytes;
              return value;

            case ProtoBuf.TYPES["message"]:
              nBytes = buffer.readVarint32();
              return this.resolvedType.decode(buffer, nBytes);

            case ProtoBuf.TYPES["group"]:
              return this.resolvedType.decode(buffer, -1, id);
          }

          throw Error("[INTERNAL] Illegal decode type");
        };

        ElementPrototype.valueFromString = function (str) {
          if (!this.isMapKey) {
            throw Error("valueFromString() called on non-map-key element");
          }

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
            case ProtoBuf.TYPES["sint32"]:
            case ProtoBuf.TYPES["sfixed32"]:
            case ProtoBuf.TYPES["uint32"]:
            case ProtoBuf.TYPES["fixed32"]:
              return this.verifyValue(parseInt(str));

            case ProtoBuf.TYPES["int64"]:
            case ProtoBuf.TYPES["sint64"]:
            case ProtoBuf.TYPES["sfixed64"]:
            case ProtoBuf.TYPES["uint64"]:
            case ProtoBuf.TYPES["fixed64"]:
              return this.verifyValue(str);

            case ProtoBuf.TYPES["bool"]:
              return str === "true";

            case ProtoBuf.TYPES["string"]:
              return this.verifyValue(str);

            case ProtoBuf.TYPES["bytes"]:
              return ByteBuffer.fromBinary(str);
          }
        };

        ElementPrototype.valueToString = function (value) {
          if (!this.isMapKey) {
            throw Error("valueToString() called on non-map-key element");
          }

          if (this.type === ProtoBuf.TYPES["bytes"]) {
            return value.toString("binary");
          } else {
            return value.toString();
          }
        };

        Reflect.Element = Element;

        var Message = function (builder, parent, name, options, isGroup, syntax) {
          Namespace.call(this, builder, parent, name, options, syntax);
          this.className = "Message";
          this.extensions = undefined;
          this.clazz = null;
          this.isGroup = !!isGroup;
          this._fields = null;
          this._fieldsById = null;
          this._fieldsByName = null;
        };

        var MessagePrototype = Message.prototype = Object.create(Namespace.prototype);

        MessagePrototype.build = function (rebuild) {
          if (this.clazz && !rebuild) {
            return this.clazz;
          }

          var clazz = function (ProtoBuf, T) {
            var fields = T.getChildren(ProtoBuf.Reflect.Message.Field),
                oneofs = T.getChildren(ProtoBuf.Reflect.Message.OneOf);

            var Message = function (values, var_args) {
              ProtoBuf.Builder.Message.call(this);

              for (var i = 0, k = oneofs.length; i < k; ++i) {
                this[oneofs[i].name] = null;
              }

              for (i = 0, k = fields.length; i < k; ++i) {
                var field = fields[i];
                this[field.name] = field.repeated ? [] : field.map ? new ProtoBuf.Map(field) : null;

                if ((field.required || T.syntax === "proto3") && field.defaultValue !== null) {
                  this[field.name] = field.defaultValue;
                }
              }

              if (arguments.length > 0) {
                var value;

                if (arguments.length === 1 && values !== null && typeof values === "object" && (typeof values.encode !== "function" || values instanceof Message) && !Array.isArray(values) && !(values instanceof ProtoBuf.Map) && !ByteBuffer.isByteBuffer(values) && !(values instanceof ArrayBuffer) && !(ProtoBuf.Long && values instanceof ProtoBuf.Long)) {
                  this.$set(values);
                } else {
                  for (i = 0, k = arguments.length; i < k; ++i) {
                    if (typeof (value = arguments[i]) !== "undefined" && fields && fields[i]) {
                      this.$set(fields[i].name, value);
                    }
                  }
                }
              }
            };

            var MessagePrototype = Message.prototype = Object.create(ProtoBuf.Builder.Message.prototype);

            MessagePrototype.add = function (key, value, noAssert) {
              var field = T._fieldsByName[key];

              if (!noAssert) {
                if (!field) {
                  throw Error(this + "#" + key + " is undefined");
                }

                if (!(field instanceof ProtoBuf.Reflect.Message.Field)) {
                  throw Error(this + "#" + key + " is not a field: " + field.toString(true));
                }

                if (!field.repeated) {
                  throw Error(this + "#" + key + " is not a repeated field");
                }

                value = field.verifyValue(value, true);
              }

              if (this[key] === null) {
                this[key] = [];
              }

              this[key].push(value);
              return this;
            };

            MessagePrototype.$add = MessagePrototype.add;

            MessagePrototype.set = function (keyOrObj, value, noAssert) {
              if (keyOrObj && typeof keyOrObj === "object") {
                noAssert = value;

                for (var ikey in keyOrObj) {
                  if (keyOrObj.hasOwnProperty(ikey) && typeof (value = keyOrObj[ikey]) !== "undefined") {
                    this.$set(ikey, value, noAssert);
                  }
                }

                return this;
              }

              var field = T._fieldsByName[keyOrObj];

              if (!noAssert) {
                if (!field) {
                  throw Error(this + "#" + keyOrObj + " is not a field: undefined");
                }

                if (!(field instanceof ProtoBuf.Reflect.Message.Field)) {
                  throw Error(this + "#" + keyOrObj + " is not a field: " + field.toString(true));
                }

                this[field.name] = value = field.verifyValue(value);
              } else {
                this[keyOrObj] = value;
              }

              if (field && field.oneof) {
                var currentField = this[field.oneof.name];

                if (value !== null) {
                  if (currentField !== null && currentField !== field.name) {
                    this[currentField] = null;
                  }

                  this[field.oneof.name] = field.name;
                } else {
                  if (currentField === keyOrObj) {
                    this[field.oneof.name] = null;
                  }
                }
              }

              return this;
            };

            MessagePrototype.$set = MessagePrototype.set;

            MessagePrototype.get = function (key, noAssert) {
              if (noAssert) {
                return this[key];
              }

              var field = T._fieldsByName[key];

              if (!field || !(field instanceof ProtoBuf.Reflect.Message.Field)) {
                throw Error(this + "#" + key + " is not a field: undefined");
              }

              if (!(field instanceof ProtoBuf.Reflect.Message.Field)) {
                throw Error(this + "#" + key + " is not a field: " + field.toString(true));
              }

              return this[field.name];
            };

            MessagePrototype.$get = MessagePrototype.get;

            for (var i = 0; i < fields.length; i++) {
              var field = fields[i];

              if (field instanceof ProtoBuf.Reflect.Message.ExtensionField) {
                continue;
              }

              if (T.builder.options["populateAccessors"]) {
                (function (field) {
                  var Name = field.originalName.replace(/(_[a-zA-Z])/g, function (match) {
                    return match.toUpperCase().replace("_", "");
                  });
                  Name = Name.substring(0, 1).toUpperCase() + Name.substring(1);
                  var name = field.originalName.replace(/([A-Z])/g, function (match) {
                    return "_" + match;
                  });

                  var setter = function (value, noAssert) {
                    this[field.name] = noAssert ? value : field.verifyValue(value);
                    return this;
                  };

                  var getter = function () {
                    return this[field.name];
                  };

                  if (T.getChild("set" + Name) === null) {
                    MessagePrototype["set" + Name] = setter;
                  }

                  if (T.getChild("set_" + name) === null) {
                    MessagePrototype["set_" + name] = setter;
                  }

                  if (T.getChild("get" + Name) === null) {
                    MessagePrototype["get" + Name] = getter;
                  }

                  if (T.getChild("get_" + name) === null) {
                    MessagePrototype["get_" + name] = getter;
                  }
                })(field);
              }
            }

            MessagePrototype.encode = function (buffer, noVerify) {
              if (typeof buffer === "boolean") {
                noVerify = buffer, buffer = undefined;
              }

              var isNew = false;

              if (!buffer) {
                buffer = new ByteBuffer(), isNew = true;
              }

              var le = buffer.littleEndian;

              try {
                T.encode(this, buffer.LE(), noVerify);
                return (isNew ? buffer.flip() : buffer).LE(le);
              } catch (e) {
                buffer.LE(le);
                throw e;
              }
            };

            Message.encode = function (data, buffer, noVerify) {
              return new Message(data).encode(buffer, noVerify);
            };

            MessagePrototype.calculate = function () {
              return T.calculate(this);
            };

            MessagePrototype.encodeDelimited = function (buffer, noVerify) {
              var isNew = false;

              if (!buffer) {
                buffer = new ByteBuffer(), isNew = true;
              }

              var enc = new ByteBuffer().LE();
              T.encode(this, enc, noVerify).flip();
              buffer.writeVarint32(enc.remaining());
              buffer.append(enc);
              return isNew ? buffer.flip() : buffer;
            };

            MessagePrototype.encodeAB = function () {
              try {
                return this.encode().toArrayBuffer();
              } catch (e) {
                if (e["encoded"]) {
                  e["encoded"] = e["encoded"].toArrayBuffer();
                }

                throw e;
              }
            };

            MessagePrototype.toArrayBuffer = MessagePrototype.encodeAB;

            MessagePrototype.encodeNB = function () {
              try {
                return this.encode().toBuffer();
              } catch (e) {
                if (e["encoded"]) {
                  e["encoded"] = e["encoded"].toBuffer();
                }

                throw e;
              }
            };

            MessagePrototype.toBuffer = MessagePrototype.encodeNB;

            MessagePrototype.encode64 = function () {
              try {
                return this.encode().toBase64();
              } catch (e) {
                if (e["encoded"]) {
                  e["encoded"] = e["encoded"].toBase64();
                }

                throw e;
              }
            };

            MessagePrototype.toBase64 = MessagePrototype.encode64;

            MessagePrototype.encodeHex = function () {
              try {
                return this.encode().toHex();
              } catch (e) {
                if (e["encoded"]) {
                  e["encoded"] = e["encoded"].toHex();
                }

                throw e;
              }
            };

            MessagePrototype.toHex = MessagePrototype.encodeHex;

            function cloneRaw(obj, binaryAsBase64, longsAsStrings, resolvedType) {
              if (obj === null || typeof obj !== "object") {
                if (resolvedType && resolvedType instanceof ProtoBuf.Reflect.Enum) {
                  var name = ProtoBuf.Reflect.Enum.getName(resolvedType.object, obj);

                  if (name !== null) {
                    return name;
                  }
                }

                return obj;
              }

              if (ByteBuffer.isByteBuffer(obj)) {
                return binaryAsBase64 ? obj.toBase64() : obj.toBuffer();
              }

              if (ProtoBuf.Long.isLong(obj)) {
                return longsAsStrings ? obj.toString() : ProtoBuf.Long.fromValue(obj);
              }

              var clone;

              if (Array.isArray(obj)) {
                clone = [];
                obj.forEach(function (v, k) {
                  clone[k] = cloneRaw(v, binaryAsBase64, longsAsStrings, resolvedType);
                });
                return clone;
              }

              clone = {};

              if (obj instanceof ProtoBuf.Map) {
                var it = obj.entries();

                for (var e = it.next(); !e.done; e = it.next()) {
                  clone[obj.keyElem.valueToString(e.value[0])] = cloneRaw(e.value[1], binaryAsBase64, longsAsStrings, obj.valueElem.resolvedType);
                }

                return clone;
              }

              var type = obj.$type,
                  field = undefined;

              for (var i in obj) {
                if (obj.hasOwnProperty(i)) {
                  if (type && (field = type.getChild(i))) {
                    clone[i] = cloneRaw(obj[i], binaryAsBase64, longsAsStrings, field.resolvedType);
                  } else {
                    clone[i] = cloneRaw(obj[i], binaryAsBase64, longsAsStrings);
                  }
                }
              }

              return clone;
            }

            MessagePrototype.toRaw = function (binaryAsBase64, longsAsStrings) {
              return cloneRaw(this, !!binaryAsBase64, !!longsAsStrings, this.$type);
            };

            MessagePrototype.encodeJSON = function () {
              return JSON.stringify(cloneRaw(this, true, true, this.$type));
            };

            Message.decode = function (buffer, length, enc) {
              if (typeof length === "string") {
                enc = length, length = -1;
              }

              if (typeof buffer === "string") {
                buffer = ByteBuffer.wrap(buffer, enc ? enc : "base64");
              } else {
                if (!ByteBuffer.isByteBuffer(buffer)) {
                  buffer = ByteBuffer.wrap(buffer);
                }
              }

              var le = buffer.littleEndian;

              try {
                var msg = T.decode(buffer.LE(), length);
                buffer.LE(le);
                return msg;
              } catch (e) {
                buffer.LE(le);
                throw e;
              }
            };

            Message.decodeDelimited = function (buffer, enc) {
              if (typeof buffer === "string") {
                buffer = ByteBuffer.wrap(buffer, enc ? enc : "base64");
              } else {
                if (!ByteBuffer.isByteBuffer(buffer)) {
                  buffer = ByteBuffer.wrap(buffer);
                }
              }

              if (buffer.remaining() < 1) {
                return null;
              }

              var off = buffer.offset,
                  len = buffer.readVarint32();

              if (buffer.remaining() < len) {
                buffer.offset = off;
                return null;
              }

              try {
                var msg = T.decode(buffer.slice(buffer.offset, buffer.offset + len).LE());
                buffer.offset += len;
                return msg;
              } catch (err) {
                buffer.offset += len;
                throw err;
              }
            };

            Message.decode64 = function (str) {
              return Message.decode(str, "base64");
            };

            Message.decodeHex = function (str) {
              return Message.decode(str, "hex");
            };

            Message.decodeJSON = function (str) {
              return new Message(JSON.parse(str));
            };

            MessagePrototype.toString = function () {
              return T.toString();
            };

            var $optionsS;
            var $options;
            var $typeS;
            var $type;

            if (Object.defineProperty) {
              Object.defineProperty(Message, "$options", {
                "value": T.buildOpt()
              }), Object.defineProperty(MessagePrototype, "$options", {
                "value": Message["$options"]
              }), Object.defineProperty(Message, "$type", {
                "value": T
              }), Object.defineProperty(MessagePrototype, "$type", {
                "value": T
              });
            }

            return Message;
          }(ProtoBuf, this);

          this._fields = [];
          this._fieldsById = {};
          this._fieldsByName = {};

          for (var i = 0, k = this.children.length, child; i < k; i++) {
            child = this.children[i];

            if (child instanceof Enum || child instanceof Message || child instanceof Service) {
              if (clazz.hasOwnProperty(child.name)) {
                throw Error("Illegal reflect child of " + this.toString(true) + ": " + child.toString(true) + " cannot override static property '" + child.name + "'");
              }

              clazz[child.name] = child.build();
            } else {
              if (child instanceof Message.Field) {
                child.build(), this._fields.push(child), this._fieldsById[child.id] = child, this._fieldsByName[child.name] = child;
              } else {
                if (!(child instanceof Message.OneOf) && !(child instanceof Extension)) {
                  throw Error("Illegal reflect child of " + this.toString(true) + ": " + this.children[i].toString(true));
                }
              }
            }
          }

          return this.clazz = clazz;
        };

        MessagePrototype.encode = function (message, buffer, noVerify) {
          var fieldMissing = null,
              field;

          for (var i = 0, k = this._fields.length, val; i < k; ++i) {
            field = this._fields[i];
            val = message[field.name];

            if (field.required && val === null) {
              if (fieldMissing === null) {
                fieldMissing = field;
              }
            } else {
              field.encode(noVerify ? val : field.verifyValue(val), buffer, message);
            }
          }

          if (fieldMissing !== null) {
            var err = Error("Missing at least one required field for " + this.toString(true) + ": " + fieldMissing);
            err["encoded"] = buffer;
            throw err;
          }

          return buffer;
        };

        MessagePrototype.calculate = function (message) {
          for (var n = 0, i = 0, k = this._fields.length, field, val; i < k; ++i) {
            field = this._fields[i];
            val = message[field.name];

            if (field.required && val === null) {
              throw Error("Missing at least one required field for " + this.toString(true) + ": " + field);
            } else {
              n += field.calculate(val, message);
            }
          }

          return n;
        };

        function skipTillGroupEnd(expectedId, buf) {
          var tag = buf.readVarint32(),
              wireType = tag & 7,
              id = tag >>> 3;

          switch (wireType) {
            case ProtoBuf.WIRE_TYPES.VARINT:
              do {
                tag = buf.readUint8();
              } while ((tag & 128) === 128);

              break;

            case ProtoBuf.WIRE_TYPES.BITS64:
              buf.offset += 8;
              break;

            case ProtoBuf.WIRE_TYPES.LDELIM:
              tag = buf.readVarint32();
              buf.offset += tag;
              break;

            case ProtoBuf.WIRE_TYPES.STARTGROUP:
              skipTillGroupEnd(id, buf);
              break;

            case ProtoBuf.WIRE_TYPES.ENDGROUP:
              if (id === expectedId) {
                return false;
              } else {
                throw Error("Illegal GROUPEND after unknown group: " + id + " (" + expectedId + " expected)");
              }

            case ProtoBuf.WIRE_TYPES.BITS32:
              buf.offset += 4;
              break;

            default:
              throw Error("Illegal wire type in unknown group " + expectedId + ": " + wireType);
          }

          return true;
        }

        MessagePrototype.decode = function (buffer, length, expectedGroupEndId) {
          if (typeof length !== "number") {
            length = -1;
          }

          var start = buffer.offset,
              msg = new this.clazz(),
              tag,
              wireType,
              id,
              field;

          while (buffer.offset < start + length || length === -1 && buffer.remaining() > 0) {
            tag = buffer.readVarint32();
            wireType = tag & 7;
            id = tag >>> 3;

            if (wireType === ProtoBuf.WIRE_TYPES.ENDGROUP) {
              if (id !== expectedGroupEndId) {
                throw Error("Illegal group end indicator for " + this.toString(true) + ": " + id + " (" + (expectedGroupEndId ? expectedGroupEndId + " expected" : "not a group") + ")");
              }

              break;
            }

            if (!(field = this._fieldsById[id])) {
              switch (wireType) {
                case ProtoBuf.WIRE_TYPES.VARINT:
                  buffer.readVarint32();
                  break;

                case ProtoBuf.WIRE_TYPES.BITS32:
                  buffer.offset += 4;
                  break;

                case ProtoBuf.WIRE_TYPES.BITS64:
                  buffer.offset += 8;
                  break;

                case ProtoBuf.WIRE_TYPES.LDELIM:
                  var len = buffer.readVarint32();
                  buffer.offset += len;
                  break;

                case ProtoBuf.WIRE_TYPES.STARTGROUP:
                  while (skipTillGroupEnd(id, buffer)) {}

                  break;

                default:
                  throw Error("Illegal wire type for unknown field " + id + " in " + this.toString(true) + "#decode: " + wireType);
              }

              continue;
            }

            if (field.repeated && !field.options["packed"]) {
              msg[field.name].push(field.decode(wireType, buffer));
            } else {
              if (field.map) {
                var keyval = field.decode(wireType, buffer);
                msg[field.name].set(keyval[0], keyval[1]);
              } else {
                msg[field.name] = field.decode(wireType, buffer);

                if (field.oneof) {
                  var currentField = msg[field.oneof.name];

                  if (currentField !== null && currentField !== field.name) {
                    msg[currentField] = null;
                  }

                  msg[field.oneof.name] = field.name;
                }
              }
            }
          }

          for (var i = 0, k = this._fields.length; i < k; ++i) {
            field = this._fields[i];

            if (msg[field.name] === null) {
              if (this.syntax === "proto3") {
                msg[field.name] = field.defaultValue;
              } else {
                if (field.required) {
                  var err = Error("Missing at least one required field for " + this.toString(true) + ": " + field.name);
                  err["decoded"] = msg;
                  throw err;
                } else {
                  if (ProtoBuf.populateDefaults && field.defaultValue !== null) {
                    msg[field.name] = field.defaultValue;
                  }
                }
              }
            }

            if (msg[field.name] === null) {
              delete msg[field.name];
            }
          }

          return msg;
        };

        Reflect.Message = Message;

        var Field = function (builder, message, rule, keytype, type, name, id, options, oneof, syntax) {
          T.call(this, builder, message, name);
          this.className = "Message.Field";
          this.required = rule === "required";
          this.repeated = rule === "repeated";
          this.map = rule === "map";
          this.keyType = keytype || null;
          this.type = type;
          this.resolvedType = null;
          this.id = id;
          this.options = options || {};
          this.defaultValue = null;
          this.oneof = oneof || null;
          this.syntax = syntax || "proto2";
          this.originalName = this.name;
          this.element = null;
          this.keyElement = null;

          if (this.builder.options["convertFieldsToCamelCase"] && !(this instanceof Message.ExtensionField)) {
            this.name = ProtoBuf.Util.toCamelCase(this.name);
          }
        };

        var FieldPrototype = Field.prototype = Object.create(T.prototype);

        FieldPrototype.build = function () {
          this.element = new Element(this.type, this.resolvedType, false, this.syntax, this.name);

          if (this.map) {
            this.keyElement = new Element(this.keyType, undefined, true, this.syntax, this.name);
          }

          if (this.syntax === "proto3" && !this.repeated && !this.map) {
            this.defaultValue = Element.defaultFieldValue(this.type);
          } else {
            if (typeof this.options["default"] !== "undefined") {
              this.defaultValue = this.verifyValue(this.options["default"]);
            }
          }
        };

        FieldPrototype.verifyValue = function (value, skipRepeated) {
          skipRepeated = skipRepeated || false;
          var self = this;

          function fail(val, msg) {
            throw Error("Illegal value for " + self.toString(true) + " of type " + self.type.name + ": " + val + " (" + msg + ")");
          }

          if (value === null) {
            if (this.required) {
              fail(typeof value, "required");
            }

            if (this.syntax === "proto3" && this.type !== ProtoBuf.TYPES["message"]) {
              fail(typeof value, "proto3 field without field presence cannot be null");
            }

            return null;
          }

          var i;

          if (this.repeated && !skipRepeated) {
            if (!Array.isArray(value)) {
              value = [value];
            }

            var res = [];

            for (i = 0; i < value.length; i++) {
              res.push(this.element.verifyValue(value[i]));
            }

            return res;
          }

          if (this.map && !skipRepeated) {
            if (!(value instanceof ProtoBuf.Map)) {
              if (!(value instanceof Object)) {
                fail(typeof value, "expected ProtoBuf.Map or raw object for map field");
              }

              return new ProtoBuf.Map(this, value);
            } else {
              return value;
            }
          }

          if (!this.repeated && Array.isArray(value)) {
            fail(typeof value, "no array expected");
          }

          return this.element.verifyValue(value);
        };

        FieldPrototype.hasWirePresence = function (value, message) {
          if (this.syntax !== "proto3") {
            return value !== null;
          }

          if (this.oneof && message[this.oneof.name] === this.name) {
            return true;
          }

          switch (this.type) {
            case ProtoBuf.TYPES["int32"]:
            case ProtoBuf.TYPES["sint32"]:
            case ProtoBuf.TYPES["sfixed32"]:
            case ProtoBuf.TYPES["uint32"]:
            case ProtoBuf.TYPES["fixed32"]:
              return value !== 0;

            case ProtoBuf.TYPES["int64"]:
            case ProtoBuf.TYPES["sint64"]:
            case ProtoBuf.TYPES["sfixed64"]:
            case ProtoBuf.TYPES["uint64"]:
            case ProtoBuf.TYPES["fixed64"]:
              return value.low !== 0 || value.high !== 0;

            case ProtoBuf.TYPES["bool"]:
              return value;

            case ProtoBuf.TYPES["float"]:
            case ProtoBuf.TYPES["double"]:
              return value !== 0;

            case ProtoBuf.TYPES["string"]:
              return value.length > 0;

            case ProtoBuf.TYPES["bytes"]:
              return value.remaining() > 0;

            case ProtoBuf.TYPES["enum"]:
              return value !== 0;

            case ProtoBuf.TYPES["message"]:
              return value !== null;

            default:
              return true;
          }
        };

        FieldPrototype.encode = function (value, buffer, message) {
          if (this.type === null || typeof this.type !== "object") {
            throw Error("[INTERNAL] Unresolved type in " + this.toString(true) + ": " + this.type);
          }

          if (value === null || this.repeated && value.length == 0) {
            return buffer;
          }

          try {
            if (this.repeated) {
              var i;

              if (this.options["packed"] && ProtoBuf.PACKABLE_WIRE_TYPES.indexOf(this.type.wireType) >= 0) {
                buffer.writeVarint32(this.id << 3 | ProtoBuf.WIRE_TYPES.LDELIM);
                buffer.ensureCapacity(buffer.offset += 1);
                var start = buffer.offset;

                for (i = 0; i < value.length; i++) {
                  this.element.encodeValue(this.id, value[i], buffer);
                }

                var len = buffer.offset - start,
                    varintLen = ByteBuffer.calculateVarint32(len);

                if (varintLen > 1) {
                  var contents = buffer.slice(start, buffer.offset);
                  start += varintLen - 1;
                  buffer.offset = start;
                  buffer.append(contents);
                }

                buffer.writeVarint32(len, start - varintLen);
              } else {
                for (i = 0; i < value.length; i++) {
                  buffer.writeVarint32(this.id << 3 | this.type.wireType), this.element.encodeValue(this.id, value[i], buffer);
                }
              }
            } else {
              if (this.map) {
                value.forEach(function (val, key, m) {
                  var length = ByteBuffer.calculateVarint32(1 << 3 | this.keyType.wireType) + this.keyElement.calculateLength(1, key) + ByteBuffer.calculateVarint32(2 << 3 | this.type.wireType) + this.element.calculateLength(2, val);
                  buffer.writeVarint32(this.id << 3 | ProtoBuf.WIRE_TYPES.LDELIM);
                  buffer.writeVarint32(length);
                  buffer.writeVarint32(1 << 3 | this.keyType.wireType);
                  this.keyElement.encodeValue(1, key, buffer);
                  buffer.writeVarint32(2 << 3 | this.type.wireType);
                  this.element.encodeValue(2, val, buffer);
                }, this);
              } else {
                if (this.hasWirePresence(value, message)) {
                  buffer.writeVarint32(this.id << 3 | this.type.wireType);
                  this.element.encodeValue(this.id, value, buffer);
                }
              }
            }
          } catch (e) {
            throw Error("Illegal value for " + this.toString(true) + ": " + value + " (" + e + ")");
          }

          return buffer;
        };

        FieldPrototype.calculate = function (value, message) {
          value = this.verifyValue(value);

          if (this.type === null || typeof this.type !== "object") {
            throw Error("[INTERNAL] Unresolved type in " + this.toString(true) + ": " + this.type);
          }

          if (value === null || this.repeated && value.length == 0) {
            return 0;
          }

          var n = 0;

          try {
            if (this.repeated) {
              var i, ni;

              if (this.options["packed"] && ProtoBuf.PACKABLE_WIRE_TYPES.indexOf(this.type.wireType) >= 0) {
                n += ByteBuffer.calculateVarint32(this.id << 3 | ProtoBuf.WIRE_TYPES.LDELIM);
                ni = 0;

                for (i = 0; i < value.length; i++) {
                  ni += this.element.calculateLength(this.id, value[i]);
                }

                n += ByteBuffer.calculateVarint32(ni);
                n += ni;
              } else {
                for (i = 0; i < value.length; i++) {
                  n += ByteBuffer.calculateVarint32(this.id << 3 | this.type.wireType), n += this.element.calculateLength(this.id, value[i]);
                }
              }
            } else {
              if (this.map) {
                value.forEach(function (val, key, m) {
                  var length = ByteBuffer.calculateVarint32(1 << 3 | this.keyType.wireType) + this.keyElement.calculateLength(1, key) + ByteBuffer.calculateVarint32(2 << 3 | this.type.wireType) + this.element.calculateLength(2, val);
                  n += ByteBuffer.calculateVarint32(this.id << 3 | ProtoBuf.WIRE_TYPES.LDELIM);
                  n += ByteBuffer.calculateVarint32(length);
                  n += length;
                }, this);
              } else {
                if (this.hasWirePresence(value, message)) {
                  n += ByteBuffer.calculateVarint32(this.id << 3 | this.type.wireType);
                  n += this.element.calculateLength(this.id, value);
                }
              }
            }
          } catch (e) {
            throw Error("Illegal value for " + this.toString(true) + ": " + value + " (" + e + ")");
          }

          return n;
        };

        FieldPrototype.decode = function (wireType, buffer, skipRepeated) {
          var value, nBytes;
          var wireTypeOK = !this.map && wireType == this.type.wireType || !skipRepeated && this.repeated && this.options["packed"] && wireType == ProtoBuf.WIRE_TYPES.LDELIM || this.map && wireType == ProtoBuf.WIRE_TYPES.LDELIM;

          if (!wireTypeOK) {
            throw Error("Illegal wire type for field " + this.toString(true) + ": " + wireType + " (" + this.type.wireType + " expected)");
          }

          if (wireType == ProtoBuf.WIRE_TYPES.LDELIM && this.repeated && this.options["packed"] && ProtoBuf.PACKABLE_WIRE_TYPES.indexOf(this.type.wireType) >= 0) {
            if (!skipRepeated) {
              nBytes = buffer.readVarint32();
              nBytes = buffer.offset + nBytes;
              var values = [];

              while (buffer.offset < nBytes) {
                values.push(this.decode(this.type.wireType, buffer, true));
              }

              return values;
            }
          }

          if (this.map) {
            var key = Element.defaultFieldValue(this.keyType);
            value = Element.defaultFieldValue(this.type);
            nBytes = buffer.readVarint32();

            if (buffer.remaining() < nBytes) {
              throw Error("Illegal number of bytes for " + this.toString(true) + ": " + nBytes + " required but got only " + buffer.remaining());
            }

            var msgbuf = buffer.clone();
            msgbuf.limit = msgbuf.offset + nBytes;
            buffer.offset += nBytes;

            while (msgbuf.remaining() > 0) {
              var tag = msgbuf.readVarint32();
              wireType = tag & 7;
              var id = tag >>> 3;

              if (id === 1) {
                key = this.keyElement.decode(msgbuf, wireType, id);
              } else {
                if (id === 2) {
                  value = this.element.decode(msgbuf, wireType, id);
                } else {
                  throw Error("Unexpected tag in map field key/value submessage");
                }
              }
            }

            return [key, value];
          }

          return this.element.decode(buffer, wireType, this.id);
        };

        Reflect.Message.Field = Field;

        var ExtensionField = function (builder, message, rule, type, name, id, options) {
          Field.call(this, builder, message, rule, null, type, name, id, options);
          this.extension;
        };

        ExtensionField.prototype = Object.create(Field.prototype);
        Reflect.Message.ExtensionField = ExtensionField;

        var OneOf = function (builder, message, name) {
          T.call(this, builder, message, name);
          this.fields = [];
        };

        Reflect.Message.OneOf = OneOf;

        var Enum = function (builder, parent, name, options, syntax) {
          Namespace.call(this, builder, parent, name, options, syntax);
          this.className = "Enum";
          this.object = null;
        };

        Enum.getName = function (enm, value) {
          var keys = Object.keys(enm);

          for (var i = 0, key; i < keys.length; ++i) {
            if (enm[key = keys[i]] === value) {
              return key;
            }
          }

          return null;
        };

        var EnumPrototype = Enum.prototype = Object.create(Namespace.prototype);

        EnumPrototype.build = function (rebuild) {
          if (this.object && !rebuild) {
            return this.object;
          }

          var enm = new ProtoBuf.Builder.Enum(),
              values = this.getChildren(Enum.Value);

          for (var i = 0, k = values.length; i < k; ++i) {
            enm[values[i]["name"]] = values[i]["id"];
          }

          if (Object.defineProperty) {
            Object.defineProperty(enm, "$options", {
              "value": this.buildOpt(),
              "enumerable": false
            });
          }

          return this.object = enm;
        };

        Reflect.Enum = Enum;

        var Value = function (builder, enm, name, id) {
          T.call(this, builder, enm, name);
          this.className = "Enum.Value";
          this.id = id;
        };

        Value.prototype = Object.create(T.prototype);
        Reflect.Enum.Value = Value;

        var Extension = function (builder, parent, name, field) {
          T.call(this, builder, parent, name);
          this.field = field;
        };

        Extension.prototype = Object.create(T.prototype);
        Reflect.Extension = Extension;

        var Service = function (builder, root, name, options) {
          Namespace.call(this, builder, root, name, options);
          this.className = "Service";
          this.clazz = null;
        };

        var ServicePrototype = Service.prototype = Object.create(Namespace.prototype);

        ServicePrototype.build = function (rebuild) {
          if (this.clazz && !rebuild) {
            return this.clazz;
          }

          return this.clazz = function (ProtoBuf, T) {
            var Service = function (rpcImpl) {
              ProtoBuf.Builder.Service.call(this);

              this.rpcImpl = rpcImpl || function (name, msg, callback) {
                setTimeout(callback.bind(this, Error("Not implemented, see: https://github.com/dcodeIO/ProtoBuf.js/wiki/Services")), 0);
              };
            };

            var ServicePrototype = Service.prototype = Object.create(ProtoBuf.Builder.Service.prototype);
            var rpc = T.getChildren(ProtoBuf.Reflect.Service.RPCMethod);

            for (var i = 0; i < rpc.length; i++) {
              (function (method) {
                ServicePrototype[method.name] = function (req, callback) {
                  try {
                    try {
                      req = method.resolvedRequestType.clazz.decode(ByteBuffer.wrap(req));
                    } catch (err) {
                      if (!(err instanceof TypeError)) {
                        throw err;
                      }
                    }

                    if (req === null || typeof req !== "object") {
                      throw Error("Illegal arguments");
                    }

                    if (!(req instanceof method.resolvedRequestType.clazz)) {
                      req = new method.resolvedRequestType.clazz(req);
                    }

                    this.rpcImpl(method.fqn(), req, function (err, res) {
                      if (err) {
                        callback(err);
                        return;
                      }

                      if (res === null) {
                        res = "";
                      }

                      try {
                        res = method.resolvedResponseType.clazz.decode(res);
                      } catch (notABuffer) {}

                      if (!res || !(res instanceof method.resolvedResponseType.clazz)) {
                        callback(Error("Illegal response type received in service method " + T.name + "#" + method.name));
                        return;
                      }

                      callback(null, res);
                    });
                  } catch (err) {
                    setTimeout(callback.bind(this, err), 0);
                  }
                };

                Service[method.name] = function (rpcImpl, req, callback) {
                  new Service(rpcImpl)[method.name](req, callback);
                };

                if (Object.defineProperty) {
                  Object.defineProperty(Service[method.name], "$options", {
                    "value": method.buildOpt()
                  }), Object.defineProperty(ServicePrototype[method.name], "$options", {
                    "value": Service[method.name]["$options"]
                  });
                }
              })(rpc[i]);
            }

            var $optionsS;
            var $options;
            var $typeS;
            var $type;

            if (Object.defineProperty) {
              Object.defineProperty(Service, "$options", {
                "value": T.buildOpt()
              }), Object.defineProperty(ServicePrototype, "$options", {
                "value": Service["$options"]
              }), Object.defineProperty(Service, "$type", {
                "value": T
              }), Object.defineProperty(ServicePrototype, "$type", {
                "value": T
              });
            }

            return Service;
          }(ProtoBuf, this);
        };

        Reflect.Service = Service;

        var Method = function (builder, svc, name, options) {
          T.call(this, builder, svc, name);
          this.className = "Service.Method";
          this.options = options || {};
        };

        var MethodPrototype = Method.prototype = Object.create(T.prototype);
        MethodPrototype.buildOpt = NamespacePrototype.buildOpt;
        Reflect.Service.Method = Method;

        var RPCMethod = function (builder, svc, name, request, response, request_stream, response_stream, options) {
          Method.call(this, builder, svc, name, options);
          this.className = "Service.RPCMethod";
          this.requestName = request;
          this.responseName = response;
          this.requestStream = request_stream;
          this.responseStream = response_stream;
          this.resolvedRequestType = null;
          this.resolvedResponseType = null;
        };

        RPCMethod.prototype = Object.create(Method.prototype);
        Reflect.Service.RPCMethod = RPCMethod;
        return Reflect;
      }(ProtoBuf);

      ProtoBuf.Builder = function (ProtoBuf, Lang, Reflect) {
        var Builder = function (options) {
          this.ns = new Reflect.Namespace(this, null, "");
          this.ptr = this.ns;
          this.resolved = false;
          this.result = null;
          this.files = {};
          this.importRoot = null;
          this.options = options || {};
        };

        var BuilderPrototype = Builder.prototype;

        Builder.isMessage = function (def) {
          if (typeof def["name"] !== "string") {
            return false;
          }

          if (typeof def["values"] !== "undefined" || typeof def["rpc"] !== "undefined") {
            return false;
          }

          return true;
        };

        Builder.isMessageField = function (def) {
          if (typeof def["rule"] !== "string" || typeof def["name"] !== "string" || typeof def["type"] !== "string" || typeof def["id"] === "undefined") {
            return false;
          }

          return true;
        };

        Builder.isEnum = function (def) {
          if (typeof def["name"] !== "string") {
            return false;
          }

          if (typeof def["values"] === "undefined" || !Array.isArray(def["values"]) || def["values"].length === 0) {
            return false;
          }

          return true;
        };

        Builder.isService = function (def) {
          if (typeof def["name"] !== "string" || typeof def["rpc"] !== "object" || !def["rpc"]) {
            return false;
          }

          return true;
        };

        Builder.isExtend = function (def) {
          if (typeof def["ref"] !== "string") {
            return false;
          }

          return true;
        };

        BuilderPrototype.reset = function () {
          this.ptr = this.ns;
          return this;
        };

        BuilderPrototype.define = function (namespace) {
          if (typeof namespace !== "string" || !Lang.TYPEREF.test(namespace)) {
            throw Error("illegal namespace: " + namespace);
          }

          namespace.split(".").forEach(function (part) {
            var ns = this.ptr.getChild(part);

            if (ns === null) {
              this.ptr.addChild(ns = new Reflect.Namespace(this, this.ptr, part));
            }

            this.ptr = ns;
          }, this);
          return this;
        };

        BuilderPrototype.create = function (defs) {
          if (!defs) {
            return this;
          }

          if (!Array.isArray(defs)) {
            defs = [defs];
          } else {
            if (defs.length === 0) {
              return this;
            }

            defs = defs.slice();
          }

          var stack = [defs];

          while (stack.length > 0) {
            defs = stack.pop();

            if (!Array.isArray(defs)) {
              throw Error("not a valid namespace: " + JSON.stringify(defs));
            }

            while (defs.length > 0) {
              var def = defs.shift();

              if (Builder.isMessage(def)) {
                var obj = new Reflect.Message(this, this.ptr, def["name"], def["options"], def["isGroup"], def["syntax"]);
                var oneofs = {};

                if (def["oneofs"]) {
                  Object.keys(def["oneofs"]).forEach(function (name) {
                    obj.addChild(oneofs[name] = new Reflect.Message.OneOf(this, obj, name));
                  }, this);
                }

                if (def["fields"]) {
                  def["fields"].forEach(function (fld) {
                    if (obj.getChild(fld["id"] | 0) !== null) {
                      throw Error("duplicate or invalid field id in " + obj.name + ": " + fld["id"]);
                    }

                    if (fld["options"] && typeof fld["options"] !== "object") {
                      throw Error("illegal field options in " + obj.name + "#" + fld["name"]);
                    }

                    var oneof = null;

                    if (typeof fld["oneof"] === "string" && !(oneof = oneofs[fld["oneof"]])) {
                      throw Error("illegal oneof in " + obj.name + "#" + fld["name"] + ": " + fld["oneof"]);
                    }

                    fld = new Reflect.Message.Field(this, obj, fld["rule"], fld["keytype"], fld["type"], fld["name"], fld["id"], fld["options"], oneof, def["syntax"]);

                    if (oneof) {
                      oneof.fields.push(fld);
                    }

                    obj.addChild(fld);
                  }, this);
                }

                var subObj = [];

                if (def["enums"]) {
                  def["enums"].forEach(function (enm) {
                    subObj.push(enm);
                  });
                }

                if (def["messages"]) {
                  def["messages"].forEach(function (msg) {
                    subObj.push(msg);
                  });
                }

                if (def["services"]) {
                  def["services"].forEach(function (svc) {
                    subObj.push(svc);
                  });
                }

                if (def["extensions"]) {
                  if (typeof def["extensions"][0] === "number") {
                    obj.extensions = [def["extensions"]];
                  } else {
                    obj.extensions = def["extensions"];
                  }
                }

                this.ptr.addChild(obj);

                if (subObj.length > 0) {
                  stack.push(defs);
                  defs = subObj;
                  subObj = null;
                  this.ptr = obj;
                  obj = null;
                  continue;
                }

                subObj = null;
              } else {
                if (Builder.isEnum(def)) {
                  obj = new Reflect.Enum(this, this.ptr, def["name"], def["options"], def["syntax"]);
                  def["values"].forEach(function (val) {
                    obj.addChild(new Reflect.Enum.Value(this, obj, val["name"], val["id"]));
                  }, this);
                  this.ptr.addChild(obj);
                } else {
                  if (Builder.isService(def)) {
                    obj = new Reflect.Service(this, this.ptr, def["name"], def["options"]);
                    Object.keys(def["rpc"]).forEach(function (name) {
                      var mtd = def["rpc"][name];
                      obj.addChild(new Reflect.Service.RPCMethod(this, obj, name, mtd["request"], mtd["response"], !!mtd["request_stream"], !!mtd["response_stream"], mtd["options"]));
                    }, this);
                    this.ptr.addChild(obj);
                  } else {
                    if (Builder.isExtend(def)) {
                      obj = this.ptr.resolve(def["ref"], true);

                      if (obj) {
                        def["fields"].forEach(function (fld) {
                          if (obj.getChild(fld["id"] | 0) !== null) {
                            throw Error("duplicate extended field id in " + obj.name + ": " + fld["id"]);
                          }

                          if (obj.extensions) {
                            var valid = false;
                            obj.extensions.forEach(function (range) {
                              if (fld["id"] >= range[0] && fld["id"] <= range[1]) {
                                valid = true;
                              }
                            });

                            if (!valid) {
                              throw Error("illegal extended field id in " + obj.name + ": " + fld["id"] + " (not within valid ranges)");
                            }
                          }

                          var name = fld["name"];

                          if (this.options["convertFieldsToCamelCase"]) {
                            name = ProtoBuf.Util.toCamelCase(name);
                          }

                          var field = new Reflect.Message.ExtensionField(this, obj, fld["rule"], fld["type"], this.ptr.fqn() + "." + name, fld["id"], fld["options"]);
                          var ext = new Reflect.Extension(this, this.ptr, fld["name"], field);
                          field.extension = ext;
                          this.ptr.addChild(ext);
                          obj.addChild(field);
                        }, this);
                      } else {
                        if (!/\.?google\.protobuf\./.test(def["ref"])) {
                          throw Error("extended message " + def["ref"] + " is not defined");
                        }
                      }
                    } else {
                      throw Error("not a valid definition: " + JSON.stringify(def));
                    }
                  }
                }
              }

              def = null;
              obj = null;
            }

            defs = null;
            this.ptr = this.ptr.parent;
          }

          this.resolved = false;
          this.result = null;
          return this;
        };

        function propagateSyntax(parent) {
          if (parent["messages"]) {
            parent["messages"].forEach(function (child) {
              child["syntax"] = parent["syntax"];
              propagateSyntax(child);
            });
          }

          if (parent["enums"]) {
            parent["enums"].forEach(function (child) {
              child["syntax"] = parent["syntax"];
            });
          }
        }

        BuilderPrototype["import"] = function (json, filename) {
          var delim = "/";

          if (typeof filename === "string") {
            if (ProtoBuf.Util.IS_NODE) {
              filename = require("path")["resolve"](filename);
            }

            if (this.files[filename] === true) {
              return this.reset();
            }

            this.files[filename] = true;
          } else {
            if (typeof filename === "object") {
              var root = filename.root;

              if (ProtoBuf.Util.IS_NODE) {
                root = require("path")["resolve"](root);
              }

              if (root.indexOf("\\") >= 0 || filename.file.indexOf("\\") >= 0) {
                delim = "\\";
              }

              var fname = root + delim + filename.file;

              if (this.files[fname] === true) {
                return this.reset();
              }

              this.files[fname] = true;
            }
          }

          if (json["imports"] && json["imports"].length > 0) {
            var importRoot,
                resetRoot = false;

            if (typeof filename === "object") {
              this.importRoot = filename["root"];
              resetRoot = true;
              importRoot = this.importRoot;
              filename = filename["file"];

              if (importRoot.indexOf("\\") >= 0 || filename.indexOf("\\") >= 0) {
                delim = "\\";
              }
            } else {
              if (typeof filename === "string") {
                if (this.importRoot) {
                  importRoot = this.importRoot;
                } else {
                  if (filename.indexOf("/") >= 0) {
                    importRoot = filename.replace(/\/[^\/]*$/, "");

                    if (importRoot === "") {
                      importRoot = "/";
                    }
                  } else {
                    if (filename.indexOf("\\") >= 0) {
                      importRoot = filename.replace(/\\[^\\]*$/, "");
                      delim = "\\";
                    } else {
                      importRoot = ".";
                    }
                  }
                }
              } else {
                importRoot = null;
              }
            }

            for (var i = 0; i < json["imports"].length; i++) {
              if (typeof json["imports"][i] === "string") {
                if (!importRoot) {
                  throw Error("cannot determine import root");
                }

                var importFilename = json["imports"][i];

                if (importFilename === "google/protobuf/descriptor.proto") {
                  continue;
                }

                importFilename = importRoot + delim + importFilename;

                if (this.files[importFilename] === true) {
                  continue;
                }

                if (/\.proto$/i.test(importFilename) && !ProtoBuf.DotProto) {
                  importFilename = importFilename.replace(/\.proto$/, ".json");
                }

                var contents = ProtoBuf.Util.fetch(importFilename);

                if (contents === null) {
                  throw Error("failed to import '" + importFilename + "' in '" + filename + "': file not found");
                }

                if (/\.json$/i.test(importFilename)) {
                  this["import"](JSON.parse(contents + ""), importFilename);
                } else {
                  this["import"](ProtoBuf.DotProto.Parser.parse(contents), importFilename);
                }
              } else {
                if (!filename) {
                  this["import"](json["imports"][i]);
                } else {
                  if (/\.(\w+)$/.test(filename)) {
                    this["import"](json["imports"][i], filename.replace(/^(.+)\.(\w+)$/, function ($0, $1, $2) {
                      return $1 + "_import" + i + "." + $2;
                    }));
                  } else {
                    this["import"](json["imports"][i], filename + "_import" + i);
                  }
                }
              }
            }

            if (resetRoot) {
              this.importRoot = null;
            }
          }

          if (json["package"]) {
            this.define(json["package"]);
          }

          if (json["syntax"]) {
            propagateSyntax(json);
          }

          var base = this.ptr;

          if (json["options"]) {
            Object.keys(json["options"]).forEach(function (key) {
              base.options[key] = json["options"][key];
            });
          }

          if (json["messages"]) {
            this.create(json["messages"]), this.ptr = base;
          }

          if (json["enums"]) {
            this.create(json["enums"]), this.ptr = base;
          }

          if (json["services"]) {
            this.create(json["services"]), this.ptr = base;
          }

          if (json["extends"]) {
            this.create(json["extends"]);
          }

          return this.reset();
        };

        BuilderPrototype.resolveAll = function () {
          var res;

          if (this.ptr == null || typeof this.ptr.type === "object") {
            return this;
          }

          if (this.ptr instanceof Reflect.Namespace) {
            this.ptr.children.forEach(function (child) {
              this.ptr = child;
              this.resolveAll();
            }, this);
          } else {
            if (this.ptr instanceof Reflect.Message.Field) {
              if (!Lang.TYPE.test(this.ptr.type)) {
                if (!Lang.TYPEREF.test(this.ptr.type)) {
                  throw Error("illegal type reference in " + this.ptr.toString(true) + ": " + this.ptr.type);
                }

                res = (this.ptr instanceof Reflect.Message.ExtensionField ? this.ptr.extension.parent : this.ptr.parent).resolve(this.ptr.type, true);

                if (!res) {
                  throw Error("unresolvable type reference in " + this.ptr.toString(true) + ": " + this.ptr.type);
                }

                this.ptr.resolvedType = res;

                if (res instanceof Reflect.Enum) {
                  this.ptr.type = ProtoBuf.TYPES["enum"];

                  if (this.ptr.syntax === "proto3" && res.syntax !== "proto3") {
                    throw Error("proto3 message cannot reference proto2 enum");
                  }
                } else {
                  if (res instanceof Reflect.Message) {
                    this.ptr.type = res.isGroup ? ProtoBuf.TYPES["group"] : ProtoBuf.TYPES["message"];
                  } else {
                    throw Error("illegal type reference in " + this.ptr.toString(true) + ": " + this.ptr.type);
                  }
                }
              } else {
                this.ptr.type = ProtoBuf.TYPES[this.ptr.type];
              }

              if (this.ptr.map) {
                if (!Lang.TYPE.test(this.ptr.keyType)) {
                  throw Error("illegal key type for map field in " + this.ptr.toString(true) + ": " + this.ptr.keyType);
                }

                this.ptr.keyType = ProtoBuf.TYPES[this.ptr.keyType];
              }
            } else {
              if (this.ptr instanceof ProtoBuf.Reflect.Service.Method) {
                if (this.ptr instanceof ProtoBuf.Reflect.Service.RPCMethod) {
                  res = this.ptr.parent.resolve(this.ptr.requestName, true);

                  if (!res || !(res instanceof ProtoBuf.Reflect.Message)) {
                    throw Error("Illegal type reference in " + this.ptr.toString(true) + ": " + this.ptr.requestName);
                  }

                  this.ptr.resolvedRequestType = res;
                  res = this.ptr.parent.resolve(this.ptr.responseName, true);

                  if (!res || !(res instanceof ProtoBuf.Reflect.Message)) {
                    throw Error("Illegal type reference in " + this.ptr.toString(true) + ": " + this.ptr.responseName);
                  }

                  this.ptr.resolvedResponseType = res;
                } else {
                  throw Error("illegal service type in " + this.ptr.toString(true));
                }
              } else {
                if (!(this.ptr instanceof ProtoBuf.Reflect.Message.OneOf) && !(this.ptr instanceof ProtoBuf.Reflect.Extension) && !(this.ptr instanceof ProtoBuf.Reflect.Enum.Value)) {
                  throw Error("illegal object in namespace: " + typeof this.ptr + ": " + this.ptr);
                }
              }
            }
          }

          return this.reset();
        };

        BuilderPrototype.build = function (path) {
          this.reset();

          if (!this.resolved) {
            this.resolveAll(), this.resolved = true, this.result = null;
          }

          if (this.result === null) {
            this.result = this.ns.build();
          }

          if (!path) {
            return this.result;
          }

          var part = typeof path === "string" ? path.split(".") : path,
              ptr = this.result;

          for (var i = 0; i < part.length; i++) {
            if (ptr[part[i]]) {
              ptr = ptr[part[i]];
            } else {
              ptr = null;
              break;
            }
          }

          return ptr;
        };

        BuilderPrototype.lookup = function (path, excludeNonNamespace) {
          return path ? this.ns.resolve(path, excludeNonNamespace) : this.ns;
        };

        BuilderPrototype.toString = function () {
          return "Builder";
        };

        Builder.Message = function () {};

        Builder.Enum = function () {};

        Builder.Service = function () {};

        return Builder;
      }(ProtoBuf, ProtoBuf.Lang, ProtoBuf.Reflect);

      ProtoBuf.Map = function (ProtoBuf, Reflect) {
        var Map = function (field, contents) {
          if (!field.map) {
            throw Error("field is not a map");
          }

          this.field = field;
          this.keyElem = new Reflect.Element(field.keyType, null, true, field.syntax);
          this.valueElem = new Reflect.Element(field.type, field.resolvedType, false, field.syntax);
          this.map = {};
          Object.defineProperty(this, "size", {
            get: function () {
              return Object.keys(this.map).length;
            }
          });

          if (contents) {
            var keys = Object.keys(contents);

            for (var i = 0; i < keys.length; i++) {
              var key = this.keyElem.valueFromString(keys[i]);
              var val = this.valueElem.verifyValue(contents[keys[i]]);
              this.map[this.keyElem.valueToString(key)] = {
                key: key,
                value: val
              };
            }
          }
        };

        var MapPrototype = Map.prototype;

        function arrayIterator(arr) {
          var idx = 0;
          return {
            next: function () {
              if (idx < arr.length) {
                return {
                  done: false,
                  value: arr[idx++]
                };
              }

              return {
                done: true
              };
            }
          };
        }

        MapPrototype.clear = function () {
          this.map = {};
        };

        MapPrototype["delete"] = function (key) {
          var keyValue = this.keyElem.valueToString(this.keyElem.verifyValue(key));
          var hadKey = (keyValue in this.map);
          delete this.map[keyValue];
          return hadKey;
        };

        MapPrototype.entries = function () {
          var entries = [];
          var strKeys = Object.keys(this.map);

          for (var i = 0, entry; i < strKeys.length; i++) {
            entries.push([(entry = this.map[strKeys[i]]).key, entry.value]);
          }

          return arrayIterator(entries);
        };

        MapPrototype.keys = function () {
          var keys = [];
          var strKeys = Object.keys(this.map);

          for (var i = 0; i < strKeys.length; i++) {
            keys.push(this.map[strKeys[i]].key);
          }

          return arrayIterator(keys);
        };

        MapPrototype.values = function () {
          var values = [];
          var strKeys = Object.keys(this.map);

          for (var i = 0; i < strKeys.length; i++) {
            values.push(this.map[strKeys[i]].value);
          }

          return arrayIterator(values);
        };

        MapPrototype.forEach = function (cb, thisArg) {
          var strKeys = Object.keys(this.map);

          for (var i = 0, entry; i < strKeys.length; i++) {
            cb.call(thisArg, (entry = this.map[strKeys[i]]).value, entry.key, this);
          }
        };

        MapPrototype.set = function (key, value) {
          var keyValue = this.keyElem.verifyValue(key);
          var valValue = this.valueElem.verifyValue(value);
          this.map[this.keyElem.valueToString(keyValue)] = {
            key: keyValue,
            value: valValue
          };
          return this;
        };

        MapPrototype.get = function (key) {
          var keyValue = this.keyElem.valueToString(this.keyElem.verifyValue(key));

          if (!(keyValue in this.map)) {
            return undefined;
          }

          return this.map[keyValue].value;
        };

        MapPrototype.has = function (key) {
          var keyValue = this.keyElem.valueToString(this.keyElem.verifyValue(key));
          return keyValue in this.map;
        };

        return Map;
      }(ProtoBuf, ProtoBuf.Reflect);

      ProtoBuf.loadProto = function (proto, builder, filename) {
        if (typeof builder === "string" || builder && typeof builder["file"] === "string" && typeof builder["root"] === "string") {
          filename = builder, builder = undefined;
        }

        return ProtoBuf.loadJson(ProtoBuf.DotProto.Parser.parse(proto), builder, filename);
      };

      ProtoBuf.protoFromString = ProtoBuf.loadProto;

      ProtoBuf.loadProtoFile = function (filename, callback, builder) {
        if (callback && typeof callback === "object") {
          builder = callback, callback = null;
        } else {
          if (!callback || typeof callback !== "function") {
            callback = null;
          }
        }

        if (callback) {
          return ProtoBuf.Util.fetch(typeof filename === "string" ? filename : filename["root"] + "/" + filename["file"], function (contents) {
            if (contents === null) {
              callback(Error("Failed to fetch file"));
              return;
            }

            try {
              callback(null, ProtoBuf.loadProto(contents, builder, filename));
            } catch (e) {
              callback(e);
            }
          });
        }

        var contents = ProtoBuf.Util.fetch(typeof filename === "object" ? filename["root"] + "/" + filename["file"] : filename);
        return contents === null ? null : ProtoBuf.loadProto(contents, builder, filename);
      };

      ProtoBuf.protoFromFile = ProtoBuf.loadProtoFile;

      ProtoBuf.newBuilder = function (options) {
        options = options || {};

        if (typeof options["convertFieldsToCamelCase"] === "undefined") {
          options["convertFieldsToCamelCase"] = ProtoBuf.convertFieldsToCamelCase;
        }

        if (typeof options["populateAccessors"] === "undefined") {
          options["populateAccessors"] = ProtoBuf.populateAccessors;
        }

        return new ProtoBuf.Builder(options);
      };

      ProtoBuf.loadJson = function (json, builder, filename) {
        if (typeof builder === "string" || builder && typeof builder["file"] === "string" && typeof builder["root"] === "string") {
          filename = builder, builder = null;
        }

        if (!builder || typeof builder !== "object") {
          builder = ProtoBuf.newBuilder();
        }

        if (typeof json === "string") {
          json = JSON.parse(json);
        }

        builder["import"](json, filename);
        builder.resolveAll();
        return builder;
      };

      ProtoBuf.loadJsonFile = function (filename, callback, builder) {
        if (callback && typeof callback === "object") {
          builder = callback, callback = null;
        } else {
          if (!callback || typeof callback !== "function") {
            callback = null;
          }
        }

        if (callback) {
          return ProtoBuf.Util.fetch(typeof filename === "string" ? filename : filename["root"] + "/" + filename["file"], function (contents) {
            if (contents === null) {
              callback(Error("Failed to fetch file"));
              return;
            }

            try {
              callback(null, ProtoBuf.loadJson(JSON.parse(contents), builder, filename));
            } catch (e) {
              callback(e);
            }
          });
        }

        var contents = ProtoBuf.Util.fetch(typeof filename === "object" ? filename["root"] + "/" + filename["file"] : filename);
        return contents === null ? null : ProtoBuf.loadJson(JSON.parse(contents), builder, filename);
      };

      return ProtoBuf;
    });
  }).call(root);
})( // The environment-specific global.
function () {
  if (typeof globalThis !== 'undefined') return globalThis;
  if (typeof self !== 'undefined') return self;
  if (typeof window !== 'undefined') return window;
  if (typeof global !== 'undefined') return global;
  if (typeof this !== 'undefined') return this;
  return {};
}.call(this));