/*
 * 基于本地数据拼音匹配引擎
 * Version 0.3 简体中文
 * Date: 2011-09-19
 * http://code.google.com/p/pinyin-engine/
 * (c) 2009-2010 TangBin, http://www.planeArt.cn
 *
 * This is licensed under the GNU LGPL, version 2.1 or later.
 * For details, see: http://creativecommons.org/licenses/LGPL/2.1/
 */
 
var pinyinEngine = function () {
	pinyinEngine.cache();
	return (this instanceof pinyinEngine) ? this.resetCache() : new pinyinEngine;
};

pinyinEngine.prototype = {

	/**
	 * 查询
	 * @param		{String}	关键字
	 * @param		{Function}	回调函数(每一次成功匹配都将运行，
	 *							函数接收的第一个参数为匹配到的内容数据)
	 * @return		{Array}		所有匹配到的内容数据列表
	 * @example		search(function (val) {html += '<p>' + val + '</p>'});
	 */
	search: function (keyword, callback) {
		var cache = this._cache,
			history = this._history,
			value = [];
		
		keyword = keyword.toLowerCase();
		callback = callback || function () {};
		
		// 在上一次搜索结果中查询
		if (history.length && keyword.indexOf(history.keyword) === 1) {
			cache = history.content;
		};
		
		for (var i = 0, len = cache.length, val; i < len; i ++) {
			val = cache[i];
			if (val.tags.indexOf(keyword) !== -1) {
				value.push(val);
				callback(val.content);
			};
		};

		// 缓存本次查询结果
		this._history = {
			keyword: keyword,
			value: value,
			length: value.length
		};
		
		return value;
	},
	
	/**
	 * 建立索引缓存
	 * @param		{Array}		标签
	 * @param		{Any}		被索引的内容
	 * @example		setCache(['art', 'js', 'web', 'html5'], 'http://planeart.cn');
	 */
	setCache: function (tags, content) {
		var keys, excision = '\u0001',
			toPinyin = pinyinEngine.toPinyin;
		
		keys = tags;
		for (var i = 0, len = tags.length; i < len; i ++) {
			 keys.push(toPinyin(tags[i], false, excision));
		};
		keys = keys.join(excision);
		
		var obj = {
			tags: keys,
			content: content
		};
		this._cache.push(obj);
	},
	
	/**
	 * 重置索引缓存
	 */
	resetCache: function () {
		this._cache = [];
		this._history = {};
	}
};



/**
 * 支持多音字的汉字转拼音算法
 * @version	2011-07-22
 * @see		https://github.com/hotoo/pinyin.js
 * @author	闲耘, 唐斌
 * @param	{String}		要转为拼音的目标字符串（汉字）
 * @param	{Boolean}		是否仅保留匹配的第一个拼音
 * @param	{String}		返回结果的分隔符，默认返回数组集合
 * @return	{String, Array} 如果 sp 为 null，则返回 Array
 *							否则，返回以 sp 分隔的字符串
 */
pinyinEngine.toPinyin = function () {
	
    // 笛卡尔乘积，返回两个数组的所有可能的组合
    function product (a, b, sp) {
        var r = [], val, str = [];
        for (var i = 0, l = a.length; i < l; i ++) {
            for (var j = 0, m = b.length; j < m; j ++) {
                val = r[r.length] = (a[i] instanceof Array) ? a[i].concat(b[j]) : [].concat(a[i],b[j]);
				str.push(val.join(""));
            };
        };
        return {
			array: r,
			string: str.join(sp || "")
		};
    };
	
    return function (keyword, single, sp) {
		var cache = pinyinEngine.cache();
        var len = keyword.length, pys, py, pyl, i, y;
		
        if (len === 0) {return single ? "" : []};
        if (len === 1) {
            y = cache[keyword];
            if (single) {return y && y[0] ? y[0] : keyword};
            return y || [keyword];
        } else {
            var py = [];
            for (i = 0; i < len; i ++) {
                y = cache[keyword.charAt(i)];
                if (y) {
                    py[py.length] = single ? y[0] : y;
                } else {
                    py[py.length] = single ? keyword.charAt(i) : [keyword.charAt(i)];
                };
            };
            if (single) {return sp == null ? py : py.join(sp || "")};

            pys = py[0];
			pyl = py.length;
			var prt;
            for (i = 1; i < pyl; i++) {
				prt = product(pys, py[i], sp);
                pys = prt.array;
            };
            return sp == null ? pys : prt.string;
        };
    };
	
}();

/**
 * 汉字拼音索引缓存
 * @return	{Object}	名为汉字值拼音的对象
 * @example	var pinyin = pinyinEngine.cache(); pinyin['乐']; // ["le", "yue"]
 */
pinyinEngine.cache = function () {
	var txt, w, k, l,
		data = pinyinEngine._data || [],
		word = data[0],
		key = data[1],
		cache = {},
		fromCharCode = String.fromCharCode;
	
	if (!data.length) return cache;
	
	// 对原始数据解压缩
	if (typeof key === 'string') key = key.split(',');
	for (var i = 0, arrow = 0, len = word.length; i < len; i ++, arrow ++) {
		k = word[i];
		
		// 处理占位符：负值与空值
		if (k < 0 || k === undefined) {
			arrow += Math.abs(k || 0);
			continue;
		};
		
		txt = fromCharCode(arrow + 19968);
		
		if (typeof k === 'number') {
			cache[txt] = [key[k]];
		// 多音字数据
		} else {
			cache[txt] = [];
			l = k.length;
			for (w = 0; w < l; w ++) {
				cache[txt].push(key[k[w]]);
			};
		};
	};
	
	pinyinEngine.cache = function () {
		return cache;
	};
	return cache;
};

// 拼音对照简体中文数据
pinyinEngine._data = [[350,65,,241,-2,324,372,271,280,332,,20,356,,87,38,,383,[134,246],233,285,,250,18,349,47,67,295,35,-2,66,,167,346,,272,287,,91,345,,378,-1,82,,41,,170,,62,,324,56,326,380,,164,134,236,-2,201,,133,,[321,374],186,350,,377,330,369,116,77,[161,358],,238,228,245,,99,,[35,284],350,-1,[193,213],133,241,349,331,334,-3,287,-1,124,-5,182,177,,264,-9,243,-6,[161,168],,356,376,,285,76,39,356,153,,359,116,241,330,131,-1,93,,345,336,-2,124,,325,141,,128,106,350,30,113,199,,334,131,314,167,-2,247,19,,336,-3,259,-3,350,[283,285],259,161,65,365,130,240,38,,372,130,129,,260,47,-1,179,,24,388,285,303,372,85,333,-2,315,259,243,,91,,55,171,350,-1,271,,350,199,188,-2,347,,378,233,,330,126,125,-2,259,,81,,79,,241,-6,141,350,-1,330,124,85,77,340,-4,378,355,123,[121,150],356,-1,271,326,[41,383],,345,,280,31,179,24,-1,326,380,-2,19,97,,209,,7,,171,,283,,295,,295,125,-4,62,-1,56,-5,326,61,380,395,355,,310,,110,,321,282,356,350,83,395,96,215,315,209,-1,354,322,243,,138,,230,-1,160,-1,347,6,-1,125,201,,124,-3,312,128,,350,285,-2,140,377,-2,36,149,,164,,285,,380,,355,-1,198,-1,67,-2,95,,350,-1,332,-1,176,,128,375,26,245,150,29,,217,,330,115,-4,247,-8,[14,234],-2,49,74,250,-3,137,-2,391,245,164,354,-4,298,85,,164,,238,9,[287,356],[241,295],,337,,356,-1,38,,346,[165,167],164,-1,126,340,85,,134,,226,-3,82,,2,-2,13,-12,100,10,,287,-1,58,,136,-2,306,115,350,,310,-1,129,,31,-2,146,135,,134,243,209,,387,329,180,-10,370,,377,,248,-3,346,-2,125,124,-2,268,,326,234,-4,336,-3,395,,314,-7,126,-5,389,-8,224,316,-2,[174,176],-1,81,31,153,-3,85,-1,164,-3,8,-20,55,,306,,17,39,222,-1,51,-4,4,-6,36,277,-18,334,-5,331,-2,168,-10,133,276,-3,128,126,384,-3,298,,127,-4,233,-9,131,342,-2,56,-5,264,-13,162,-28,76,330,359,,357,339,37,373,,333,101,,143,,191,-2,71,-1,317,295,346,-2,57,,68,-4,131,-1,264,-1,252,-1,5,95,172,331,,158,95,,100,338,18,241,134,62,388,,347,126,286,-1,124,30,-2,206,-1,89,255,-1,26,361,-3,185,-1,191,,262,,336,,137,217,-2,100,,378,,357,195,-5,67,-1,[82,238],18,116,37,136,,152,349,163,-2,67,333,169,-1,131,-2,241,,386,296,,167,,63,171,-1,126,,48,-8,170,,215,-1,124,78,-1,82,-5,85,,238,,139,120,-1,60,-1,339,,317,4,39,124,56,107,,364,58,63,,259,-1,81,246,350,,140,-1,39,327,-1,338,117,-2,358,169,172,365,89,42,,39,-1,279,-2,227,-2,229,164,,16,,131,98,,58,-1,148,-1,377,288,252,[28,277],46,143,,102,,102,139,73,124,310,-5,[245,335,343],,156,243,-1,98,126,-1,310,-6,324,,124,-1,[279,346],-2,[9,19],,134,,284,126,-3,85,-1,91,-9,235,,[32,128],,245,136,-4,233,-6,123,-1,350,-6,164,,252,7,95,125,330,182,169,-3,67,380,218,129,251,281,-2,164,130,160,-9,110,285,-2,19,-2,354,,191,,344,-3,189,,161,-2,341,,140,-5,199,-3,247,-10,336,-7,9,281,-2,96,330,359,-3,9,47,,339,-3,240,,229,85,-3,13,117,10,,[36,285],-2,360,-1,127,152,,332,-1,102,-2,80,-2,153,-9,233,251,350,-1,14,209,,285,,243,,269,,284,330,121,7,-2,117,336,,10,[49,391],387,,56,182,202,-1,19,,[19,20],,14,20,371,138,175,355,175,,98,329,-2,326,-1,377,185,351,326,,124,253,177,,135,336,,130,-3,248,-1,31,,74,314,164,-1,164,,345,346,282,-6,26,-1,164,,115,-1,53,,357,-1,334,346,,136,[277,332],,39,133,-3,295,-6,295,-3,251,-2,333,,271,[23,283],-4,355,28,124,355,291,78,-2,77,-1,287,,251,286,14,341,,227,-2,297,64,-1,147,97,134,171,,[58,307],147,377,128,373,5,65,143,304,36,285,355,,239,349,109,295,305,,161,63,124,-2,[341,356],,36,91,,348,,[91,110],124,63,,315,195,115,164,317,334,369,332,,176,0,181,-2,137,,170,320,351,80,13,,247,-1,[84,233],5,72,81,,107,314,[108,145],293,241,,377,351,,330,32,-1,331,43,,327,115,,330,,345,-1,74,-1,55,,35,,[90,97],85,-3,200,,85,350,55,224,164,10,357,104,,244,330,-4,205,,171,,355,,379,-7,98,388,326,,110,204,332,230,-1,283,116,195,-1,134,,360,,73,,229,-3,[360,365,369],[110,116,123],,133,354,85,,379,,138,97,138,-1,173,67,215,,295,-4,101,369,,350,169,388,193,190,377,348,124,,[91,138,180],,362,,143,121,-2,333,-1,340,,346,,350,1,237,283,,114,,73,322,105,361,-1,226,334,1,93,152,345,54,335,13,358,,117,,150,73,,124,217,198,353,-4,91,[74,223],36,281,164,200,-1,148,335,-2,374,369,-5,20,,113,94,,91,,346,-3,302,44,,1,-4,331,306,395,-1,330,-5,181,-3,160,,302,364,119,-1,124,-1,82,,116,,286,326,288,31,,164,-2,353,-3,277,,321,116,,379,,144,387,,280,-1,170,0,-4,51,-3,307,56,-4,[40,45],-3,80,-1,233,277,156,365,-1,225,,274,383,213,-5,19,65,159,335,-1,36,310,,133,,138,354,326,202,279,-1,156,129,115,107,64,-2,[222,258],,351,-1,329,-2,41,121,-1,331,110,,153,-6,342,-8,164,,[28,369],,192,,231,,153,,356,,174,148,-4,0,340,-2,298,-1,[1,350],302,-2,143,[54,303],272,34,,297,-4,285,91,,129,,328,,295,36,109,302,,106,,247,213,110,-1,75,-2,1,,315,,4,-4,297,297,,61,241,-5,25,125,-1,235,,86,97,-7,[285,341],-1,181,-1,162,69,-2,12,352,-1,190,-3,[40,395],232,-2,380,32,,393,,295,-1,168,,331,-2,111,-11,27,128,349,-4,60,-1,240,136,247,-1,175,-4,223,,130,-2,241,74,364,350,285,-3,[136,343],-3,86,-3,270,233,-7,264,109,-6,109,310,-2,[21,28],-14,335,-10,123,-6,256,-3,[128,136],-12,203,-8,203,-1,[54,147],-1,250,295,,[126,202],121,337,351,202,318,,[72,320],-5,116,,357,-1,154,47,-1,326,179,-2,171,97,-1,104,317,355,-2,240,356,,357,,252,356,248,-16,[119,357],-1,317,-2,284,-3,361,[326,341],91,,330,102,233,350,[59,61],-1,375,-4,152,31,241,-1,124,,377,,7,-3,137,-1,79,,11,305,140,118,395,145,-4,150,-1,126,305,164,5,330,81,385,239,-1,154,,305,,321,88,238,62,,209,,233,-2,4,-1,199,143,-2,61,33,-4,43,156,173,173,175,-3,338,67,-4,162,86,-6,73,-3,351,77,96,357,64,,144,280,,74,,62,,345,149,-2,139,,204,-2,357,-8,94,1,-6,182,-1,35,[279,346],-1,169,,[20,240],35,-1,285,344,104,-1,210,,356,20,-2,233,-7,55,,2,-3,377,-2,350,230,124,-1,273,-1,148,,306,-2,71,130,-2,232,,317,-2,243,-2,73,-2,351,-3,64,,115,9,-1,61,-4,140,-4,346,-3,69,-13,163,-6,303,35,-2,298,-1,303,-2,306,-4,[270,274],-5,91,-4,311,357,-16,287,,36,183,,131,,287,-2,354,-7,280,199,-4,244,167,-2,368,341,-7,197,72,-17,124,-3,13,-2,354,-10,110,-2,109,-13,256,-5,285,259,,384,,284,-1,[143,245],-1,116,-1,350,-7,[78,228,327],,39,-1,10,-4,85,,332,-3,153,331,323,-1,298,73,,349,-1,96,-3,351,123,,[54,304],,311,304,85,,348,347,[11,108],,285,-1,316,-1,350,149,[86,125,242],73,,152,-3,166,119,,346,-1,241,201,82,,81,-1,153,390,,241,-1,11,350,127,307,363,,331,-4,62,,282,-1,4,-12,219,218,,201,,126,303,-2,109,-2,294,264,80,325,,384,85,181,,259,-1,346,-3,69,124,-1,348,130,,192,-3,216,-3,13,356,321,-1,79,330,356,102,-1,209,379,-1,54,-5,187,,241,-1,246,-6,199,-2,388,285,-3,129,97,295,338,326,-1,279,237,,348,,127,287,-4,128,,160,-1,350,-2,124,-11,28,,351,-2,388,,326,,322,174,345,257,128,177,238,-1,164,-3,302,,326,-3,211,-2,[200,222],-1,135,283,-1,61,,74,-2,191,-6,356,322,,333,,251,-4,31,-2,74,-4,239,-1,324,15,-9,129,-3,122,-6,13,-3,131,-1,158,-8,352,30,283,314,-1,330,-3,341,-17,187,-6,187,357,-13,4,-6,233,331,,352,-1,36,-4,96,-1,125,273,-5,124,-1,333,-4,1,,237,,235,,162,-2,372,-2,164,61,,346,-1,31,-1,207,,197,-4,244,-22,331,-11,13,279,-25,212,,352,-1,197,-7,291,-14,[361,388],129,,136,146,359,,388,52,301,85,10,388,335,,189,-1,9,124,97,218,343,-1,106,177,,204,-2,287,[23,30],,388,,85,-3,264,-1,213,-2,215,,303,102,370,,356,286,2,,296,324,-1,114,-2,190,,56,,389,100,379,65,324,350,9,285,,37,283,143,342,285,355,119,-2,333,95,-3,361,-1,106,346,335,125,,34,262,-2,151,17,298,-1,124,,124,351,190,147,-3,85,-2,187,,107,356,-8,247,197,28,,98,-1,330,168,-1,370,-4,168,,119,-6,52,71,295,344,58,-1,286,,82,-1,282,,[127,244],-1,[326,356],394,-3,335,,281,-1,76,86,126,,34,,280,,86,31,-3,355,,355,168,,348,-3,86,-3,133,-1,88,-2,285,351,36,142,209,130,326,212,134,233,27,-1,134,-1,251,310,129,330,-1,285,[18,238],124,336,-2,371,-2,74,-3,[287,380],,317,176,,331,,176,134,-7,[320,386],,279,-6,350,241,-3,356,,300,241,-4,345,-2,124,243,-1,241,-2,28,,251,89,333,4,158,58,5,-4,143,96,-4,346,,340,125,171,-2,55,,358,,116,,194,2,-3,67,,153,,185,-1,350,-5,344,-5,67,-5,377,-6,332,-1,[128,245],376,177,,74,,356,-1,245,-1,82,-9,137,-5,160,157,-1,146,37,-5,241,-4,51,,345,-3,136,-1,104,-4,335,,346,296,,12,-2,371,97,-4,[323,326],-7,361,348,-7,124,-1,284,187,243,-10,262,-1,356,,174,-10,296,,388,326,-1,53,-3,124,-12,372,-21,170,-2,60,-24,350,-12,62,-6,326,-14,41,379,-1,344,32,-1,95,395,245,134,95,,330,-1,[28,29,46],250,,124,350,295,5,-1,334,-4,344,130,-1,13,285,20,,289,78,,285,-2,331,-1,326,372,306,-1,230,225,313,,166,377,379,19,377,61,-7,55,375,-4,331,8,-1,38,-4,326,31,-1,365,104,185,-3,190,,329,85,-5,120,-6,183,199,-4,372,-1,85,-1,78,[42,384],-14,88,238,210,,18,,338,,348,119,355,355,,101,233,-2,384,,248,13,-1,42,102,-2,341,175,330,,148,352,61,229,62,,192,94,,85,,228,80,334,-3,340,[69,73],395,-4,314,-4,13,,2,287,141,354,321,-3,356,-9,166,159,-5,130,4,155,-1,168,-3,30,-11,336,,170,-10,346,314,-1,126,-3,210,139,14,350,241,217,-2,350,,13,350,-2,285,,285,,95,,351,,85,114,-1,36,-2,61,372,-3,190,333,116,,218,131,-1,190,,324,,268,-6,[56,305],[127,244],,13,-2,96,-16,102,57,,175,318,121,377,-3,350,-2,[236,271],338,,315,,346,-1,22,15,,17,232,-1,372,352,,36,-2,228,,350,,33,13,-2,325,376,49,,131,55,,344,112,347,118,176,115,-2,341,,317,-1,157,,59,226,331,-1,31,-3,356,-5,120,,344,-1,348,326,-5,376,,59,-3,128,121,-4,337,,13,350,-1,160,-1,124,259,,30,305,308,[308,319],-2,52,377,325,184,-2,311,-1,378,37,-1,330,-1,355,-1,[296,378],150,,14,377,-1,34,-2,210,-1,216,-1,337,,116,139,81,118,304,296,330,224,31,42,-2,32,-1,395,367,347,-1,218,,376,225,20,-1,116,,54,166,295,-1,55,350,-2,124,232,338,357,209,99,85,-2,246,-4,39,-4,389,71,-1,350,,207,344,285,-6,166,,120,-1,146,,113,-1,287,-2,347,121,-1,125,-3,121,388,341,-1,219,112,75,143,67,311,95,,331,242,-1,144,-1,[74,330],,315,346,139,26,204,359,,354,-2,154,245,-3,331,-1,310,107,-3,350,,121,,10,-2,296,252,,153,,330,355,-1,119,-1,358,,214,-1,253,342,243,,194,,80,10,,51,-2,124,-1,338,58,-7,248,38,-2,131,324,-4,123,-2,310,-1,325,,116,,331,[31,306],,326,121,-4,62,134,23,35,,10,246,23,56,100,73,-1,334,385,,120,-1,258,338,-4,245,38,-3,243,,356,356,-2,194,13,350,-4,74,-3,356,-3,88,359,-1,163,81,,153,153,-2,298,-18,357,-7,46,,[243,246],,120,,283,-1,282,-2,199,-6,308,-3,183,-3,121,139,-6,326,-3,354,,141,-18,16,-1,368,-4,245,-7,71,-8,37,107,241,-1,131,-9,39,-5,107,-2,67,-4,336,,4,185,-4,188,158,,170,-16,222,-13,189,-8,350,-5,89,,91,,330,126,341,287,262,331,35,329,129,-1,244,123,244,371,,241,125,-2,124,,140,124,,87,60,-3,129,,126,,175,-3,45,55,-1,116,-4,116,164,79,302,14,,132,-2,279,116,80,,286,,22,369,-1,240,[5,225],54,260,-2,321,-1,141,-6,147,-1,243,377,,155,188,273,347,216,7,33,257,-1,7,-1,85,-1,233,-1,74,,373,35,124,-2,32,-3,136,5,-5,350,287,381,,316,68,141,374,,85,229,-2,318,147,179,244,,116,9,-1,232,-1,233,304,-3,9,-2,61,-2,[181,197],,34,345,38,,194,-1,85,,380,56,29,199,210,156,85,,7,226,170,,99,,134,321,5,,321,[4,216],134,387,227,373,6,-1,209,-1,173,126,,354,158,215,19,365,-1,155,285,129,376,,95,,252,290,,360,142,-3,237,[349,382],285,200,,36,98,-3,377,246,2,-3,149,-1,312,-3,322,-2,377,177,,[329,381],303,336,204,57,128,376,124,121,-1,1,,221,53,-2,375,-1,[269,277,302],-5,350,314,-1,324,-3,330,137,,315,154,-1,387,,176,5,107,281,213,135,-3,20,-7,160,301,,126,119,58,-2,232,,169,-2,6,134,-1,1,-3,43,129,-1,200,210,-3,333,,62,-3,73,286,63,-1,372,,124,307,242,,226,-2,349,,136,-6,178,,305,33,,129,,146,319,346,53,,134,311,243,,6,-1,175,[99,104],-1,377,56,,30,,100,,357,-4,356,,153,-1,263,-2,390,342,192,310,,28,-2,350,-6,2,,345,329,,40,-4,139,133,-1,129,-3,[64,282,349],,357,,349,,369,-3,158,,247,30,91,174,-1,128,-4,40,371,-1,19,39,-1,53,273,-5,126,297,,90,,294,272,-3,222,-2,306,,7,54,-5,243,-4,336,-1,28,,91,,75,-1,282,287,6,348,17,,305,-6,18,,289,-2,370,-4,180,-7,51,,197,-2,377,-9,197,197,374,-6,168,,352,136,,236,-8,35,269,-1,295,107,-1,394,-3,384,-4,33,-3,168,-1,245,19,53,,383,-3,210,,336,175,,50,,107,-2,88,,162,-1,279,-6,25,248,,119,,247,-3,233,19,-4,297,-2,387,,338,,21,-24,227,-7,123,-7,[50,362],-4,256,-11,392,-4,136,-1,203,377,-4,[78,228,327],286,,355,87,,95,-1,79,376,-4,97,-1,335,190,-1,61,-1,194,,133,-2,36,4,-1,128,,166,,13,31,-2,88,271,-1,[71,72],-3,128,131,-2,[287,294],,245,,376,-1,85,-14,327,-2,370,17,-2,80,7,,158,-2,68,,168,,116,336,-1,375,,329,-1,130,36,,85,,371,,387,,70,,295,337,-7,79,-1,356,285,-2,228,,371,185,176,230,-3,342,131,,209,391,-1,172,-2,350,241,-7,330,,124,-1,261,56,133,377,364,-1,344,341,86,156,88,107,-3,285,152,-1,325,-4,359,,3,365,-1,154,-2,109,,31,,195,122,-2,350,331,337,-2,305,-2,362,,338,352,-3,44,,187,395,-3,373,,285,,356,-1,185,209,31,-4,379,,333,-1,32,,120,-6,130,280,-1,346,-1,278,335,349,359,121,107,-1,324,-3,284,,20,-1,330,,121,,34,-4,240,131,331,-2,248,,131,102,-1,377,-2,167,-2,362,,342,-1,332,-3,153,-3,287,-3,220,2,-4,195,-8,1,124,-4,199,-4,[9,240],-3,333,-3,320,-25,287,,344,348,[9,240],-7,331,-1,203,-5,358,,251,349,94,-1,110,,25,-1,183,,[27,368],310,393,-6,358,355,265,232,,85,-1,251,-2,294,375,,159,-2,325,,[32,373],,241,-5,189,,199,,326,197,11,369,,287,,380,-1,240,73,-3,124,-1,340,-1,277,,360,252,-1,88,,28,279,-1,330,,164,338,22,52,,281,-1,372,-4,69,,241,287,89,312,-2,157,-1,347,181,192,-1,108,,10,129,,90,348,,39,,225,-3,380,,296,7,,124,-1,96,-1,233,,325,,79,-3,331,-3,375,,170,266,,187,,104,377,47,-2,287,364,,164,,126,35,,244,82,,335,,148,238,-1,377,,335,125,125,[96,134],-9,171,18,-9,[6,19],198,88,241,255,263,-2,374,332,355,,102,321,[369,395],,215,,61,-1,28,-2,133,-1,126,-1,143,201,380,,172,29,-7,35,,285,377,-3,[279,369],,15,371,377,173,67,175,,[164,358],158,,287,,290,-1,[241,331],164,-4,98,-2,[128,335],-6,341,380,-6,142,160,-2,347,110,93,-1,91,361,177,,129,113,102,307,101,326,152,,2,2,135,,387,,377,,315,272,,119,129,133,-9,345,257,375,57,241,245,117,[102,121],127,384,,302,-7,85,,315,136,-8,167,,314,,187,8,-7,97,-2,388,-2,94,-9,281,-2,189,330,164,-3,302,,310,336,-1,287,,78,-9,126,,171,-5,191,,241,,103,-3,8,-1,389,-1,124,,232,-4,306,-1,61,-9,275,,43,163,-2,143,-2,373,100,,81,-3,104,-2,350,-4,167,,377,[43,385],,134,,128,-11,69,243,-2,180,-7,321,-1,349,-2,70,-3,[283,375],-2,41,,44,-1,[28,369],-16,336,-1,126,-1,39,-1,166,163,,202,-1,187,-1,342,-3,124,-1,39,-1,48,-4,139,250,352,-1,174,-2,237,,87,,158,,356,34,176,134,-2,336,-5,159,262,-1,134,-1,375,8,-9,80,369,-1,301,,336,-2,51,-1,172,-1,253,-2,303,-4,90,-7,294,,43,,28,,118,-2,90,-5,[126,140],-2,[17,18],380,-11,241,-3,116,-9,25,,130,-9,78,-11,39,306,-5,372,,197,-5,331,,113,-3,244,,352,-2,245,-6,394,358,-4,88,-1,245,-7,321,-6,134,35,,136,-4,334,-2,380,-10,39,-6,175,-1,357,-2,305,-2,331,-8,247,,346,162,-4,19,-7,248,-7,170,,28,189,-114,243,46,119,337,356,-1,224,-9,356,-3,331,,350,241,-2,151,-3,277,-1,337,336,,243,-1,91,-11,[282,331],-7,377,376,46,20,330,241,-1,323,-13,55,,295,126,-3,197,49,347,311,,55,280,,344,287,23,,235,-3,359,166,-1,377,-2,56,124,-4,17,-7,350,-7,287,224,70,,[346,351],-6,62,,121,97,-1,350,-4,330,,199,,187,-1,69,356,13,13,13,233,,13,,185,-4,371,-7,199,109,-2,305,-2,51,,271,-2,287,-2,126,-6,31,240,175,-4,251,,285,61,194,,184,241,236,201,,58,333,41,81,-2,85,,67,248,,351,,106,347,2,345,143,-2,56,176,248,,359,,292,-2,354,-3,320,-1,314,377,250,-2,50,121,,107,28,-4,331,-2,241,279,,107,-2,344,295,264,95,127,36,330,-1,306,-2,190,97,325,-4,304,,124,,14,,327,-1,339,-2,241,81,-1,247,350,329,,357,108,,[34,283],34,-1,72,-1,241,199,-1,[54,303],191,-3,277,,230,-2,96,,[187,197],,82,224,164,179,24,,326,116,197,,287,134,-1,321,73,110,-3,80,355,,377,373,97,371,346,-3,336,250,-2,252,19,,190,-2,161,-1,114,88,77,185,295,-2,78,-1,215,,171,229,19,241,,209,-1,380,,162,342,-1,227,194,304,347,,354,,12,343,[173,280],175,,180,336,239,365,131,-1,129,-1,121,-1,351,-2,347,169,,124,-2,269,-3,331,,380,127,180,-1,67,-5,130,,326,-1,114,341,-1,307,-1,76,379,264,,344,-2,119,,123,322,242,226,-1,172,,125,,243,127,128,375,,387,26,,150,124,172,,122,116,217,344,-3,374,[137,344],,8,,387,,331,-1,119,-1,240,-1,109,159,-2,85,-4,356,-1,106,130,-2,187,-4,317,-1,213,-1,335,282,-1,354,,333,-1,298,,135,,310,-4,307,,160,157,166,326,329,,119,61,,267,126,372,274,85,100,-1,290,345,-1,349,-1,107,-1,110,-5,387,62,-2,388,331,335,241,-2,170,306,-3,287,-3,204,,307,47,-2,80,296,,233,56,-1,356,,88,-3,351,51,,118,-1,283,,44,-2,122,,346,,311,192,-7,248,-3,357,,175,388,69,,126,[191,284],-1,356,,283,283,-1,380,-1,356,-1,251,69,,369,19,329,-2,327,,336,,326,,89,-1,342,,143,-2,355,,192,-7,226,187,-7,318,191,-3,231,126,,116,,334,-1,371,-2,120,-10,[128,250],-1,346,-14,324,285,-2,153,,126,341,-1,87,-4,306,357,-6,143,-2,172,-1,195,-1,350,-1,240,,164,-1,331,-3,298,,247,297,,340,,262,122,-1,209,303,,264,-2,39,228,-3,62,-2,388,-2,85,,117,,388,307,309,,13,-1,103,-2,377,346,282,183,352,,176,158,177,,17,305,-9,61,-3,116,-7,235,-2,241,-1,175,-4,174,-2,164,346,25,-9,197,-2,158,-3,342,350,183,,184,,[180,303],,287,,372,-1,119,-6,347,-6,352,335,-2,166,,326,-9,227,-2,243,,175,-2,120,-2,[160,168],-5,305,32,-2,281,,380,-2,279,,30,,315,-6,35,-2,33,88,-1,295,287,232,-12,158,-3,364,-4,164,-10,4,-1,30,-1,[56,305],-5,124,,166,-5,300,-6,157,17,-10,13,,109,264,-11,240,387,-32,[9,240],-7,107,352,-6,336,-16,81,-2,358,-17,100,-1,109,-13,5,-11,123,,193,,60,121,-3,171,364,,133,-2,387,,361,23,347,-3,102,-2,175,43,-2,346,-2,32,,253,141,72,-1,377,-1,326,244,-12,342,134,305,[9,229],132,,304,,18,-2,380,369,62,,285,166,36,-1,116,294,158,314,-3,169,,347,-12,114,[160,180],,380,-2,346,-3,142,,78,281,349,121,,306,130,258,,331,-6,324,,232,-2,82,-10,346,107,-4,330,-1,107,,119,188,,58,10,81,-10,128,-7,32,346,346,-3,255,-13,70,-3,342,,120,,126,-12,356,,277,-4,187,,341,373,326,-4,380,-2,9,116,-3,14,,319,-1,279,-5,331,-4,339,-3,344,-3,262,-2,172,331,-4,287,350,-6,[356,359],-2,4,-5,183,,280,-2,331,-8,255,-9,168,-4,78,346,-9,356,-3,364,,300,-5,336,-9,333,-11,9,-21,136,-9,50,,[373,381],,225,-2,357,1,-2,136,85,349,5,64,,348,,291,,227,-6,234,7,-2,226,69,-3,64,-2,355,-1,345,,216,,237,,[198,199],,199,160,-2,185,199,,330,-3,126,97,-1,284,-1,243,-2,308,331,-2,330,97,331,164,-1,124,-4,69,134,,126,,234,-1,142,-11,127,-11,252,252,,78,250,-2,[2,107],,384,101,181,355,-6,359,152,216,61,-2,10,-3,229,332,,116,,80,-3,96,,134,-3,215,,112,128,-5,262,286,-1,69,332,285,150,376,356,301,356,13,-1,135,164,,351,299,159,-3,164,,333,-5,209,-2,169,-3,104,,190,31,350,-2,189,22,49,282,-1,180,116,-1,326,-2,338,380,185,326,,333,-1,204,-1,115,-1,355,187,28,-3,117,357,-12,131,-1,372,,4,-3,136,-7,168,-10,336,303,,344,-13,119,-4,342,-1,[176,289],,356,,325,-1,65,-1,124,-3,133,-3,181,-2,194,-1,17,-5,324,,187,-1,326,119,333,,171,55,-2,62,-1,331,19,-3,239,,143,-4,125,194,279,-1,375,,136,77,173,-6,95,-3,180,,380,-3,76,,348,,113,-2,7,-3,[121,122],-15,250,,159,164,-1,172,345,-3,166,302,-8,134,34,-5,387,-1,116,241,,154,,241,,324,,47,,346,-1,170,247,233,225,-4,249,-3,185,-18,332,,357,,204,116,352,356,,266,274,-12,306,-1,102,-4,348,1,-5,130,,51,47,,164,-2,342,-2,372,-1,352,,175,-10,120,,240,-7,13,23,251,-15,327,-22,362,-8,98,,64,,116,,235,7,256,,322,-6,328,224,-3,171,,238,46,-6,20,-3,375,-7,189,,12,,368,,233,-3,88,55,283,,311,-1,284,-4,284,-1,354,289,,85,354,12,,215,311,355,125,283,,62,,202,62,,65,117,-1,361,,13,-3,31,-1,78,-1,311,129,,252,326,-3,227,-3,172,11,375,[39,341],-7,178,241,-2,78,-6,[282,356],,38,-2,124,324,-4,124,-2,318,-1,127,-3,233,-2,287,,350,-1,65,,129,168,,91,133,-1,279,,221,164,347,,355,5,129,-4,350,164,,42,82,380,229,233,88,143,46,-1,56,375,-1,309,134,124,-2,125,342,369,18,,376,354,131,252,-1,350,-3,347,,377,112,345,,68,-1,315,-1,233,-2,164,377,53,,330,277,160,,119,333,-3,305,80,-1,36,-3,13,-1,97,-1,326,356,51,,380,,56,-3,115,,156,-2,356,-3,125,,350,174,273,,36,-2,327,124,,7,,172,[29,53],286,-1,54,16,305,,15,-1,180,-1,38,372,370,-1,253,-2,197,,351,352,120,-1,173,-7,1,7,-5,350,,233,-4,62,,157,-3,342,-6,62,-2,251,-7,102,-1,60,,6,6,-2,364,,[59,61],,129,120,102,-1,90,-1,128,-1,1,,109,-1,324,-1,331,-9,239,-8,233,-1,379,137,,52,-9,194,-1,356,-1,378,231,,352,,350,-1,110,3,371,346,126,110,,153,,87,58,227,-1,284,-2,189,-4,100,-7,199,65,,341,184,,377,-2,334,72,-1,227,,72,-1,[284,338],-1,191,-1,192,56,187,,140,,147,-10,350,284,-3,375,191,,357,-2,388,,369,342,-2,300,,190,-4,36,152,135,198,,312,,346,-2,[373,374,387],376,,302,-2,61,-7,157,126,-7,345,131,-4,292,300,69,,13,199,,209,-1,129,22,-11,69,-2,153,90,266,185,-2,192,38,-5,143,297,332,-1,195,183,-11,235,35,,189,-1,236,,245,,380,60,,293,-2,140,-1,315,,170,-4,371,,97,,251,-12,136,-8,39,-2,185,[100,130,247],-4,285,350,,377,,283,,134,,128,53,70,1,-3,[56,285],-1,124,,88,-3,331,78,152,57,181,277,-5,117,-1,241,140,-2,345,233,,346,,383,33,72,,346,,82,77,,369,-2,321,,61,370,375,,80,-1,156,14,-1,232,-2,239,-1,283,360,1,164,173,315,,164,,39,-3,102,,204,-3,91,,338,,67,,331,-1,294,332,245,-4,335,-11,329,172,352,184,253,-7,126,-3,232,-9,65,,63,-1,175,1,300,-1,10,,71,-2,324,62,,10,241,34,-1,64,,379,,129,,14,,13,-7,232,126,61,305,28,-3,342,-3,210,-1,46,-2,[8,228],-2,272,162,53,-3,227,-2,374,143,-2,103,-13,197,-2,248,-4,251,,60,-1,170,-1,120,-5,128,-2,72,-12,127,-9,189,-4,21,-14,19,-4,285,285,164,,282,,295,241,-3,333,,241,377,-8,85,-1,391,61,-1,395,251,116,380,283,300,46,,190,-1,334,,312,235,-3,124,,375,-6,58,123,,241,-4,18,130,-1,175,30,-3,331,-3,85,-9,387,-11,331,-10,256,-4,356,356,164,,247,110,,340,295,,317,-1,88,-1,18,,250,,378,-2,143,192,-1,13,-1,190,-5,391,-2,197,35,,247,347,,377,,287,,388,,124,[34,35],-6,129,-1,350,,121,-1,331,,159,85,-1,176,-3,35,,281,292,-4,259,-1,6,-1,377,-2,143,,38,-1,298,-14,327,-2,124,,375,,58,125,124,,90,-5,199,-9,274,-4,300,-23,256,-2,343,,133,249,331,249,146,-3,41,386,317,,246,370,,14,,348,-3,245,-2,348,377,-1,312,128,42,132,-2,50,329,,148,143,-3,153,68,,[344,351],-2,356,134,-4,356,-10,173,-10,164,-9,287,-1,371,-3,131,131,372,-1,137,,315,296,-5,129,,70,-8,380,380,-1,356,,88,-2,69,124,,5,,124,,373,301,-2,116,,335,-1,13,126,-2,284,,61,-1,36,,164,-2,312,295,85,,11,,54,388,61,,[365,395],-1,96,,125,-3,239,,126,,173,,14,-5,333,,249,,60,,130,252,-1,77,152,380,315,,54,,26,,147,,13,278,,376,-1,[137,359],,225,-10,285,-1,335,281,-1,346,,150,,38,,89,-1,243,,126,-3,13,-6,97,-1,131,-2,19,124,,299,-3,146,243,-2,100,357,-2,365,246,321,180,56,335,268,126,-2,334,-1,375,-2,380,-7,120,-3,383,234,-3,115,-3,153,,174,-4,90,80,-1,96,-2,50,-2,164,13,-2,36,-2,158,-1,164,-4,232,-3,68,,193,-7,49,-2,102,298,-1,175,-5,70,-7,62,-5,60,120,-1,362,-12,19,-5,20,379,157,-10,124,-36,190,61,-5,162,333,388,-10,81,-6,5,164,-1,239,,49,[210,371],-2,312,164,331,298,-1,388,,358,[356,379],-3,81,-2,167,-1,167,23,131,-4,51,-1,170,389,131,-1,[271,283],-2,263,,115,341,,116,,362,46,-6,90,306,250,,25,-1,[187,190],-1,364,141,-6,127,-5,222,-10,[124,331],-13,327,-20,298,,302,-3,130,-2,388,-2,162,-61,341,-7,377,-45,241,-6,248,-139,190,-4,78,-4,[348,355,379],-57,392,-23,58,-3,133,356,114,379,[243,333],91,358,124,324,152,124,259,326,359,,44,233,277,89,200,,389,[100,179],81,377,327,79,-1,216,287,333,88,336,85,166,391,283,331,377,378,379,7,85,39,281,350,131,55,8,262,129,148,257,,108,121,[92,124],342,127,180,136,128,315,94,335,135,340,,300,307,124,310,124,341,171,,341,241,80,[32,45],280,103,284,326,191,286,12,38,,172,252,[368,389],371,324,176,385,388,143,334,126,191,158,310,192,[124,241],,121,295,73,70,14,,96,385,119,61,176,14,194,357,130,85,264,375,82,,90,30,164,350,126,17,235,183,162,352,[298,302],[192,196,198],273,336,168,279,368,127,243,245,119,128,392,84,,89,,253,-6,352,,248,332,-9,100,325,-1,325,107,,180,85,,77,-3,97,,89,5,-4,346,373,393,-2,377,-1,158,287,,233,-3,164,-3,368,-1,124,-7,347,,244,,187,-4,90,-4,171,-1,61,340,244,,333,-1,254,-1,302,-6,129,306,,331,-4,162,94,-1,30,356,,350,,328,-2,36,-3,350,,350,,171,-4,334,331,-1,245,-5,[61,370],51,80,-2,380,126,-1,234,-3,110,,107,4,,350,-6,78,350,-2,348,160,,142,185,374,241,-3,64,76,288,-1,201,,162,,388,94,32,109,359,[5,225],-1,295,-2,123,,160,-1,306,224,174,,127,228,-5,197,,76,,65,349,54,296,-1,36,,56,,94,-1,213,56,-1,171,-2,168,173,377,215,-3,104,,166,-2,237,,134,-13,153,47,-5,4,-12,356,-2,298,350,,295,373,,263,,162,124,-5,120,-1,335,,379,,69,89,262,88,,329,31,97,377,,85,80,-2,126,79,386,,200,3,144,,95,356,,348,-1,[243,341],-1,80,,131,304,283,378,372,336,283,326,379,,56,-4,10,98,304,-6,[227,228],375,,395,230,125,284,377,9,-1,116,-1,351,341,,173,67,138,175,131,218,346,,149,350,101,106,91,67,,128,,339,,2,,234,208,-3,377,-2,51,-1,[182,197],124,-1,150,269,363,241,204,190,217,177,,19,,324,,128,-2,53,-12,229,-1,85,,321,212,-2,180,,166,-4,233,-6,311,,131,,156,349,[0,346],-2,85,85,,80,244,324,-2,389,65,-4,48,-3,338,,287,,202,-2,74,270,,348,126,-1,356,-3,85,333,209,191,322,309,319,8,,176,-4,91,,19,-3,90,,17,-5,15,-1,306,197,331,-4,377,-1,164,,232,,40,-7,279,-5,352,279,-3,320,166,13,354,-1,350,-2,273,,97,-21,34,-2,363,-1,388,,213,[38,340],-4,377,377,-5,375,133,,356,,348,356,37,,331,133,356,-4,282,282,-1,285,,287,,311,-5,41,293,,330,379,,41,279,350,-3,7,,108,79,7,13,,378,126,24,,380,365,73,19,333,91,41,,175,-1,331,-4,281,-1,314,,326,,189,-2,286,-7,297,,25,-3,37,-7,189,-4,93,167,126,,274,346,85,-4,350,-1,128,[1,350],201,-1,129,,324,-2,190,,243,356,,281,339,69,,241,184,-3,334,[13,233],85,-1,330,377,,279,,243,-1,147,[87,129],175,,124,247,241,[346,357],81,5,266,337,124,117,,79,330,-1,377,359,247,-2,345,[80,85],-1,47,-1,14,-1,326,164,,74,333,31,24,380,298,,357,255,171,304,[281,312],,192,248,-1,143,199,,9,96,194,350,,[134,251],236,268,148,-3,279,-2,11,,352,-1,134,-1,88,,238,,85,-4,387,185,78,[125,246],185,185,5,[46,388],197,-1,36,,131,173,,212,-1,352,249,,195,-1,351,93,[243,331],-9,126,46,-1,184,28,128,,85,,380,,127,121,351,28,,262,264,37,,315,-2,344,-1,252,-1,131,338,,25,-4,259,126,[310,350],120,,164,-4,125,257,13,,245,121,124,57,,262,122,[338,352],180,352,[243,344],130,301,351,182,114,379,348,-6,110,13,-1,61,317,300,-6,164,[85,240],-1,164,-3,[277,302],-2,134,187,-3,[283,337],-1,314,355,,[100,324],,355,-6,167,[85,235],74,197,-3,140,157,166,285,329,,333,123,355,352,352,,44,184,-1,324,131,-2,126,,97,-1,134,,137,-1,110,-3,85,,31,,296,-2,22,5,,317,19,107,-2,331,-2,240,56,-4,97,171,80,-4,346,391,-2,287,-2,241,,51,307,,13,-3,241,189,238,326,56,,[119,385],-5,201,-2,313,180,-5,352,352,352,335,269,-7,342,-5,356,-2,74,[156,180],-7,9,-9,82,-4,380,,334,283,91,242,-3,240,,67,-4,225,,116,363,125,-2,47,,326,,153,314,,331,,241,-6,61,-3,30,139,153,,127,174,,226,-7,159,,189,-1,299,-3,8,-12,150,-1,240,,294,-2,376,126,124,-1,75,,109,,375,-1,341,-3,262,328,-1,285,-1,264,302,,10,-1,13,-5,158,,124,164,-1,356,,352,197,-4,232,-2,331,-10,168,-1,341,-11,298,-3,193,,[183,324],-2,374,-1,326,-3,49,,22,-8,210,287,-9,244,68,166,170,147,1,13,-4,78,-3,344,128,266,-10,251,-1,121,-3,393,-4,266,-2,136,-8,241,,359,-3,328,124,114,-1,162,-4,[9,19],109,,326,-6,350,-10,343,13,-6,336,-2,114,,337,-2,297,287,344,-5,264,,304,-6,90,-6,[124,129],-4,24,192,-1,333,,224,-5,164,-6,309,-3,78,-16,364,-2,123,-4,113,-10,197,-3,213,-15,251,,78,-13,371,-2,209,-16,116,175,221,176,-1,243,-4,341,-2,356,-2,104,-7,37,250,,124,-1,285,-6,114,121,189,91,300,332,29,285,350,181,-6,327,266,[8,12],233,-4,351,,23,-5,345,109,-4,95,364,-1,129,85,36,68,,333,-1,250,355,369,-1,355,110,107,-2,255,-4,380,-2,97,,251,[282,350],,171,97,56,-1,164,35,251,198,-1,121,-2,347,322,,380,-1,155,128,-3,[91,105],-3,249,-2,377,183,,374,125,204,295,241,-2,[281,335],354,-3,74,,287,,82,283,-2,374,330,85,164,-1,39,-3,346,314,,319,,329,377,,80,,190,,241,-1,156,189,244,,331,-2,312,-3,356,-1,233,-1,350,-1,252,-2,248,-1,104,324,-6,352,104,30,-1,143,,336,-3,356,-2,120,,14,-5,85,-1,355,[250,355],185,-7,85,,153,-2,116,,64,-3,202,174,44,262,-2,159,228,,331,-1,357,-1,297,,262,-4,247,-2,306,-6,195,-7,183,-1,[285,374],25,36,,4,-2,306,,235,-3,180,-1,378,-1,289,-4,181,-2,185,331,-4,372,184,334,-6,232,-3,227,-3,120,-3,121,-2,279,-9,336,-3,30,-3,180,-5,166,-7,189,,264,123,-3,193,-4,164,44,-14,135,-5,69,-1,251,-2,343,-2,219,337,-5,[108,338],346,-5,333,-1,129,,345,-6,113,251,350,,20,-1,15,28,,279,34,,103,,[51,289],,200,-3,378,-4,259,247,130,,357,187,,4,212,-1,125,-1,55,,229,-3,305,-2,340,-4,322,-4,227,,185,-5,[10,233],,331,-2,85,-4,242,-3,91,-3,22,169,-1,384,57,-1,144,-3,35,-4,350,356,-1,250,254,-4,277,-1,166,166,148,126,-1,13,-6,73,15,,[31,280],230,-2,180,104,-1,[310,331],,134,-2,98,-6,14,-4,110,,9,9,-4,10,39,176,-4,54,-2,264,-3,[319,320],36,-3,243,-2,158,,374,-9,244,-1,334,-24,13,130,-5,264,-19,227,-2,331,,348,,[247,305],-1,85,-57,[126,333],100,,102,190,285,30,158,136,124,331,61,,356,96,130,251,128,-2,136,-2,97,,[388,393],,280,-3,129,,95,39,-3,298,-2,377,-2,116,-11,346,-5,114,-53,388,-8,164,-47,371,-14,356,309,-7,285,-114,126,-29,248,-62,131,-4,233,-51,124,65,85,259,124,129,114,307,256,279,241,,344,350,344,124,,127,121,224,134,345,[200,205],341,74,179,,296,82,282,79,136,376,97,110,238,391,285,,369,298,375,61,379,46,251,373,,350,350,152,162,285,98,285,129,121,35,380,283,117,56,96,252,102,344,350,376,87,334,28,122,341,,129,330,356,245,330,90,355,121,152,[292,294,358],296,,248,380,390,222,69,387,80,143,326,356,292,283,[63,312],30,167,386,300,305,,350,198,34,64,120,126,336,343,349,326,74,356,342,30,388,2,346,61,190,234,,197,57,298,336,348,8,285,243,190,130,183,374,126,196,305,367,245,158,240,136,346,243,371,34,97,-8,123,-3,68,127,,36,-1,324,-7,285,-3,320,-5,334,119,-6,109,356,-6,17,-3,377,9,29,-6,63,-1,340,-2,[109,110],197,,185,-6,233,-2,197,-131,10,375,85,,95,22,365,333,6,372,123,377,78,305,237,14,96,380,100,76,126,13,285,313,102,152,55,185,80,110,350,366,377,[97,125],121,388,170,175,363,388,87,130,250,375,157,282,85,69,124,287,280,46,-1,94,230,56,157,,385,85,383,270,365,346,362,,368,279,352,88,36,,282,202,-2,110,,374,-1,390,-1,133,85,373,88,241,-8,34,-1,[134,246],32,-3,358,251,-4,388,-1,169,-9,306,-2,251,-12,362,,391,225,9,-1,72,,85,,346,,377,303,-2,358,244,,304,-3,5,64,,321,125,,229,-3,377,85,-1,279,19,,134,[164,180],93,-2,333,128,-2,149,,102,249,153,-1,175,-2,312,,126,,245,13,333,73,124,,124,-6,343,-2,167,354,,38,-1,303,-3,45,-7,118,134,36,-1,310,19,-4,22,389,,377,,62,377,,73,-2,378,-2,40,126,-1,134,-1,336,234,263,,310,-1,126,58,53,331,303,-4,213,227,-5,49,-11,12,-1,16,-1,60,27,,78,39,,72,,49,,136,-4,240,-1,50,,364,-2,380,,39,-6,170,169,-2,30,-6,392,,336,-11,283,95,-1,251,-1,73,-6,306,-18,326,-150,[33,134],[86,345,369],102,342,,259,383,74,179,265,114,97,143,175,379,377,350,116,375,164,348,248,285,361,377,128,,252,175,128,374,85,167,210,10,121,103,325,,45,388,48,85,124,,287,230,357,332,371,175,374,170,337,97,,46,[13,233],-2,156,-3,14,14,,14,-3,34,264,-6,14,-2,168,54,-1,243,356,,241,344,,104,182,-4,352,,359,130,,345,78,330,-1,[106,119],374,,130,357,326,166,36,-1,312,,350,132,125,,55,76,61,239,,64,365,,287,-1,131,-2,190,12,124,-2,385,-1,319,296,285,307,228,115,209,-1,342,344,20,,335,,316,380,250,61,,317,,310,68,-1,315,101,,285,35,298,364,254,82,-2,164,-5,119,55,175,-4,153,326,,350,-1,180,13,,356,-1,72,300,,41,-1,356,-4,14,,74,332,120,250,58,-2,350,96,-1,172,-5,303,243,,348,-1,4,-3,364,374,-4,170,394,-6,134,,13,348,,336,300,-3,192,-1,156,-4,350,,60,,354,,107,,184,,249,,152,-2,79,338,200,-1,8,-2,336,,330,,355,107,304,250,,233,18,281,10,,61,390,349,170,-1,380,-1,356,-1,246,377,,[119,344],-1,128,-2,159,125,150,376,,359,-2,331,-2,85,90,109,-2,137,352,-2,164,359,20,-1,233,,104,,305,-3,34,-2,56,-3,[68,69],346,-2,74,,135,-19,13,-3,351,-2,346,372,-10,279,,239,-6,390,-8,171,-1,82,-1,355,65,250,387,230,379,350,88,,133,-3,341,-1,81,-1,359,304,-1,321,[49,395],107,97,298,-2,195,160,,38,,315,377,333,127,35,,317,128,187,148,299,162,-2,346,278,211,-4,230,,44,,393,,49,154,310,-1,116,341,338,-6,190,252,-5,106,306,-5,160,-1,20,128,331,-3,164,134,-3,344,-11,22,,355,285,,164,[37,378],349,167,,130,-9,85,-150,126,-24,249,-30,177,-24,330,-116,362,-42,198,-35,172,-57,4,-10,4,-102,10,-43,337,-25,86,350,375,65,373,239,168,317,243,41,279,,78,63,188,219,,29,,87,20,304,134,72,32,378,200,10,89,7,243,[348,358],247,137,330,96,141,79,123,68,216,5,356,243,376,243,97,19,143,239,20,19,358,392,199,305,125,62,355,313,19,171,294,243,185,,285,342,303,13,209,233,73,,142,160,76,-1,355,35,125,349,204,,[35,57],315,176,63,351,139,369,380,[331,333],314,66,,117,252,277,105,348,91,195,376,274,128,350,30,37,306,2,351,264,380,160,240,,157,308,166,145,335,302,164,368,39,104,90,74,340,53,178,82,337,172,139,126,266,310,167,247,134,0,244,374,222,53,185,11,,59,143,154,,331,97,180,43,385,130,,333,135,123,230,305,65,126,134,189,388,246,,139,244,295,74,28,245,378,70,297,,119,1,69,187,174,,80,187,197,375,,91,213,,135,213,200,172,90,8,350,125,17,,15,306,183,180,,354,131,61,391,342,,337,136,168,240,175,71,158,240,50,244,60,123,162,,387,166,350,28,15,-1,334,-7,[31,372],-103,188,290,279,346,,13,327,42,267,326,333,114,126,194,141,188,369,204,102,327,303,194,176,,77,91,110,154,133,358,159,,356,346,31,331,327,122,346,346,30,158,251,,155,253,110,311,,253,[107,140],,85,-1,71,,243,330,-6,7,-2,265,-1,131,79,347,351,375,129,-3,391,395,62,,[0,74],321,,10,,85,124,175,173,34,338,,174,197,[127,334],-1,333,-2,87,279,-4,13,-2,377,,68,357,,39,-1,213,359,333,230,,390,-4,43,,233,171,307,333,-12,356,173,,326,-1,300,,120,,300,351,-2,91,-1,[153,326],1,331,-1,372,-9,300,-7,331,-1,121,-1,164,-1,385,-1,301,135,202,,253,346,-1,339,345,124,97,,377,-1,46,354,134,39,-1,180,-1,63,-9,38,-6,356,356,343,-3,327,-2,164,-1,171,162,,9,-3,330,,341,124,-1,335,,314,375,230,187,-2,123,277,80,-2,209,-1,170,-4,291,,332,-10,351,-1,1,-1,333,,[174,175],-4,5,233,-3,182,-18,248,[127,167],-1,131,-1,131,,62,-1,80,,142,190,191,-1,349,-2,91,-8,130,343,,5,-4,54,-7,347,-4,336,,2,-2,54,245,,183,-2,[245,281],-6,134,-1,263,-6,134,,14,,126,-1,96,,10,-48,326,259,,107,326,359,307,133,-4,351,,359,281,-125,349,65,248,107,334,293,341,341,324,97,72,241,7,296,108,356,175,171,239,131,[129,336],125,,[91,110],352,,143,350,237,,319,107,,352,143,310,,74,383,346,74,213,183,62,272,109,,30,264,237,252,-37,82,-1,15,269,134,,297,-1,235,15,15,-2,80,[285,295],-6,301,334,-35,346,-1,23,-28,313,-36,354,307,-14,124,,338,320,331,259,356,36,78,351,126,285,9,295,,350,76,257,334,-1,128,,18,19,,74,356,206,-1,122,333,100,28,153,,297,30,,197,,172,340,130,183,271,383,203,286,153,104,334,-10,85,-1,337,-194,181,356,321,344,36,251,,19,176,363,285,295,85,134,390,380,321,218,125,350,304,335,181,,128,117,180,106,234,,164,35,346,-1,137,241,[124,241],143,385,-1,23,234,377,,273,330,4,172,243,279,15,180,47,30,379,124,,334,97,-6,316,129,-3,61,148,106,,115,,91,-2,13,143,242,-1,174,-4,151,17,-3,69,,300,-3,90,-7,154,-3,185,-3,312,,388,,255,-8,340,,124,-6,389,-3,252,-5,133,-2,17,-10,119,-2,169,-10,31,-1,91,-7,356,102,-3,153,122,5,239,187,,346,335,167,-2,325,,326,,36,-1,197,-294,356,-1,355,,175,79,-1,5,238,210,175,-1,85,,9,115,,304,102,129,,326,76,-3,124,128,333,,334,344,94,164,166,126,164,285,312,103,277,119,,124,,248,171,,390,80,154,31,97,209,210,63,131,,285,388,81,64,-4,270,74,250,85,120,-1,14,273,4,241,303,100,348,-1,161,15,343,16,183,194,354,-1,102,279,170,394,-1,164,-315,212,133,124,357,195,,224,345,,9,375,97,67,175,345,,347,,36,251,357,,321,295,377,76,98,340,-1,91,177,114,,19,164,135,[97,116],74,356,333,310,330,253,192,2,,10,232,,44,-2,116,,74,[97,116],,46,187,330,,348,-3,126,110,,352,374,172,168,128,133,356,175,-1,352,116,,100,-9,53,175,-1,124,-3,[137,254],380,-1,190,-5,241,175,-8,282,,170,-5,182,-12,251,-2,85,-1,181,,[181,186,197,348],121,-4,120,-3,114,-2,287,164,210,,111,-1,243,-2,197,-1,55,39,355,,350,332,,251,-1,248,,164,,69,23,-3,2,-8,377,,85,85,,194,-11,357,,321,65,,201,-1,97,-2,307,,233,-5,287,,81,-8,355,-1,330,-6,331,,346,,13,,250,107,-4,369,-10,241,124,-44,36,34,-1,134,171,9,312,388,[144,351],356,45,251,329,-11,173,95,140,-2,102,358],"a,ai,an,ang,ao,ba,bai,ban,bang,bao,bei,ben,beng,bi,bian,biao,bie,bin,bing,bo,bu,ca,cai,can,cang,cao,ce,ceng,cha,chai,chan,chang,chao,che,chen,cheng,chi,chong,chou,chu,chuai,chuan,chuang,chui,chun,chuo,ci,cong,cou,cu,cuan,cui,cun,cuo,da,dai,dan,dang,dao,de,deng,di,dian,diao,die,ding,diu,dong,dou,du,duan,dui,dun,duo,e,en,er,fa,fan,fang,fei,fen,feng,fo,fou,fu,ga,gai,gan,gang,gao,ge,gei,gen,geng,gong,gou,gu,gua,guai,guan,guang,gui,gun,guo,ha,hai,han,hang,hao,he,hei,hen,heng,hong,hou,hu,hua,huai,huan,huang,hui,hun,huo,ji,jia,jian,jiang,jiao,jie,jin,jing,jiong,jiu,ju,juan,jue,jun,ka,kai,kan,kang,kao,ke,ken,keng,kong,kou,ku,kua,kuai,kuan,kuang,kui,kun,kuo,la,lai,lan,lang,lao,le,lei,leng,li,lia,lian,liang,liao,lie,lin,ling,liu,long,lou,lu,lv,luan,lue,lun,luo,ma,mai,man,mang,mao,me,mei,men,meng,mi,mian,miao,mie,min,ming,miu,mo,mou,mu,na,nai,nan,nang,nao,ne,nei,nen,neng,ni,nian,niang,niao,nie,nin,ning,niu,nong,nu,nv,nuan,nue,nuo,o,ou,pa,pai,pan,pang,pao,pei,pen,peng,pi,pian,piao,pie,pin,ping,po,pu,qi,qia,qian,qiang,qiao,qie,qin,qing,qiong,qiu,qu,quan,que,qun,ran,rang,rao,re,ren,reng,ri,rong,rou,ru,ruan,rui,run,ruo,sa,sai,san,sang,sao,se,sen,seng,sha,shai,shan,shang,shao,she,shen,sheng,shi,shou,shu,shua,shuai,shuan,shuang,shui,shun,shuo,si,song,sou,su,suan,sui,sun,suo,ta,tai,tan,tang,tao,te,teng,ti,tian,tiao,tie,ting,tong,tou,tu,tuan,tui,tun,tuo,wa,wai,wan,wang,wei,wen,weng,wo,wu,xi,xia,xian,xiang,xiao,xie,xin,xing,xiong,xiu,xu,xuan,xue,xun,ya,yan,yang,yao,ye,yi,yin,ying,yo,yong,you,yu,yuan,yue,yun,za,zai,zan,zang,zao,ze,zei,zen,zeng,zha,zhai,zhan,zhang,zhao,zhe,zhen,zheng,zhi,zhong,zhou,zhu,zhua,zhuai,zhuan,zhuang,zhui,zhun,zhuo,zi,zong,zou,zu,zuan,zui,zun,zuo"];
