/**
 飞地移动两端编辑器
 v0.9
 author kk@encalvelit.com

 **/

window.Editor = {};

//MARK: - 配置项
Editor.nativeContentHeight; //原生高度

// 组件对象
Editor.cover;
Editor.title;
Editor.content;

/*
 初始化
 */
Editor.init = function() {
  //组件
  this.cover = new EditorCover();
  this.title = new EditorTitle();
  this.content = new EditorContent();
  this.callback("onDomLoaded");
};

Editor.callback = function(func) {
  if (window.EnclaveBridge) {
    EnclaveBridge["callback"].apply(EnclaveBridge, arguments);
  }
};

Editor.log = function(msg) {
  console.log(msg);
  EnclaveBridge.log(msg);
};

//MARK: Editor Public
//获取编辑器的内容
Editor.getValuesForCallback = function() {
  let cover = this.cover.getCover();
  let title = this.title.getText();
  let content = this.content.getHTML();
  if (cover == null) {
    cover = "";
  }
  if (title == null) {
    title = "";
  }
  if (content == null) {
    content = "";
  }
  Editor.callback("onValuesCallback", cover, title, content);
};

//获取编辑器的内容
Editor.setValues = function(cover, title, content) {
  Editor.cover.setCover(cover);
  Editor.title.setText(title);
  Editor.content.setHTML(content);
};

Editor.setPlaceholders = function(title, content) {
  Editor.title.setPlaceholder(title);
  Editor.content.setPlaceholder(content);
};

Editor.isFocus = function() {
  return this.content.isFocus() || this.title.isFocus();
};

Editor.blurFocus = function() {
  if (this.content.isFocus()) {
    this.content.blur();
  } else if (this.title.isFocus()) {
    this.title.blur();
  }
};

// MARK: - 夜间模式
//切换到夜间模式
Editor.switchToNightMode = function() {
  EnclaveMobile.switchToNightMode();
};

//切换到白天模式
Editor.switchToLightMode = function() {
  EnclaveMobile.switchToLightMode();
};

// MARK: - EditorTitle
function EditorTitle() {
  this.id = "title";
  this.lastText = "";
  this.wrappedObject = $("#title");
  this.setPlaceholder("标题");
  this.bindEventListeners();
}

EditorTitle.prototype.wrappedDomNode = function() {
  return this.wrappedObject[0];
};

EditorTitle.prototype.bindEventListeners = function() {
  let self = this;
  this.wrappedObject.bind("keydown", function(e) {
    if (e.keyCode == 13) {
      Editor.content.focus();
      e.preventDefault();
    }
  });
  this.wrappedObject.bind("input", function(e) {
    self.onValueChange();
  });
  this.wrappedObject.bind("paste", function(e) {
    self.onPaste(e);
  });
  this.wrappedObject.bind("focus", function(e) {
    Editor.callback("onFocus", self.id);
  });
  this.wrappedObject.bind("blur", function(e) {
    Editor.callback("onBlur", self.id);
  });
};

EditorTitle.prototype.onValueChange = function (e) {
    var currentValue = this.getText()
    if (currentValue != this.lastText) {
      this.lastText = currentValue
      Editor.callback('onValueChange', this.id, currentValue)
    }
    this.refreshPlaceholder()
}

EditorTitle.prototype.onPaste = function(e) {
  e.preventDefault();
  let clipboardData = (e.originalEvent || e).clipboardData;

  // If you copy a link from Safari using the share sheet, it's not
  // available as plain text, only as a URL. So let's first check for
  // URLs, then plain text.
  // Fixes https://github.com/wordpress-mobile/WordPress-Editor-iOS/issues/713
  let url = clipboardData.getData("text/uri-list");
  let plainText = clipboardData.getData("text/plain");

  if (url.length > 0) {
    document.execCommand("insertText", false, url);
  } else if (plainText.length > 0) {
    document.execCommand("insertText", false, plainText);
  }
};

EditorTitle.prototype.isFocus = function() {
  return document.activeElement === this.wrappedDomNode();
};

EditorTitle.prototype.focus = function() {
  this.wrappedObject.focus();
};

EditorTitle.prototype.blur = function() {
  this.wrappedObject.blur();
};

EditorTitle.prototype.getText = function() {
  return this.wrappedObject.text();
};

EditorTitle.prototype.setText = function(text) {
  this.lastText = text
  this.wrappedObject.text(text);
  this.refreshPlaceholder();
};

EditorTitle.prototype.isEmpty = function() {
  let text = this.getText();
  if (typeof text == "undefined") {
    return true;
  }
  return text.length == 0;
};

EditorTitle.prototype.setPlaceholder = function(placeholder) {
  this.wrappedObject.attr("data-placeholder", placeholder);
  this.refreshPlaceholder();
};

EditorTitle.prototype.refreshPlaceholder = function() {
  if (!this.isEmpty()) {
    this.wrappedObject.removeClass("placeholder");
  } else {
    this.wrappedObject.addClass("placeholder");
  }
};

// MARK: - EditorContent
function EditorContent() {
  this.id = "content";

  let emptyScrollingContainer = {
    scrollTop: 0,
    getBoundingClientRect: function() {
      return { top: 0, right: 0, bottom: 0, left: 0 };
    }
  };
  let options = {
    scrollingContainer: emptyScrollingContainer,
    modules: {
      enclave: true
    },
    debug: "info",
    placeholder: "正文"
  };
  this.quill = new Quill("#content", options);
  window.quill = this.quill;
  this.lastHTML = "";
  this.bindEventListeners();
}

EditorContent.prototype.bindEventListeners = function() {
  let self = this;
  this.quill.on("editor-change", function(eventName, ...args) {
    if (eventName === "selection-change") {
      if (args && args.length != 0) {
        let source = args[args.length];
        if (source === "api") return;
      }
      self.calculateEditorHeightWithCaretPosition();
      self.refreshFormatState();
    } else if (eventName === "text-change") {
      //处理iOS滚动问题
      if (EnclaveMobile.isiOS) {
        let lastBlot = self.quill.getLeaf(self.quill.getLength() - 1)[0];
        if (lastBlot && lastBlot.domNode.tagName !== "BR") {
          self.quill.insertText(self.quill.getLength(), "\n");
        }
      }
      self.onValueChange()
    }
  });
};

EditorContent.prototype.onValueChange = function () {
  var currentValue = this.getHTML()
  if (currentValue != this.lastHTML) {
    this.lastHTML = currentValue
    Editor.callback('onValueChange', this.id, currentValue)
  }
}

//原生标注失去焦点
EditorContent.prototype.nativeBlur = false;
//修复当前可见位置
EditorContent.prototype.calculateEditorHeightWithCaretPosition = function(
  fromNative = false
) {
  if (fromNative && this.nativeBlur) {
    this.nativeBlur = false;
  }
  //原生标注失去焦点、当前content无焦点
  if (this.nativeBlur || !this.isFocus()) {
    return;
  }
  let currentRange = this.quill.getSelection();
  if (currentRange == null) {
    return;
  }
  let bounds = this.quill.getBounds(currentRange.index, currentRange.length);
  if (bounds == null) {
    return;
  }
  //可见窗口高度
  let containerHeight = document.documentElement.clientHeight;
  if (Editor.nativeContentHeight) {
    //使用原生可见高度
    containerHeight = Editor.nativeContentHeight * window.devicePixelRatio;
  }
  let currentScrollTop = window.document.body.scrollTop;
  //let currentScrollTop = $(window).scrollTop();

  let newScrollTop = 0;
  let currentSelectionY = bounds.top + this.quill.container.offsetTop;
  let lineHeight = 20 * window.devicePixelRatio; //一行高度
  let direction = "none";
  if (currentSelectionY - lineHeight < currentScrollTop) {
    // 这里滚到光标头部位置
    // 光标所在位置被滚动到顶部
    newScrollTop = currentSelectionY - lineHeight;
    direction = "up";
  } else if (
    currentSelectionY + lineHeight >=
    currentScrollTop + containerHeight
  ) {
    // 光标位置在界面下面看不到
    // 这里滚到光标底部位置
    newScrollTop = currentSelectionY + lineHeight - containerHeight;
    direction = "down";
  } else {
    return;
  }
  Editor.log(
    "direction: " +
      direction +
      "\n可见高度: " +
      containerHeight +
      "\n当前光标位置:" +
      currentSelectionY +
      "\ncurrentScrollTop: " +
      currentScrollTop +
      "\nnewScrollTop: " +
      newScrollTop
  );
  if (newScrollTop == currentScrollTop) {
    return;
  }
  //$('html,body').animate({'scrollTop': newScrollTop}, 100)//0.1 ms
  window.scrollTo(0, newScrollTop);
};

EditorContent.prototype.backupRange = function() {
  let selection = this.quill.getSelection(false);
  if (selection) {
    this.currentSelection = selection;
  }
};

EditorContent.prototype.restoreRange = function() {
  if (this.currentSelection) {
    this.quill.setSelection(
      this.currentSelection.index,
      this.currentSelection.length
    );
  }
};

//获取Format状态
EditorContent.prototype.refreshFormatState = function() {
  let format = {};
  if (this.isFocus()) {
    //有焦点时才获取
    let selection = this.quill.getSelection();
    if (selection) {
      format = this.quill.getFormat(selection.index, selection.length);
    }
  }
  //Android需要转换成string
  if (EnclaveMobile.isAndroid) {
    format = JSON.stringify(format);
  }
  Editor.callback("onFormatChange", format);
};

EditorContent.prototype.setPlaceholder = function(placeholder) {
  this.quill.root.dataset.placeholder = placeholder;
};

EditorContent.prototype.setHTML = function(html) {
  this.lastHTML = html
  //停止正在播放的音频
  EnclaveAudio.stop();
  this.quill.setContents([]);
  this.quill.clipboard.dangerouslyPasteHTML(0, html);
  //如果是夜间模式则重新渲染Audio组件
  // if (EnclaveMobile.isNightMode()) {
  //   EnclaveAudio.switchToNightMode();
  // }
};

EditorContent.prototype.getHTML = function() {
    let html = this.quill.root.innerHTML;
    //清理工作
    html = html.replace(/&nbsp;/, " ");
    html = html.trim();
    html = html.replace(new RegExp('^<br>'), '');
    html = html.replace(new RegExp('<p><br></p>$'), '');

    //音频icon，正在播放中、夜间模式下
    html = html.replace(/src=['"]audio_pause.png['"]/g, "src=\"audio_play.png\"");
    html = html.replace(/src=['"]audio_play_night.png['"]/g, "src=\"audio_play.png\"");
    html = html.replace(/src=['"]audio_pause_night.png['"]/g, "src=\"audio_play.png\"");
    html = html.replace(/src=['"]audio_delete_night.png['"]/g, "src=\"audio_delete.png\"");
    //编辑器quill音频组件的onclick无效问题
    html = html.replace(new RegExp('_onclick', 'g'), 'onclick');

    if (html === '<p><br></p>' || html === '<p><br></p><p><br></p>') {
      html = '';
    }
    return html
};

EditorContent.prototype.focus = function() {
  //this.quill.focus()
  //使用quill自带的focus会先focus后reset range，会抖动
  //因此这里先reset native range，后focus
  let selection = this.quill.selection;
  if (!this.isFocus()) {
    function setNativeRange(
      startNode,
      startOffset,
      endNode = startNode,
      endOffset = startOffset,
      force = false
    ) {
      if (
        startNode != null &&
        (selection.root.parentNode == null ||
          startNode.parentNode == null ||
          endNode.parentNode == null)
      ) {
        return;
      }
      let _selection = document.getSelection();
      if (_selection == null) return;
      if (startNode != null) {
        let nativeRang = selection.getNativeRange() || {};
        let native = nativeRang.native;
        if (
          native == null ||
          force ||
          startNode !== native.startContainer ||
          startOffset !== native.startOffset ||
          endNode !== native.endContainer ||
          endOffset !== native.endOffset
        ) {
          if (startNode.tagName === "BR") {
            startOffset = Array.from(startNode.parentNode.childNodes).indexOf(
              startNode
            );
            startNode = startNode.parentNode;
          }
          if (endNode.tagName === "BR") {
            endOffset = Array.from(endNode.parentNode.childNodes).indexOf(
              endNode
            );
            endNode = endNode.parentNode;
          }
          let range = document.createRange();
          range.setStart(startNode, startOffset);
          range.setEnd(endNode, endOffset);
          _selection.removeAllRanges();
          _selection.addRange(range);
        }
      }
    }
    let savedRange = selection.savedRange;
    if (!savedRange || savedRange.index === 0) {
      selection.root.focus();
      return;
    }
    let args = selection.rangeToNative(savedRange);
    setNativeRange(...args, false);
    selection.root.focus();
  }
};

EditorContent.prototype.isFocus = function() {
  return this.quill.hasFocus();
};

EditorContent.prototype.blur = function() {
  this.nativeBlur = true;
  this.quill.blur();
};

//MARK: - Styles
EditorContent.prototype.setTextFormat = function(name, value) {
  this.quill.format(name, value);
  this.refreshFormatState();
};

EditorContent.prototype.setLineFormat = function(name, value) {
  let selection = this.quill.getSelection();
  if (selection) {
    this.quill.formatLine(selection.index, selection.length, name, value);
    this.refreshFormatState();
  }
};

EditorContent.prototype.setBold = function(value) {
  this.setTextFormat("bold", value);
};

EditorContent.prototype.setItalic = function(value) {
  this.setTextFormat("italic", value);
};

EditorContent.prototype.setUnderline = function(value) {
  this.setTextFormat("underline", value);
};

EditorContent.prototype.setStrikeThrough = function(value) {
  this.setTextFormat("strike", value);
};

EditorContent.prototype.setHeader = function(value) {
  this.setLineFormat("header", value);
};

EditorContent.prototype.setSection = function(value) {
  this.setTextFormat("section", value);
};

EditorContent.prototype.setBlockquote = function(value) {
  this.setLineFormat("blockquote", value);
};

EditorContent.prototype.setAlign = function(value) {
  this.setLineFormat("align", value);
};

EditorContent.prototype.setList = function(value) {
  this.setLineFormat("list", value);
};

//插入图片
EditorContent.prototype.insertImage = function(value) {
  this.focus();
  let range = this.quill.getSelection();
  if (range == null) {
    return;
  }
  this.quill.insertEmbed(range.index, "image", value, Quill.sources.USER);
  this.quill.insertText(range.index + 1, "\n", Quill.sources.USER);
  this.quill.setSelection(range.index + 2, Quill.sources.SILENT);
};

//插入音频
EditorContent.prototype.insertAudio = function(url, duration) {
  this.focus();
  QuillEnclave.insertAudio(url, duration);
};

// MARK: - 封面图对象
function EditorCover() {
  this.wrappedObject = $("#cover");
  this.wrappedIconObject = this.wrappedObject.children("div");
  this.wrappedImageObject = this.wrappedObject.children("img");
  this.id = this.wrappedObject.attr("id");

  //事件
  var self = this;
  this.wrappedObject.on("click", function(e) {
    self.onClick(e);
  });
  this.wrappedImageObject.on("load", function(e) {
    self.onLoad();
  });

  this.clearCover();
  //设置icon的宽度
  let width = this.wrappedObject.css("width");
  this.wrappedIconObject.css("width", width);
}

// 清除封面图
EditorCover.prototype.clearCover = function() {
  this.wrappedObject.css("height", "1.6rem");
  this.wrappedImageObject.attr("src", "");
  this.wrappedImageObject.hide();
  this.wrappedIconObject.show();
};

// 设置封面图
EditorCover.prototype.setCover = function(url) {
  if (url == undefined || url == null || url === "") {
    this.clearCover();
    return;
  }
  this.wrappedObject.css("height", "3.86rem");
  this.wrappedImageObject.attr("src", url);
  this.wrappedImageObject.show();
  this.wrappedIconObject.hide();
};

// 获取封面图url
EditorCover.prototype.getCover = function() {
  let url = "";
  if (
    this.wrappedImageObject != undefined &&
    this.wrappedImageObject.length > 0
  ) {
    url = this.wrappedImageObject.attr("src");
  }
  return url;
};

EditorCover.prototype.getCoverForCallback = function() {
  let value = this.getCover();
  // URI Encode HTML on API < 17 because of the use of WebViewClient.shouldOverrideUrlLoading. Data must
  // be decoded in shouldOverrideUrlLoading.
  if (native.androidApiLevel < 17) {
    value = encodeURIComponent(value);
  }
  let valueArgument = "value=" + value;
  let joinedArguments =
    this.callbackId() + EnclaveEditor.defaultCallbackSeparator + valueArgument;
  Editor.callback("callback-field-getvalue", joinedArguments);
};

//封面图加载成功，垂直居中处理
EditorCover.prototype.onLoad = function() {
  let imgHeight = this.wrappedImageObject.height();
  let containerHeight = this.wrappedObject.height();
  if (imgHeight > containerHeight) {
    //大于控件本身才设置
    this.wrappedImageObject.css(
      "margin-top",
      -(imgHeight - containerHeight) / 2 + "px"
    );
  } else {
    this.wrappedImageObject.css("margin-top", "0px");
  }
};

// 封面图点击事件
EditorCover.prototype.onClick = function(event) {
  event.preventDefault();
  let url = this.getCover();
  Editor.callback("onCoverTap", url);
};

//MARK: Start
Editor.init();
