// 使用的主要插件：
// 1. compressorjs - 图片压缩
// 2. merge-images - 图片合并
// 3. exif-js - 读取旋转信息

/**
 * 处理并合并图片
 * @param {Array} imageList - 图片列表，包含file和旋转信息
 * @param {Object} options - 配置选项
 * @param {Function} callback - 回调函数，返回合并后的base64图片
 */
function mergeAndCompressImagesWithRotation(imageList, options, callback) {
    // 默认配置
    var defaultOptions = {
        compressQuality: 0.7,
        mergeQuality: 0.9,
        scale: 0.5,
        outputFormat: 'image/jpeg',
        backgroundColor: '#ffffff',
        dpi: 96, // 默认DPI
        grayscale: true // 新增参数：转换为灰度图
    };

    // 合并配置
    var config = Object.assign({},  defaultOptions, options);

    // 参数验证
    if (!Array.isArray(imageList))  {
        callback(new Error('参数必须是图片数组'), null);
        return;
    }

    // 加载图片并处理
    var processedImages = [];
    var processedCount = 0;
    var hasError = false;

    imageList.forEach(function(imageInfo,  index) {
        if (hasError) return;

        loadAndProcessImage(imageInfo.file,  imageInfo.rotation,  config, function(err, processedImg) {
            if (err) {
                hasError = true;
                callback(err, null);
                return;
            }

            processedImages[index] = processedImg;
            processedCount++;

            // 所有图片处理完成后合并
            if (processedCount === imageList.length)  {
                mergeAllImages(processedImages, config, callback);
            }
        });
    });
}

/**
 * 将图片转换为灰度图
 * @param {HTMLCanvasElement} canvas - 输入canvas
 * @param {number} [quality=0.8] - 输出质量
 * @param {Function} callback - 回调函数
 */
function convertToGrayscale(canvas, quality, callback) {
    var ctx = canvas.getContext('2d');
    var imageData = ctx.getImageData(0,  0, canvas.width,  canvas.height);
    var data = imageData.data;

    // 转换为灰度
    for (var i = 0; i < data.length;  i += 4) {
        // 使用亮度公式（符合人眼感知）
        var avg = (data[i] * 0.3 + data[i + 1] * 0.59 + data[i + 2] * 0.11);
        data[i] = data[i + 1] = data[i + 2] = avg;
    }

    ctx.putImageData(imageData,  0, 0);

    // 转换为Blob
    canvas.toBlob(function(blob)  {
        callback(null, blob);
    }, 'image/jpeg', quality || 0.8);
}

/**
 * 修改后的loadAndProcessImage函数（增加灰度处理）
 */
function loadAndProcessImage(url, rotation, config, callback) {
    var img = new Image();
    img.crossOrigin  = 'Anonymous';

    img.onload  = function() {
        var canvas = document.createElement('canvas');
        var ctx = canvas.getContext('2d');

        // 应用旋转和缩放
        var rotated = applyRotation(canvas, ctx, img, rotation, config.scale);

        if (isPNG(url) || isBlackWhite(url)) {
            // 不压缩的情况
            rotated.canvas.toBlob(function(blob)  {
                callback(null, {
                    blob: blob,
                    width: rotated.width,
                    height: rotated.height
                });
            }, getMimeType(url) || 'image/jpeg');
        }else if (config.grayscale)  {
            // 转换为灰度图
            convertToGrayscale(rotated.canvas,  config.compressQuality,  function(err, blob) {
                if (err) return callback(err);
                callback(null, {
                    blob: blob,
                    width: rotated.width,
                    height: rotated.height
                });
            });
        }else {
            // 普通压缩
            compressImage(rotated.canvas,  config.compressQuality,  function(err, blob) {
                callback(err, {
                    blob: blob,
                    width: rotated.width,
                    height: rotated.height
                });
            });
        }
    };

    img.onerror  = function() {
        callback(new Error('图片加载失败: ' + url), null);
    };

    img.src  = url;
}

/**
 * 应用旋转、缩放和DPI设置（修正版）
 */
function applyRotation(canvas, ctx, img, orientation, scale) {
    // 计算缩小后的逻辑尺寸
    var width = img.width  * scale;
    var height = img.height  * scale;

    // 根据旋转调整canvas尺寸（不再乘以DPI比例）
    if (orientation ==90 || orientation ==270) {
        canvas.width  = height;  // 仅使用逻辑像素尺寸
        canvas.height  = width;
    } else {
        canvas.width  = width;
        canvas.height  = height;
    }

    // 清除画布
    ctx.clearRect(0,  0, canvas.width,  canvas.height);
    ctx.save();

    // 根据旋转信息变换画布
    switch(orientation) {
        case 2: ctx.transform(-1,  0, 0, 1, width, 0); break;
        case 180: ctx.transform(-1,  0, 0, -1, width, height); break;
        case 4: ctx.transform(1,  0, 0, -1, 0, height); break;
        case 5: ctx.transform(0,  1, 1, 0, 0, 0); break;
        case 90: ctx.transform(0,  1, -1, 0, height, 0); break;
        case 7: ctx.transform(0,  -1, -1, 0, height, width); break;
        case 270: ctx.transform(0,  -1, 1, 0, 0, width); break;
        default: break;
    }

    // 绘制缩小后的图片（使用原始绘制尺寸）
    ctx.drawImage(img,  0, 0, img.width,  img.height,  0, 0, width, height);
    ctx.restore();

    return {
        canvas: canvas,
        width: canvas.width,   // 返回实际canvas尺寸
        height: canvas.height
    };
}

/**
 * 获取图片MIME类型
 */
function getMimeType(url) {
    var ext = url.split('.').pop().toLowerCase();
    switch(ext) {
        case 'jpg':
        case 'jpeg': return 'image/jpeg';
        case 'png': return 'image/png';
        case 'webp': return 'image/webp';
        default: return null;
    }
}

/**
 * 检查是否是PNG图片
 */
function isPNG(url) {
    return url.toLowerCase().endsWith('.png')  ||
        url.toLowerCase().indexOf('image/png')  !== -1;
}

/**
 * 检查是否是黑白图片
 */
function isBlackWhite(url) {
    // 简单判断黑白图 - 实际项目中可能需要更复杂的检测
    var filename = url.split('/').pop().toLowerCase();
    return filename.indexOf('bw')  !== -1 ||
        filename.indexOf('blackwhite')  !== -1 ||
        filename.indexOf('monochrome')  !== -1;
}

/**
 * 压缩图片
 */
function compressImage(canvas, quality, callback) {
    canvas.toBlob(function(blob)  {
        try {
            new Compressor(blob, {
                quality: quality,
                maxWidth: canvas.width,
                maxHeight: canvas.height,
                success: function(compressedBlob) {
                    callback(null, compressedBlob);
                },
                error: function(err) {
                    callback(err, null);
                }
            });
        } catch (e) {
            callback(e, null);
        }
    }, 'image/jpeg', 0.8);
}

/**
 * 合并所有图片（修正版）
 */
function mergeAllImages(images, config, callback) {
    // 过滤有效图片
    var validImages = images.filter(function(img)  { return img !== null; });

    if (validImages.length  === 0) {
        callback(new Error('没有有效的图片可以合并'), null);
        return;
    }

    // 计算合并后的画布尺寸（单位：CSS像素）
    var canvasWidth = Math.max.apply(null,  validImages.map(function(img)  {
        return img.width;
    }));

    var canvasHeight = validImages.reduce(function(sum,  img) {
        return sum + img.height;
    }, 0);

    // 创建背景Canvas
    var bgCanvas = document.createElement('canvas');
    bgCanvas.width  = canvasWidth;
    bgCanvas.height  = canvasHeight;
    var bgCtx = bgCanvas.getContext('2d');
    bgCtx.fillStyle  = config.backgroundColor;
    bgCtx.fillRect(0,  0, canvasWidth, canvasHeight);

    // 准备合并配置（背景图作为第一层）
    var mergeConfig = [
        { src: bgCanvas.toDataURL(),  x: 0, y: 0 }
    ].concat(validImages.map(function(img,  index) {
        return {
            src: URL.createObjectURL(img.blob),
            x: 0,
            y: validImages.slice(0,  index).reduce(function(sum, i) {
                return sum + i.height;
            }, 0),
            width: img.width,
            height: img.height
        };
    }));

    // 修正方案：直接使用浏览器原生Canvas
    var mergeOptions = {
        width: canvasWidth,
        height: canvasHeight,
        format: config.outputFormat,
        quality: config.mergeQuality,
    };

    // 添加调试日志
    // console.log('Merge  config:', mergeConfig);
    // console.log('Merge  options:', mergeOptions);

    mergeImages(mergeConfig, mergeOptions)
        .then(function(dataUrl) {
            // 清理对象URL
            mergeConfig.forEach(function(item)  {
                URL.revokeObjectURL(item.src);
            });
            callback(null, dataUrl);
        })
        .catch(function(err) {
            console.error(' 合并错误详情:', err);
            // 清理对象URL
            mergeConfig.forEach(function(item)  {
                URL.revokeObjectURL(item.src);
            });
            callback(new Error('图片合并失败: ' + err.message),  null);
        });
}

// 计算图片大小
function calculateImageSize(dataUrl) {
    const base64Length = dataUrl.length  - 'data:image/jpeg;base64,'.length;
    return Math.round(base64Length  * 3 / 4);
}

// 格式化文件大小
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes)  / Math.log(k));
    return parseFloat((bytes / Math.pow(k,  i)).toFixed(2)) + ' ' + sizes[i];
}
