// 规格信息
var specifications;

// 用法用量信息
var dosage;

function setupParam(specificationsObj, dosageObj) {
    specifications = specificationsObj;
    dosage = dosageObj;
    
    // 保留6位小数
    if (dosage && dosage.dosePerTime) {
        dosage.dosePerTime = Math.floor(dosage.dosePerTime * 1000000) / 1000000;
    }
    
    if (specifications) {
        if (specifications.minPackingNum) {
            specifications.minPackingNum = Math.floor(specifications.minPackingNum * 1000000) / 1000000;
        }
        
        if (specifications.dose) {
            specifications.dose = Math.floor(specifications.dose * 1000000) / 1000000;
        }
    }
}

function setupParamWithJsonString(specificationsJson, dosageJson) {
    specifications = JSON.parse(specificationsJson);
    dosage = JSON.parse(dosageJson);
    
    // 保留6位小数
    if (dosage && dosage.dosePerTime) {
        dosage.dosePerTime = Math.floor(dosage.dosePerTime * 1000000) / 1000000;
    }
    
    if (specifications) {
        if (specifications.minPackingNum) {
            specifications.minPackingNum = Math.floor(specifications.minPackingNum * 1000000) / 1000000;
        }
        
        if (specifications.dose) {
            specifications.dose = Math.floor(specifications.dose * 1000000) / 1000000;
        }
    }
}

function checkConsistencyUnit() {
    var capacityUnitArr = ['litre', 'milliter'];
    var weightUnitArr = ['gram', 'milgram', 'nanogram', 'micgram'];

    if (specifications.isCompound) {
        if (specifications.preparationUnit == dosage.doseUnit) {
            return true
        }

        if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
            || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
            return true;
        }

        return false;
    }

    if (specifications.doseUnit == dosage.doseUnit || specifications.preparationUnit == dosage.doseUnit) {
        return true;
    }

    if ((capacityUnitArr.includes(specifications.doseUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.doseUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        return true;
    }

    if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        return true;
    }

    return false;
}

function uniformDoseUnit(value, unit) {
    if (unit == 'milliter') {
        return value;
    }

    if (unit == 'litre') {
        return value * 1000;
    }

    if (unit == 'nanogram') {
        return value;
    }

    if (unit == 'micgram') {
        return value * 1000;
    }

    if (unit == 'milgram') {
        return value * 1000 * 1000;
    }

    if (unit == 'gram') {
        return value * 1000 * 1000 * 1000;
    }

    return value;
}

function checkParameters() {
    if (specifications && specifications.dose &&
        specifications.doseUnit &&
        specifications.minPackingNum &&
        specifications.preparationUnit) {
        if (dosage &&
            dosage.dosePerTime &&
            dosage.doseUnit &&
            dosage.time &&
            dosage.times) {
            if (dosage.times > 0 && dosage.times <= 10 && dosage.time > 0 && dosage.time <= 24) {
                return true;
            }
        }
    }

    return false;
}

function frequencyOfHourType(hour) {
    if (!hour) {
        return -1;
    }

    if (hour && hour > 24) {
        return -1;
    }

    switch (hour) {
        case 1:
            return 16;
        case 2:
            return 8;
        case 3:
            return 5;
        case 4:
        case 5:
        case 6:
            return 4;
        case 7:
        case 8:
        case 9:
            return 3;
        case 10:
        case 11:
        case 12:
            return 2;
        default:
            return 1;

    }
}

function numberOfDosesPerUnitTime() {
    // 时
    if (dosage.timeType == 0) {
        return frequencyOfHourType(dosage.time);
    }

    // 每
    if (dosage.intervalType == 0) {
        if (dosage.timeType == 1) {
            return floatObj.divide(dosage.times, dosage.time, 2);
        } else if (dosage.timeType == 2) {
            return floatObj.divide(dosage.times, dosage.time * 7, 2);
        } else if (dosage.timeType == 3) {
            return floatObj.divide(dosage.times, dosage.time * 30, 2);
        } else if (dosage.timeType == 4) {
            return floatObj.divide(dosage.times, dosage.time * 365, 2);
        }

        return -1;
    } else if (dosage.intervalType == 1) {
        if (dosage.timeType == 1) {
            return floatObj.divide(dosage.times, (dosage.time + 1), 2);
        } else if (dosage.timeType == 2) {
            return floatObj.divide(dosage.times,((dosage.time + 1) * 7), 2);
        } else if (dosage.timeType == 3) {
            return floatObj.divide(dosage.times,((dosage.time + 1) * 30), 2);
        } else if (dosage.timeType == 4) {
            return floatObj.divide(dosage.times,((dosage.time + 1) * 365), 2);
        }
    }

    return -1
}

function calculateMedicationDays(durgCount) {
    if (durgCount == 0) {
        return 0;
    }

    if (!checkParameters()) {
        return -1;
    }

    // 频率是否可映射为频次
    if (numberOfDosesPerUnitTime() < 0) {
        return -2;
    }

    var capacityUnitArr = ['litre', 'milliter'];
    var weightUnitArr = ['gram', 'milgram', 'nanogram', 'micgram'];
    
    if (specifications.isCompound) {
        if (specifications.preparationUnit == dosage.doseUnit) {
            var numerator = floatObj.multiply(specifications.minPackingNum, durgCount, 2);
            var denominator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
            var days = floatObj.divide(numerator, denominator, 2);
            days = Math.ceil(days);
            if (days > 1) {
                return days;
            }

            return 1;
        }

        if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
            || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
            var specificationsPackingNum = uniformDoseUnit(specifications.minPackingNum, specifications.preparationUnit);
            var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);

            var numerator = floatObj.multiply(specificationsPackingNum, durgCount, 2);
            var denominator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
            var days = floatObj.divide(numerator, denominator, 2);
            days = Math.ceil(days);
            if (days > 1) {
                return days;
            }

            return 1;
        }

        return -3;
    }

    if (specifications.doseUnit == dosage.doseUnit) {
        var numerator = floatObj.multiply(specifications.dose, specifications.minPackingNum, 2);
        numerator = floatObj.multiply(numerator, durgCount, 2);
        var denominator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
        var days = floatObj.divide(numerator, denominator, 2);
        days = Math.ceil(days);
        if (days > 1) {
            return days;
        }

        return 1;
    }

    if (specifications.preparationUnit == dosage.doseUnit) {
        var numerator = floatObj.multiply(specifications.minPackingNum, durgCount, 2);
        var denominator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
        var days = floatObj.divide(numerator, denominator, 2);
        days = Math.ceil(days);
        if (days > 1) {
            return days;
        }

        return 1;
    }

    if ((capacityUnitArr.includes(specifications.doseUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.doseUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        var specificationsDose = uniformDoseUnit(specifications.dose, specifications.doseUnit);
        var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);
        
        var numerator = floatObj.multiply(specificationsDose, specifications.minPackingNum, 2);
        numerator = floatObj.multiply(numerator, durgCount, 2);
        var denominator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
        var days = floatObj.divide(numerator, denominator, 2);
        days = Math.ceil(days);
        if (days > 1) {
            return days;
        }

        return 1;
    }

    if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        var specificationsPackingNum = uniformDoseUnit(specifications.minPackingNum, specifications.preparationUnit);
        var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);

        var numerator = floatObj.multiply(specificationsPackingNum, durgCount, 2);
        var denominator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
        var days = floatObj.divide(numerator, denominator, 2);
        days = Math.ceil(days);
        if (days > 1) {
            return days;
        }

        return 1;
    }

    return -4;
}

function calculateDrugCount(medicationDays) {
    if (medicationDays == 0) {
        return 0;
    }

    if (!checkParameters()) {
        return -1;
    }

    if (numberOfDosesPerUnitTime() < 0) {
        return -2;
    }

    var capacityUnitArr = ['litre', 'milliter'];
    var weightUnitArr = ['gram', 'milgram', 'nanogram', 'micgram'];

    if (specifications.isCompound) {
        // 复方算法
        if (specifications.preparationUnit == dosage.doseUnit) {
            var numerator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
            numerator = floatObj.multiply(numerator, medicationDays, 2);
            var denominator = specifications.minPackingNum;
            var drugCount = floatObj.divide(numerator, denominator, 2);
            drugCount = Math.ceil(drugCount);
            if (drugCount > 1) {
                return drugCount;
            }
            
            return 1;
        }

        if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
            || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
            var specificationsPackingNum = uniformDoseUnit(specifications.minPackingNum, specifications.preparationUnit);
            var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);

            var numerator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
            numerator = floatObj.multiply(numerator, medicationDays, 2);
            var denominator = specificationsPackingNum;
            var drugCount = floatObj.divide(numerator, denominator, 2);
            drugCount = Math.ceil(drugCount);
            if (drugCount > 1) {
                return drugCount;
            }

            return 1;
        }

        return -3;
    }


    if (specifications.doseUnit == dosage.doseUnit) {
        var numerator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
        numerator = floatObj.multiply(numerator, medicationDays, 2);
        var denominator = floatObj.multiply(specifications.dose, specifications.minPackingNum, 2);
        var drugCount = floatObj.divide(numerator, denominator, 2);
        drugCount = Math.ceil(drugCount);
        if (drugCount > 1) {
            return drugCount;
        }

        return 1;
    }

    if (specifications.preparationUnit == dosage.doseUnit) {
        var numerator = floatObj.multiply(dosage.dosePerTime, numberOfDosesPerUnitTime(), 2);
        numerator = floatObj.multiply(numerator, medicationDays, 2);
        var denominator = specifications.minPackingNum;
        var drugCount = floatObj.divide(numerator, denominator, 2);
        drugCount = Math.ceil(drugCount);
        if (drugCount > 1) {
            return drugCount;
        }
        
        return 1;
    }

    if ((capacityUnitArr.includes(specifications.doseUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.doseUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        var specificationsDose = uniformDoseUnit(specifications.dose, specifications.doseUnit);
        var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);
        
        var numerator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
        numerator = floatObj.multiply(numerator, medicationDays, 2);
        var denominator = floatObj.multiply(specificationsDose, specifications.minPackingNum, 2);
        var drugCount = floatObj.divide(numerator, denominator, 2);
        drugCount = Math.ceil(drugCount);
        if (drugCount > 1) {
            return drugCount;
        }
        
        return 1;
    }

    if ((capacityUnitArr.includes(specifications.preparationUnit) && capacityUnitArr.includes(dosage.doseUnit))
        || weightUnitArr.includes(specifications.preparationUnit) && weightUnitArr.includes(dosage.doseUnit)) {
        var specificationsPackingNum = uniformDoseUnit(specifications.minPackingNum, specifications.preparationUnit);
        var dosageDosePerTime = uniformDoseUnit(dosage.dosePerTime, dosage.doseUnit);

        var numerator = floatObj.multiply(dosageDosePerTime, numberOfDosesPerUnitTime(), 2);
        numerator = floatObj.multiply(numerator, medicationDays, 2);
        var denominator = specificationsPackingNum;
        var drugCount = floatObj.divide(numerator, denominator, 2);
        drugCount = Math.ceil(drugCount);
        if (drugCount > 1) {
            return drugCount;
        }
        
        return 1;
    }

    return -4;
}

var floatObj = function() {
    function isInteger(obj) {
        return Math.floor(obj) === obj
    }

    /*
     * 将一个浮点数转成整数，返回整数和倍数。如 3.14 >> 314，倍数是 100
     * @param floatNum {number} 小数
     * @return {object}
     *   {times:100, num: 314}
     */
    function toInteger(floatNum) {
        var ret = {times: 1, num: 0}
        if (isInteger(floatNum)) {
            ret.num = floatNum
            return ret
        }
        var strfi  = floatNum + ''
        var dotPos = strfi.indexOf('.')
        var len    = strfi.substr(dotPos+1).length
        var times  = Math.pow(10, len)
        var intNum = Number(floatNum.toString().replace('.',''))
        ret.times  = times
        ret.num    = intNum
        return ret
    }

    /*
     * 核心方法，实现加减乘除运算，确保不丢失精度
     * 思路：把小数放大为整数（乘），进行算术运算，再缩小为小数（除）
     *
     * @param a {number} 运算数1
     * @param b {number} 运算数2
     * @param digits {number} 精度，保留的小数点数，比如 2, 即保留为两位小数
     * @param op {string} 运算类型，有加减乘除（add/subtract/multiply/divide）
     *
     */
    function operation(a, b, digits, op) {
        var o1 = toInteger(a)
        var o2 = toInteger(b)
        var n1 = o1.num
        var n2 = o2.num
        var t1 = o1.times
        var t2 = o2.times
        var max = t1 > t2 ? t1 : t2
        var result = null
        switch (op) {
            case 'add':
                if (t1 === t2) { // 两个小数位数相同
                    result = n1 + n2
                } else if (t1 > t2) { // o1 小数位 大于 o2
                    result = n1 + n2 * (t1 / t2)
                } else { // o1 小数位 小于 o2
                    result = n1 * (t2 / t1) + n2
                }
                return result / max
            case 'subtract':
                if (t1 === t2) {
                    result = n1 - n2
                } else if (t1 > t2) {
                    result = n1 - n2 * (t1 / t2)
                } else {
                    result = n1 * (t2 / t1) - n2
                }
                return result / max
            case 'multiply':
                result = (n1 * n2) / (t1 * t2)
                return result
            case 'divide':
                result = (n1 / n2) * (t2 / t1)
                return result
        }
    }

    function add(a, b, digits) {
        return operation(a, b, digits, 'add')
    }
    
    function subtract(a, b, digits) {
        return operation(a, b, digits, 'subtract')
    }
    
    function multiply(a, b, digits) {
        return operation(a, b, digits, 'multiply')
    }
    
    function divide(a, b, digits) {
        return operation(a, b, digits, 'divide')
    }

    return {
        add: add,
        subtract: subtract,
        multiply: multiply,
        divide: divide
    }
}();
