package com.dingyue.statistics

import android.app.Application
import android.support.annotation.MainThread
import android.util.Log
import com.dingyue.statistics.client.SimpleAd
import com.dingyue.statistics.client.SimpleBook
import com.dingyue.statistics.core.StatRealService
import com.dingyue.statistics.db.dao.bean.BaseChapterFeedback

/**
 * Desc 统计功能对外提供的API
 * Author jiaxing_sun
 * Mail jiaxing_sun@dingyuegroup.cn
 * Date 2018/8/23 10:36
 */
object DyStatService {

    class Config {
        var debugMode: Boolean = false
        //用来做登录用户跟踪
        var loginUserId: String? = null
            set(value) {
                field = value
                StatRealService.configBizUserId(value)
            }
        //本地日志保存是否启用
        var enableSaveLog: Boolean = false
        //crash日志功能是否启用
        var enableCrashFunc: Boolean = true
        //toast提示
        var enableToast: Boolean = false
        //统计功能开关
        var enableStatServiceFunc: Boolean = true

        interface ConfigDelegate {
            fun config(config: Config)
        }
    }

    @JvmStatic
    var config: Config = Config()
        private set(value) {
            field = value
        }

    /**
     * SDK初始化
     * @param application 启动的application
     * @param udid 外部维护的udid
     */
    @JvmStatic
    @JvmOverloads
    @MainThread
    fun init(application: Application, udid: String = "", appChannel: String? = null, config: Config.ConfigDelegate? = null) {
        if (udid.isBlank()) {
            Log.e("DyStatService", "必须传递合法udid")
            return
        }
        try {
            StatRealService.preInit(application)
            config?.config(this.config)
            //初始化异常让直接crash
            StatRealService.init(udid, appChannel)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    @JvmStatic
    fun release() {
        StatRealService.release()
    }

    /**
     * 设置位置信息
     * @param longitude 经度
     * @param latitude 纬度
     * @param cityInfo 城市名称
     * @param cityCode 城市编码
     * @param locationDetail 详细地址
     */
    @JvmStatic
    fun setLocationInfo(longitude: Double, latitude: Double, cityInfo: String, cityCode: String, locationDetail: String) {
        StatRealService.setLocationInfo(longitude, latitude, cityInfo, cityCode, locationDetail)
    }

    /**
     * 打点统计
     * @param pageAndIdentify 格式 pageCode_identify
     * @param extraParam 额外参数
     */
    @JvmStatic
    @JvmOverloads
    fun onEvent(pageAndIdentify: String, extraParam: Map<String, String>? = null) {
        StatRealService.onEvent(pageAndIdentify, extraParam)
    }

    /**
     * 打点统计
     * @param pageAndIdentify 格式 pageCode[split]identify
     * @param split [pageAndIdentify] 中页面编码与点位编码的分割符，默认为_
     * @param extraParam 额外参数
     */
    @JvmStatic
    @JvmOverloads
    fun onEventBy(pageAndIdentify: String, split: String = "_", extraParam: Map<String, String>? = null) {
        StatRealService.onEventBy(pageAndIdentify, split, extraParam)
    }

    /**
     * 打点统计
     * @param pageCode 页面标识
     * @param identify 点位标识
     * @param extraParam 附件参数  可为null
     */
    @JvmStatic
    @JvmOverloads
    fun onEvent(pageCode: String, identify: String, extraParam: Map<String, String>? = null) {
        StatRealService.onEvent(pageCode, identify, extraParam)
    }

    /**
     * 广告曝光事件
     * @param book
     * @param ad
     * @param extraParam
     */
    @JvmStatic
    @JvmOverloads
    fun onAdShowEvent(book: SimpleBook, ad: SimpleAd, extraParam: Map<String, String>? = null) {
        StatRealService.onAdShowEvent(book, ad, extraParam)
    }

    /**
     * 广告点击事件
     * @param book
     * @param ad
     * @param extraParam
     */
    @JvmStatic
    @JvmOverloads
    fun onAdClickEvent(book: SimpleBook, ad: SimpleAd, extraParam: Map<String, String>? = null) {
        StatRealService.onAdClickEvent(book, ad, extraParam)
    }

    /**
     * 阅读PV统计 （下一章时调用统计）
     * @param book
     */
    @JvmStatic
    @JvmOverloads
    fun sendPVData(book: SimpleBook) {
        StatRealService.sendPVData(book)
    }

    /**
     * 章节反馈
     * successMessage为反馈成功提示信息,默认为 反馈成功
     * sameErrorMessage为同一章节同一问题反馈次数超过sameFeedbackTime预设定的值时的提示信息,默认为 请勿重复反馈
     */
    @JvmStatic
    @JvmOverloads
    fun onChapterError(chapterFeedback: BaseChapterFeedback?, successMessage: String? = null, sameErrorMessage: String? = null) {
        StatRealService.onChapterError(chapterFeedback, successMessage, sameErrorMessage)
    }

    /**
     * 上传自定义异常
     * @param throwable 异常
     * @param extraParam 额外信息
     */
    @JvmStatic
    fun onError(throwable: Throwable?, extraParam: Map<String, String>? = null) {
        StatRealService.onError(throwable, extraParam)
    }
}


