package com.dingyue.statistics.aliyun;

import android.util.Base64;

import com.dingyue.statistics.DyStatService;
import com.dingyue.statistics.core.LogCollectorStorage;
import com.dingyue.statistics.core.entity.LogGroup;
import com.dingyue.statistics.core.entity.ServerLog;
import com.dingyue.statistics.exception.LogException;
import com.dingyue.statistics.time.TimeManager;
import com.dingyue.statistics.utils.SdkLog;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.zip.Deflater;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import kotlin.Unit;
import kotlin.jvm.functions.Function0;

/**
 * Created by wangjwchn on 16/8/2.
 */
public class LogClient {
    private String mEndPoint;
    private String mAccessKeyID;
    private String mAccessKeySecret;
    private String mAccessToken;
    private String mProject;
    private String protocol;

    public LogClient(String endPoint, String appKey, String appSecret) {
        protocol = "http://";
        if (!"".equals(endPoint)) {
            mEndPoint = endPoint;
        } else {
            throw new NullPointerException("endpoint is null");
        }
        if (mEndPoint.startsWith("http://")) {
            mEndPoint = mEndPoint.substring("http://".length());
        } else if (mEndPoint.startsWith("https://")) {
            mEndPoint = mEndPoint.substring("https://".length());
            protocol = "https://";
        }
        while (mEndPoint.endsWith("/")) {
            mEndPoint = mEndPoint.substring(0, mEndPoint.length() - 1);
        }

        if (!"".equals(appKey)) {
            mAccessKeyID = appKey;
        } else {
            throw new NullPointerException("appKey is null");
        }

        if (!"".equals(appSecret)) {
            mAccessKeySecret = appSecret;
        } else {
            throw new NullPointerException("appSecret is null");
        }

        mAccessToken = "";
    }

    public void setProject(String mProject) {
        this.mProject = mProject;
    }

    private static String getMGTTime() {
        SimpleDateFormat sdf = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
        sdf.setTimeZone(TimeZone.getTimeZone("GMT")); // 设置时区为GMT
        return sdf.format(TimeManager.Companion.serviceTimeMillis());
    }

    private static String hmac_sha1(String encryptText, String encryptKey) throws Exception {
        byte[] keyBytes = encryptKey.getBytes("UTF-8");
        byte[] dataBytes = encryptText.getBytes("UTF-8");
        Mac mac = Mac.getInstance("HmacSHA1");
        mac.init(new SecretKeySpec(keyBytes, "HmacSHA1"));

        return Base64.encodeToString(mac.doFinal(dataBytes), Base64.NO_WRAP);
    }

    public void postLog(final LogGroup logGroup, String logStoreName) throws LogException {
        final String httpUrl = protocol + mProject + "." + mEndPoint + "/logstores/" + logStoreName + "/shards/lb";
        byte[] httpPostBody = null;
        try {
            String json = logGroup.toJsonString();
            if (null != json) {
                httpPostBody = json.getBytes("UTF-8");
            }
        } catch (UnsupportedEncodingException e) {
            throw new LogException("LogClientError", "Failed to pass log to utf-8 bytes", e, "");
        }
        if (null == httpPostBody) {
            return;
        }
        final byte[] httpPostBodyZipped = gzipFrom(httpPostBody);
        final Map<String, String> httpHeaders = getHttpHeadersFrom(logStoreName, httpPostBody, httpPostBodyZipped);
        this.httpPostRequest(logGroup.getLogs(), httpUrl, httpHeaders, httpPostBodyZipped);
    }

    private long minResponseTime = Long.MAX_VALUE;

    private void httpPostRequest(final List<ServerLog> logList, String url, Map<String, String> headers, byte[] body) throws LogException {
        URL u;
        long startTime = System.nanoTime();
        try {
            u = new URL(url);
        } catch (MalformedURLException e) {
            DyStatService.onError(new LogException("LogClientError", "illegal post url", e, ""), null);
            return;
        }

        HttpURLConnection conn;
        try {
            conn = (HttpURLConnection) u.openConnection();
        } catch (IOException e) {
            throw new LogException("LogClientError", "fail to create HttpURLConnection", e, "");
        }

        try {
            conn.setRequestMethod("POST");
        } catch (ProtocolException e) {
            DyStatService.onError(new LogException("LogClientError", "fail to set http request method to  POST", e, ""), null);
            return;
        }
        conn.setDoOutput(true);

        for (Map.Entry<String, String> entry : headers.entrySet()) {
            conn.setRequestProperty(entry.getKey(), entry.getValue());
        }

        SdkLog.INSTANCE.d("prepare to write data to net stream.");

        try {
            DataOutputStream wr = new DataOutputStream(conn.getOutputStream());
            wr.write(body);
            wr.flush();
            wr.close();
        } catch (IOException e) {
            throw new LogException("LogClientError", "fail to post data to URL:" + url, e, "");
        }

        long responseTime = System.nanoTime() - startTime;

        if (responseTime < minResponseTime) {
            long date = conn.getDate();
            if (date != 0) {
                TimeManager.Companion.getInstance().initServerTime(date);
                minResponseTime = responseTime;
            } else  {
                SdkLog.INSTANCE.e("TimeManager: --- 获取服务器时间失败!!!");
            }
        }

        if (logList != null && !logList.isEmpty()) {
            for (ServerLog log: logList) {
                Function0<Unit> serviceBack = log.getServiceBack();
                if (serviceBack != null) {
                    serviceBack.invoke();
                    return;
                }
            }
        }

        try {
            int responseCode = conn.getResponseCode();
            String request_id = conn.getHeaderField("x-log-requestid");

            if (request_id == null) {
                request_id = "";
            }
            SdkLog.INSTANCE.d("push response: "+responseCode);
            if (responseCode != 200) {
                InputStream error_stream = conn.getErrorStream();
                if (error_stream != null) {
                    BufferedReader in = new BufferedReader(
                            new InputStreamReader(error_stream));
                    String inputLine;
                    StringBuilder response = new StringBuilder();

                    while ((inputLine = in.readLine()) != null) {
                        response.append(inputLine);
                    }
                    in.close();
                    checkError(response.toString(), request_id);
                    throw new LogException("LogServerError", "Response code:"
                            + String.valueOf(responseCode) + "\nMessage:"
                            + response.toString(), request_id);
                } else {
                    throw new LogException("LogServerError", "Response code:"
                            + String.valueOf(responseCode)
                            + "\nMessage: fail to connect to the server",
                            request_id);
                }
            }// else success
            if (null != logList) {
                LogCollectorStorage.INSTANCE.onLogPushSuccess(logList);
            }
        } catch (IOException e) {
            throw new LogException("LogServerError", "Failed to parse response data", "");
        }
    }

    private void checkError(String error_message, String request_id) throws LogException {
        try {
            JSONObject obj = new JSONObject(error_message);
            if (obj.has("errorCode") && obj.has("errorMessage")) {
                throw new LogException(obj.getString("errorCode"), obj.getString("errorMessage"), request_id);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private Map<String, String> getHttpHeadersFrom(String logStoreName, byte[] body, byte[] bodyZipped) throws LogException {
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("x-log-apiversion", "0.6.0");
        headers.put("x-log-signaturemethod", "hmac-sha1");
        headers.put("Content-Type", "application/json");
        headers.put("Date", getMGTTime());
        headers.put("Content-MD5", parseToMd5U32From(bodyZipped));
        headers.put("Content-Length", String.valueOf(bodyZipped.length));
        headers.put("x-log-bodyrawsize", String.valueOf(body.length));
        headers.put("x-log-compresstype", "deflate");
        headers.put("Host", mProject + "." + mEndPoint);

        StringBuilder signStringBuf = new StringBuilder("POST" + "\n").
                append(headers.get("Content-MD5") + "\n").
                append(headers.get("Content-Type") + "\n").
                append(headers.get("Date") + "\n");
        String token = mAccessToken;
        if (token != null && token != "") {
            headers.put("x-acs-security-token", token);
            signStringBuf.append("x-acs-security-token:" + headers.get("x-acs-security-token") + "\n");
        }
        signStringBuf.append("x-log-apiversion:0.6.0\n").
                append("x-log-bodyrawsize:" + headers.get("x-log-bodyrawsize") + "\n").
                append("x-log-compresstype:deflate\n").
                append("x-log-signaturemethod:hmac-sha1\n").
                append("/logstores/" + logStoreName + "/shards/lb");
        String signString = signStringBuf.toString();
        try {
            String sign = hmac_sha1(signString, mAccessKeySecret);
            headers.put("Authorization", "LOG " + mAccessKeyID + ":" + sign);
        } catch (Exception e) {
            throw new LogException("LogClientError", "fail to get encode signature", e, "");
        }
        return headers;
    }

    private String parseToMd5U32From(byte[] bytes) throws LogException {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            String res = new BigInteger(1, md.digest(bytes)).toString(16).toUpperCase();
            StringBuilder zeros = new StringBuilder();
            for (int i = 0; i + res.length() < 32; i++) {
                zeros.append("0");
            }
            return zeros.toString() + res;
        } catch (NoSuchAlgorithmException e) {
            throw new LogException("LogClientError", "Not Supported signature method " + "MD5", e, "");
        }
    }

    private byte[] gzipFrom(byte[] jsonByte) throws LogException {
        ByteArrayOutputStream out = null;
        Deflater compresser = new Deflater();
        try {
            out = new ByteArrayOutputStream(jsonByte.length);
            compresser.setInput(jsonByte);
            compresser.finish();
            byte[] buf = new byte[10240];
            while (!compresser.finished()) {
                int count = compresser.deflate(buf);
                out.write(buf, 0, count);
            }
            return out.toByteArray();
        } catch (Exception e) {
            throw new LogException("LogClientError", "fail to zip data", "");
        } finally {
            compresser.end();
            try {
                if (null != out && out.size() != 0) {
                    out.close();
                }
            } catch (IOException e) {
            }
        }
    }
}
