package com.dingyue.statistics.common

import com.dingyue.statistics.activitylife.ActivityLifecycleHelper
import com.dingyue.statistics.core.LogCollectorStorage
import com.dingyue.statistics.core.entity.ServerLog
import com.dingyue.statistics.core.EventLogRecorder
import com.dingyue.statistics.db.dao.bean.LogType
import com.dingyue.statistics.utils.SdkLog
import com.dingyue.statistics.utils.DataFieldTransfer
import java.io.PrintWriter
import java.io.StringWriter

/**
 * Desc 崩溃日志捕捉
 * Author jiaxing_sun
 * Mail jiaxing_sun@dingyuegroup.cn
 * Date 2018/8/23 11:09
 */
class CrashHandler(val parent: Thread.UncaughtExceptionHandler) : Thread.UncaughtExceptionHandler {

    val lock = java.lang.Object()

    override fun uncaughtException(thread: Thread?, throwable: Throwable?) {
        if (thread == null || throwable == null) {
            parent.uncaughtException(thread, throwable)
            return
        }

        //防止死循环
        Thread.setDefaultUncaughtExceptionHandler(parent)

        Thread {
            recordCrashLog(throwable, thread, "app")
            synchronized(lock) {
                lock.notify()
            }
        }.start()

        synchronized(lock) {
            lock.wait(5000)
            // 必须交给上级处理
            parent.uncaughtException(thread, throwable)
        }
    }

    companion object {
        fun recordCrashLog(throwable: Throwable?, thread: Thread?, tag: String, extraParam: Map<String, String>? = null) {
            if (null == throwable) {
                return
            }
            val stackTrace = StringWriter()
            throwable.printStackTrace(PrintWriter(stackTrace))

            val log = CommonParams.appendCommonParams(ServerLog(PLItemKey.ZN_APP_CRASH))

            log.eventType = LogType.MINORITY

            log.appendContent("activity_life", ActivityLifecycleHelper.getActivityNames())
            log.appendContent("envents", EventLogRecorder.getEvents())
            log.appendContent("crash_thread", "${thread?.id}:${thread?.name}")
            log.appendContent("crash_type", throwable.javaClass.name)
            log.appendContent("crash_cause", throwable.cause.toString())

            try {
                log.appendContent("crash_stack", stackTrace.toString().replace("\n\t", "#@").replace("\n", "#").trim())
            } catch (e: Exception) {
                e.printStackTrace()
                log.appendContent("crash_stack", stackTrace.toString())
            }

            log.appendContent("catch_type", tag)

            if (extraParam != null) {
                log.appendContent("data", DataFieldTransfer.transfer(extraParam))
            }

            LogCollectorStorage.accept(log)
            SdkLog.e(PLItemKey.ZN_APP_CRASH.key, log.content.toString())
        }
    }
}