package com.dingyue.statistics.common;

import android.Manifest;
import android.content.Context;
import android.text.TextUtils;

import com.dingyue.statistics.application.ApplicationWrapper;
import com.dingyue.statistics.utils.AppInfo;
import com.dingyue.statistics.utils.DeviceInfo;
import com.dingyue.statistics.utils.PermissionGrant;
import com.dingyue.statistics.utils.ScreenInfo;
import com.dingyue.statistics.utils.SdkLog;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * @author Ling Ji Nian
 *
 * 环境数据收集,用到的地方从这里进行获取
 */
public class EnvCollectionConfig {
    private static EnvCollectionConfig instance = null;
    private EnvCollectionConfig() {}

    public static EnvCollectionConfig newInstance() {
        if (null == instance) {
            synchronized (EnvCollectionConfig.class) {
                if (null == instance) {
                    instance = new EnvCollectionConfig();
                    instance.init();
                }
            }
        }

        return instance;
    }

    private Map<String, String> params = new HashMap<>();

    private void init() {
        Context context = ApplicationWrapper.getInstance().getContext();
        if (null == context) {
            SdkLog.INSTANCE.e("初始化时机异常，context未赋值[可能在非主线程做了初始化]");
            return;
        }
        ScreenInfo si = ScreenInfo.newInstance(context);

        addPrams(ParamKey.OS_TYPE, "android");
        addPrams(ParamKey.MODEL, DeviceInfo.getPhoneModel());//手机型号
        addPrams(ParamKey.SYSTEM_VERSION, DeviceInfo.getRelease());//系统版本号
        addPrams(ParamKey.RESOLUTION, String.format(Locale.getDefault(), "%d*%d", si.screenWidth, si.screenHeight));//屏幕分辨率
        addPrams(ParamKey.MAC_ADDRESS, DeviceInfo.getMacAddress(context));//MAC地址
        addPrams(ParamKey.WLAN_NAME, DeviceInfo.getWLanMacName(context));//WLAN名称
        addPrams(ParamKey.WEBSITE, DeviceInfo.getNetState(context));//网络类型
        addPrams(ParamKey.IP, DeviceInfo.getIPAddress(context));//IP
//        addPrams(ParamKey.BLUE, DeviceInfo.getBluetoothID());//蓝牙
        addPrams(ParamKey.STORAGE, DeviceInfo.getSDTotalSize(context));//内存大小
        addPrams(ParamKey.STORAGE_USED, DeviceInfo.getSDAvailableSize(context));//已使用内存
        addPrams(ParamKey.CPU, DeviceInfo.getCpuName());//处理器
        addPrams(ParamKey.CORE_VERSION, DeviceInfo.getSystemInnerVersion());//内核版本
        addPrams(ParamKey.BATTERY, DeviceInfo.getBattery(context));//电池电量
        addPrams(ParamKey.X86_ARCH, DeviceInfo.getX86());//X86架构
        addPrams(ParamKey.VPN, String.valueOf(DeviceInfo.getIsVPNUsed()));//是否启用VPN
        addPrams(ParamKey.WIFI_MAC, DeviceInfo.getWifiMacAddress(context));//WiFi-Mac地址
        addPrams(ParamKey.NETWORK_TYPE, DeviceInfo.getNetState(context));
        addPrams(ParamKey.BRAND, DeviceInfo.getPhoneBrand());
        addPrams(ParamKey.VENDOR, DeviceInfo.getPhoneBrand() + "," + DeviceInfo.getPhoneModel() + "," + DeviceInfo.getRelease());
        addPrams(ParamKey.RESOLUTION_RATIO, DeviceInfo.getMetrics(context));

        AppInfo ai = AppInfo.getInstance().init(context);
        addPrams(ParamKey.PACKAGE_NAME, ai.applicationId);
        addPrams(ParamKey.VERSION_NAME, ai.versionName);
        addPrams(ParamKey.VERSION_CODE, String.valueOf(ai.versionCode));
        addPrams(ParamKey.INSTALL_TIME, String.valueOf(ai.installTime));
        addPrams(ParamKey.UPDATE_TIME, String.valueOf(ai.lastUpdateTime));

        this.rebindByPermission();

        Location.getInstance();

//        this.print();
    }

    public void rebindByPermission() {
        Context context = ApplicationWrapper.getInstance().getContext();
        if (PermissionGrant.hasPermission(context, Manifest.permission.READ_PHONE_STATE)) {
            addPrams(ParamKey.PHONE_IDENTITY, DeviceInfo.getIMEI(context));
        }
        if (PermissionGrant.hasPermission(context, Manifest.permission.READ_PHONE_STATE)) {
            addPrams(ParamKey.PROVIDERS, DeviceInfo.getProvidersName(context));//运营商
        }
        if (PermissionGrant.hasPermission(context, Manifest.permission.READ_PHONE_STATE)) {
            addPrams(ParamKey.MEID, DeviceInfo.getIMEI(context));//MEID
        }
    }

    public void addPrams(@ParamKey String key, String value) {
        if (params.containsKey(key) && !TextUtils.isEmpty(params.get(key))) {
            return;
        }
        this.set(key, value);
    }

    public Map<String, String> getParams() {
        return params;
    }

    public String get(@ParamKey String key) {
        this.flush(key);
        return params.containsKey(key) ? params.get(key) : "";
    }

    public void set(@ParamKey String key, String value) {
        if (!TextUtils.isEmpty(value)) {
            value = value.replaceAll(" ", "_");
        }
        params.put(key, value);
    }

    private void flush(@ParamKey String key) {
        Context context = ApplicationWrapper.getInstance().getContext();
        if (ParamKey.BATTERY.equals(key)) {
            set(ParamKey.BATTERY, DeviceInfo.getBattery(context));
        }
    }

    private void print() {
        SdkLog.INSTANCE.e("--[start print env] ----------------->");
        for (String key: params.keySet()) {
            SdkLog.INSTANCE.d(key+" : "+params.get(key));
        }
        SdkLog.INSTANCE.e("--[end print env] ----------------->");
    }

    public @interface ParamKey {
        String OS_TYPE = "os_type";
        String RESOLUTION_RATIO = "resolution_ratio";
        String VENDOR = "vendor";
        String PROVIDERS = "operator";//运营商
        String WEBSITE = "website";
        String NETWORK_TYPE = "network_type";
        String CITY_INFO = "city_info";
        String LOCATION_DETAIL = "location_detail";
        String PHONE_IDENTITY = "phone_identity";
        String MODEL = "model";
        String SYSTEM_VERSION = "sys_version";
        String RESOLUTION = "resolution";
        String MAC_ADDRESS = "mac_addr";
        String WLAN_NAME = "wlan_name";
        String PACKAGE_NAME = "package_name";
        String VERSION_NAME = "version_name";
        String VERSION_CODE = "version_code";
        String CHANNEL_ID = "channel_id";
        String UDID = "udid";
        String LONGITUDE = "longitude";
        String LATITUDE = "latitude";
        String CITY_CODE = "city_code";//城市编码
        String PROVINCE = "province";
        String CITY = "city";
        String AREA_CODE = "area_code";//地区编码
        String IP = "ip";
        String BLUE = "blue";
        String STORAGE = "storage";
        String STORAGE_USED = "storage_used";
        String CPU = "cpu";
        String CORE_VERSION = "core_version";
        String BATTERY = "battery";
        String X86_ARCH = "x86_arch";
        String VPN = "vpn";
        String MEID = "meid";
        String WIFI_MAC = "wifi_mac";
        String BRAND = "brand";
        String INSTALL_TIME = "install_time";
        String UPDATE_TIME = "update_time";
    }
}
