package com.dingyue.statistics.core

import android.Manifest
import android.database.sqlite.SQLiteDiskIOException
import android.database.sqlite.SQLiteFullException
import com.dingyue.statistics.application.ApplicationWrapper
import com.dingyue.statistics.common.CommonParams
import com.dingyue.statistics.config.AndroidLogConfig
import com.dingyue.statistics.config.Config
import com.dingyue.statistics.core.entity.ServerLog
import com.dingyue.statistics.db.dao.bean.LocalLog
import com.dingyue.statistics.db.dao.bean.LogType
import com.dingyue.statistics.db.repository.SdkRepositoryFactory
import com.dingyue.statistics.fix.PageCodeOnSystemFix
import com.dingyue.statistics.utils.DateUtil
import com.dingyue.statistics.utils.FileUtil
import com.dingyue.statistics.utils.PermissionGrant
import com.dingyue.statistics.utils.SdkLog
import java.util.concurrent.ConcurrentLinkedQueue
import java.util.concurrent.Executors
import java.util.concurrent.ThreadFactory
import java.util.concurrent.atomic.AtomicInteger

/**
 * Desc AndroidLog存储上传功能
 * Author qiantao
 * Mail tao_qian@dingyuegroup.cn
 * Date 2018/3/19 0019 15:36
 *
 * 日志收集器
 * 1. 数据处理
 * 2. 入库操作
 * 3. 触发trigger动作
 */
object LogCollectorStorage {

    private var latestMillis = System.currentTimeMillis()

    private var logRepository = SdkRepositoryFactory.loadSdkRepository().logRepository

    private val dbSingleThread = Executors.newSingleThreadExecutor(object :ThreadFactory{
        private val atomicInteger = AtomicInteger(0)
        override fun newThread(r: Runnable?): Thread {
            val c = atomicInteger.incrementAndGet()
            return Thread(r,"$c+统计SDK:dbSingleThread executor")
        }
    })

    private val majorityLogQueue = ConcurrentLinkedQueue<LocalLog>()

    /**
     * 接收log
     */
    fun accept(serverLog: ServerLog) {

        //尝试修复
        PageCodeOnSystemFix.tryFixSystemEvent(serverLog)

        // 参数校验
        try {
            CommonParams.paramsVerify(serverLog)
        } catch (e: Exception) {
            SdkLog.e("error:${e.message}")
            if (StatRealService.config.enableSaveLog) {
                StatRealService.toastUtil?.postMessage(e.message.orEmpty())
            }
            return
        }
        // 添加统一log
        SdkLog.e("sdklog", serverLog.content.toString())
        // 写入文本文件
        if (StatRealService.config.enableSaveLog && PermissionGrant.hasPermission(ApplicationWrapper.getInstance().context, Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            val file = Config.getSdkLogPath(ApplicationWrapper.getInstance().context) + DateUtil.format(System.currentTimeMillis()) + "/${serverLog.content["logstore"]}.txt"
            FileUtil.appendLog(file, serverLog.content.toString())
        }

        val type = serverLog.eventType
        val localLog = LocalLog(type, serverLog.content)

        when (type) {
            LogType.MINORITY -> {//直接存入数据库
                dbSingleThread.execute {
                    SdkLog.e("type $type")
                    if (!insertLog2DB(arrayListOf(localLog))) return@execute

                    LogSenderTrigger.triggerLogPush(type)
                }
            }
            LogType.IMMEDIATELY -> {//立刻上传
                dbSingleThread.execute {
                    LogSenderTrigger.triggerLogPushNow(localLog)
                }
            }
            LogType.MAJORITY -> {
                majorityLogQueue.add(localLog)
                SdkLog.e("type $type 入队列 ${majorityLogQueue.size}")
                val currentMillis = System.currentTimeMillis()

                SdkLog.e("timeS: ${(currentMillis - latestMillis) / 1000}  configCache: ${AndroidLogConfig.CONSUME_TIMEOUT_SEC}")

                val timeOut = (currentMillis - latestMillis) / 1000 >= AndroidLogConfig.CONSUME_TIMEOUT_SEC
                if (timeOut or (majorityLogQueue.size >= AndroidLogConfig.CACHE_SIZE)) {
                    if (timeOut) {
                        SdkLog.e("$type 超时，准备入数据库  size: ${majorityLogQueue.size}")
                    } else {
                        SdkLog.e("$type 队列满，入数据库  size: ${majorityLogQueue.size}")
                    }
                    dbSingleThread.execute {
                        val list = majorityLogQueue.asList(majorityLogQueue.size)
                        if (!insertLog2DB(list)) return@execute

                        //让触发的频率高一些，深层次的频率由trigger本身控制
                        LogSenderTrigger.triggerLogPush(type)
                    }
                }

                //记录一下MAJORITY日志的时间
                latestMillis = currentMillis
            }
        }
    }

    /**
     * 该方法从层次上由 {@link LogCollectorStorage}
     */
    fun onLogPushSuccess(logList: List<ServerLog>) {
        val ids = logList.map {
            it.id
        }.toList()
        try {
            // 删除数据
            logRepository.deleteByIds(ids)
            SdkLog.e("log push success, delete ${logList.size} logs, $ids")
        }catch (e:java.lang.Exception){
            //todo 这里是为了暂时处理数据库报错
            e.printStackTrace()
        }
    }

    @Synchronized
    private fun ConcurrentLinkedQueue<LocalLog>.asList(size: Int): ArrayList<LocalLog> {
        val list = ArrayList<LocalLog>()
        for (i in 0 until size) {
            val log = this.poll()
            if (log != null) {
                list.add(log)
            }
        }
        return list
    }

    private fun insertLog2DB(logs: ArrayList<LocalLog>): Boolean {
        try {
            logRepository.insertOrUpdate(logs)
            return true
        } catch (e: SQLiteFullException) {
            // 数据库满,  清除 按日期排序后的 MAJORITY 类型的200条数据
            e.printStackTrace()

            try {
                val deleteCondition = logRepository.queryByTime(LogType.MAJORITY, 200).last().time
                logRepository.deleteOutOfDate(deleteCondition.toLong())
            } catch (e: Exception) {
                e.printStackTrace()
            }
        } catch (e: SQLiteDiskIOException) {
            // 数据库异常
            e.printStackTrace()
            StatRealService.onError(e)
        } catch (e:Exception){
            //todo 这里是为了暂时处理数据库报错
            e.printStackTrace()
        }
        return false
    }

    fun forceFlush() {
        SdkLog.d("invoke forceFlush() to db .  ${majorityLogQueue.size}")
        if (majorityLogQueue.isNotEmpty()) {
            try {
                insertLog2DB(majorityLogQueue.asList(majorityLogQueue.size))
                majorityLogQueue.clear()
            } catch (e: java.lang.Exception) {}
        }
    }

    fun release() {
        this.forceFlush()
    }
}