package com.dingyue.statistics.core

import com.dingyue.statistics.aliyun.AliyunUploadLogROImpl
import com.dingyue.statistics.application.ApplicationWrapper
import com.dingyue.statistics.config.AndroidLogConfig
import com.dingyue.statistics.core.entity.LogGroup
import com.dingyue.statistics.core.entity.ServerLog
import com.dingyue.statistics.db.dao.bean.LocalLog
import com.dingyue.statistics.db.dao.bean.LogType
import com.dingyue.statistics.db.repository.LogRepository
import com.dingyue.statistics.db.repository.SdkRepositoryFactory
import com.dingyue.statistics.remote.IUploadLogRO
import com.dingyue.statistics.utils.NetworkUtil
import com.dingyue.statistics.utils.SdkLog
import org.jetbrains.annotations.NotNull
import java.util.*
import java.util.concurrent.ConcurrentLinkedQueue

/**
 * modified by yangxiaowei
 * 日志触发器
 * 日志收集器[参数|频率控制]->触发器[从数据库读取日志|分类]->往服务端发送
 */
object LogSenderTrigger : ILogSenderTrigger{

    // 定义发送队列
    private val sendQueue = ConcurrentLinkedQueue<LogGroup>()
    private var senderThread: LogSendThread
    private var uploadLogRO: IUploadLogRO

    private var logRepository: LogRepository

    init {
        senderThread = LogSendThread()
        senderThread.start()
        uploadLogRO = AliyunUploadLogROImpl()
        logRepository = SdkRepositoryFactory.loadSdkRepository().logRepository
    }

    override fun triggerLogPush() {
        this.triggerLogPushInner()
    }


    private fun triggerLogPushInner(cc: Int = 0, start: Int = 0) {
        if(ApplicationWrapper.getInstance().isInited && !NetworkUtil.isNetworkConnected(ApplicationWrapper.getInstance().context)) {
            return
        }
        //控制下次数
        if (cc >= 20) {
            return
        }
        val fetchSize = 600
        //去除所有数据按type优先级
        var logs: List<LocalLog>? = null
        try {
            logs = logRepository.queryIdleAllByLimit(start, fetchSize)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        if (logs?.isEmpty() != false) {
            SdkLog.d("sdk init -> triggerLogPush find log records: [empty]")
            return
        }
        val size = logs.size

        val minorities = ArrayList<LocalLog>()
        val majorities = ArrayList<LocalLog>()
        logs.forEach {
            when (it.type) {
                LogType.MINORITY -> minorities.add(it)
                LogType.MAJORITY -> majorities.add(it)
            }
        }

        SdkLog.d("sdk init -> triggerLogPush  find log records: $size -> minority: ${minorities.size} , majority: ${majorities.size}")

        val lastId = logs[size - 1].id
        (logs as MutableList).clear()

        this.groupLogs(minorities)
        this.groupLogs(majorities)
        if (size >= fetchSize) {
            this.triggerLogPushInner(cc+1, lastId)
        }
    }

    override fun triggerLogPush(@LogType @NotNull logType: String) {
        if (sendQueue.isNotEmpty()) {
            //频次一次控制
            return
        }

        //扩大查询范围[可能会有多组]，具体条数控制在push时
        val size = AndroidLogConfig.send_log_count
        val logs = logRepository.queryIdle(logType, size * 2)
        SdkLog.d("trigger-msg: triggerLogPush -> $logType  query: ${logs.size} , configSize: $size || ${logs.map { it.id }.toList()}")

        if (logType == LogType.MAJORITY) {
            if (logs.size < size * 2) {
                //频次二次控制 对于非实时上传的，攒够一定条数
                (logs as MutableList).clear()
                return
            }
        }

        this.groupLogs(logs)
    }

    fun triggerLogPushNow(localLog: LocalLog) {
        //扩大查询范围[可能会有多组]，具体条数控制在push时
        SdkLog.d("trigger-msg: triggerLogPushNow -> ${LogType.MINORITY}  query: 1 , config Size: 1}")
        this.groupLogs(listOf(localLog))
    }

    private fun groupLogs(logs: List<LocalLog>?) {
        if (!NetworkUtil.isNetworkConnected(ApplicationWrapper.getInstance().context)) {
            SdkLog.w("net work not connected. return .")
            return
        }

        SdkLog.d("prepare to group logs: ${logs?.size}")
        try {
            if (null == logs || logs.isEmpty()) {
                return
            }

            val logList = logs.map {
                ServerLog(it.id, it.type, it.contentJson.orEmpty())
            }.toMutableList()

            SdkLog.d("prepare to mark log record to waiting status")
            logRepository.resetToWaiting(logs.map { it.id }.toList())

            (logs as MutableList).clear()

            //按照project和logstore分组，所以log中必须包含project和logstore两个key
            val map:HashMap<String, List<LogGroup>> = HashMap()
            for (i in logList.indices) {
                val log = logList[i]
                val con = log.content
                if (!con.containsKey("project") || !con.containsKey("logstore")) {
                    continue
                }
                val project = con["project"] as String
                val logstore = con["logstore"] as String
                val unikey = project + "_" + logstore

                var list = map[unikey]
                if(null == list || list[list.size - 1].logs.size >= AndroidLogConfig.send_log_count) {
                    if (null == list) {
                        list = mutableListOf()
                        map[unikey] = list
                    }
                    (list as MutableList).add(LogGroup("", "", project, logstore))
                }

                list[list.size - 1].putLog(log)
            }

            logList.clear()
            SdkLog.d("prepare to add log to queue.")
            if (!map.values.isEmpty()) {
                this.printGroup(map)
                SdkLog.d("handle ......")
                //这里对同一组日志被分批的情况做下处理，比如第一批100条，第二批2条  这样的话完全进行将第二批合并到第一批一起发送
                //主要是将太过分散的数据进行一次压缩，以减少网络次数
                for (list in map.values) {
                    if (list.size > 1) {
                        val last = list[list.size - 1]
                        if (last.logs.size <= 10) {
                            val pre = list[list.size - 2]
                            last.logs.forEach {
                                pre.putLog(it)
                            }
                            (list as MutableList).removeAt(list.size - 1)
                        }
                    }
                }
                this.printGroup(map)
                //发送队列
                for (list in map.values) {
                    for (group in list) {
                        sendQueue.add(group)
                        senderThread.wakeUp()
                    }
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun printGroup(map: HashMap<String, List<LogGroup>>) {
        if (map.isEmpty()) {
            return
        }

        SdkLog.e("[start]print group-------------")
        map.forEach {
            it.value.forEach {log ->
                SdkLog.d("${it.key} -> group: ${it.value.size}  data: ${log.logs.size}")
            }
        }
        SdkLog.e("[end]print group-------------")
    }

    // 开启一个线程专门用来发送数据
    private class LogSendThread : Thread("统计sdk：LogSendThread") {
        private var loop = true
        private val obj = Object()
        override fun run() {
            while (loop) {
                if (sendQueue.isEmpty()) {
                    synchronized(obj) {
                        try {
                            obj.wait()
                        } catch (e: InterruptedException) {
                        }
                    }
                    continue
                }
                val logGroup = sendQueue.poll()
                this.tryPushControl(logGroup)
            }
        }

        fun tryPushControl(logGroup: LogGroup, tryC: Int = 0) {
            SdkLog.d("tryPushControl: 第 ${tryC + 1} 次 - ${logGroup.project} - ${logGroup.logstore}")
            if (tryC >= 3) {
                //3次尝试均失败，需要重置状态，等待下次触发
                logRepository.resetToIdle(logGroup.logs.map { it.id }.toList())
                return
            }
            val project = logGroup.project
            val logstore = logGroup.logstore
            try {
                if (project != null && logstore != null) {
                    uploadLogRO.postLog(logGroup)
                }
            } catch (e: Exception) {
                e.printStackTrace()
                this.tryPushControl(logGroup, tryC + 1)
            }
        }

        fun stopThread() {
            loop = false
            synchronized(obj) {
                obj.notify()
            }
        }

        fun wakeUp() {
            synchronized(obj) {
                obj.notify()
            }
        }
    }

    fun release() {
        try {
            senderThread.stopThread()
            if (!senderThread.isInterrupted) {
                senderThread.interrupt()
            }
        } catch (e: java.lang.Exception) { }

        try {
            sendQueue.clear()
        } catch (e: java.lang.Exception) {}
    }
}
