package com.dingyue.statistics.core

import android.app.Application
import android.content.Context
import android.content.IntentFilter
import android.os.Looper
import android.support.annotation.MainThread
import com.dingyue.statistics.DyStatService
import com.dingyue.statistics.activitylife.ActivityLifecycleHelper
import com.dingyue.statistics.application.ApplicationWrapper
import com.dingyue.statistics.client.Ids
import com.dingyue.statistics.client.SimpleAd
import com.dingyue.statistics.client.SimpleBook
import com.dingyue.statistics.common.*
import com.dingyue.statistics.core.entity.ServerLog
import com.dingyue.statistics.core.monitor.AppVisibilityMonitor
import com.dingyue.statistics.db.dao.bean.BaseChapterFeedback
import com.dingyue.statistics.db.dao.bean.LogType
import com.dingyue.statistics.db.repository.SdkRepositoryFactory
import com.dingyue.statistics.exception.LogException
import com.dingyue.statistics.receiver.NetworkChangeReceiver
import com.dingyue.statistics.time.TimeManager
import com.dingyue.statistics.utils.*
import java.util.*
import java.util.concurrent.Executors
import java.util.concurrent.ThreadFactory
import java.util.concurrent.atomic.AtomicInteger

/**
 * @author 170303 - yangxiaowei
 * @date 2019/8/13 17:41
 * @email hn_yangxiaowei@163.com
 * @desc
 */
object StatRealService {
    var toastUtil: ToastUtil? = null

    private var sameFeedbackTime = 2
    private val prePageList = ArrayList<String>()
    private val statThread by lazy {
        Executors.newFixedThreadPool(5, object : ThreadFactory {
            private val atomicInteger = AtomicInteger(0)
            override fun newThread(r: Runnable?): Thread {
                val c = atomicInteger.incrementAndGet()
                return Thread(r, "$c +:统计SDK：statThread executor")
            }
        })
    }
    private val sharedUtil by lazy { SharedPreferencesUtil(ApplicationWrapper.getInstance().context.getSharedPreferences("log_config", Context.MODE_PRIVATE)) }


    var config: DyStatService.Config = DyStatService.config
        private set(value) {
            field = value
        }

    private fun isDisableStatServiceFunc(): Boolean = !config.enableStatServiceFunc

    @JvmStatic
    fun configBizUserId(userId: String?) {
        CommonParams.bizUserId = userId
    }

    /**
     * @param application 启动的application
     */
    @JvmStatic
    @MainThread
    fun preInit(application: Application) {
        if (Looper.getMainLooper().thread == Thread.currentThread()) {
            ApplicationWrapper.getInstance().init(application)
        } else {
            ApplicationWrapper.getInstance().mainThreadHandler.post {
                // 设置全局Context
                ApplicationWrapper.getInstance().init(application)
            }
        }
    }

    /**
     * SDK初始化
     * @param udid 外部维护的udid,调用者传null标识由SDK内部维护udid
     */
    @JvmStatic
    @JvmOverloads
    @MainThread
    fun init(udid: String? = null, appChannel: String? = null) {
        if (Looper.getMainLooper().thread == Thread.currentThread()) {
            this.initInner(udid, appChannel)
        } else {
            ApplicationWrapper.getInstance().mainThreadHandler.post {
                this.initInner(udid, appChannel)
            }
        }
    }

    @MainThread
    private fun initInner(udid: String? = null, appChannel: String? = null) {

        config = DyStatService.config
        CommonParams.bizUserId = config.loginUserId

        // activity生命周期监控
        ApplicationWrapper.getInstance().application.registerActivityLifecycleCallbacks(ActivityLifecycleHelper.build())

        // 电量监控
        ApplicationWrapper.getInstance().application.registerReceiver(NetworkChangeReceiver.build(), IntentFilter("android.net.conn.CONNECTIVITY_CHANGE"))

        EnvCollectionConfig.newInstance()

        // 初始化Toast辅助类
        toastUtil = ToastUtil()
        // 设置通用参数
        CommonParams.init(udid, appChannel)
        // 设置异常捕捉
        if (config.enableCrashFunc) {
            Thread.setDefaultUncaughtExceptionHandler(CrashHandler(Thread.getDefaultUncaughtExceptionHandler()))
        }

        // 获取服务器时间
        this.recordInitLogs()

        statThread.execute {
            // 清理无效数据
            LogClear.tryClear()
            LogCorrector.tryCorrect()

            //首次对日志数据处理
            LogSenderTrigger.triggerLogPush()
            handleEventBySplit(pageAndIdentify = Ids.PI_SDK_INIT, split = Ids.SPLIT, logType = LogType.MINORITY)
        }

//        ApplicationWrapper.getInstance().registerMonitor(AppVisibilityMonitor.getInstance())
    }

    @JvmStatic
    fun release() {
        if (ApplicationWrapper.getInstance().isInited) {
            LogCollectorStorage.release()
            LogSenderTrigger.release()
            ApplicationWrapper.getInstance().release()
            ApplicationWrapper.getInstance().application.unregisterActivityLifecycleCallbacks(ActivityLifecycleHelper.build())
            ApplicationWrapper.getInstance().application.unregisterReceiver(NetworkChangeReceiver.build())
            SdkRepositoryFactory.loadSdkRepository().release()
        }
    }

    /**
     * 设置位置信息
     * @param longitude 经度
     * @param latitude 纬度
     * @param cityInfo 城市名称
     * @param cityCode 城市编码
     * @param locationDetail 详细地址
     */
    @JvmStatic
    fun setLocationInfo(longitude: Double, latitude: Double, cityInfo: String, cityCode: String, locationDetail: String) {
        SdkLog.e("xxxxx", "setLocationInfo longitude:[$longitude] latitude:[$latitude] cityInfo:[$cityInfo] cityCode:[$cityCode] locationDetail:[$locationDetail]")
        val li = Location.LocationInfo()
        li.longitude = longitude
        li.latitude = latitude
        li.cityCode = cityCode
        li.cityInfo = cityInfo
        li.locationDetail = locationDetail
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("setLocationInfo()  未初始化")
        } else {
            Location.getInstance().config(li)
        }
    }

    /**
     * 打点统计
     * @param pageAndIdentify 格式 pageCode_identify
     * @param extraParam 额外参数
     */
    @JvmStatic
    @JvmOverloads
    fun onEvent(pageAndIdentify: String, extraParam: Map<String, String>? = null) {
        onEventBy(pageAndIdentify, "_", extraParam)
    }

    /**
     * 打点统计
     * @param pageAndIdentify 格式 pageCode[split]identify
     * @param split [pageAndIdentify] 中页面编码与点位编码的分割符，默认为_
     * @param extraParam 额外参数
     */
    @JvmStatic
    @JvmOverloads
    fun onEventBy(pageAndIdentify: String, split: String = "_", extraParam: Map<String, String>? = null) {
        /** 验证 参数格式  pageCode[split]identify **/
        val i = pageAndIdentify.indexOf(split)
        if (split.isBlank() || i == -1) {
            toastUtil?.postMessage("pageAndIdentify 参数格式不正确，请检查 ")
        } else {
            // 获取pageCode和identify
            onEvent(pageAndIdentify.substring(0, i), pageAndIdentify.substring(i + 1), extraParam)
        }
    }

    /**
     * 打点统计
     * @param pageCode 页面标识
     * @param identify 点位标识
     * @param extraParam 附件参数  可为null
     */
    @JvmStatic
    @JvmOverloads
    fun onEvent(pageCode: String, identify: String, extraParam: Map<String, String>? = null) {
        handleEvent(pageCode, identify, extraParam = extraParam)
    }

    @JvmStatic
    private fun handleEventBySplit(pageAndIdentify: String, split: String = "_", @LogType logType: String = LogType.MAJORITY, extraParam: Map<String, String>? = null) {
        val i = pageAndIdentify.indexOf(split)
        if (split.isBlank() || i == -1) {
            toastUtil?.postMessage("pageAndIdentify 参数格式不正确，请检查 ")
        } else {
            // 获取pageCode和identify
            handleEvent(pageAndIdentify.substring(0, i), pageAndIdentify.substring(i + 1), logType, extraParam)
        }
    }

    @JvmStatic
    private fun handleEvent(pageCode: String, identify: String, @LogType logType: String = LogType.MAJORITY, extraParam: Map<String, String>? = null) {
        if (this.isDisableStatServiceFunc()) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("handleEvent($pageCode - $identify - $logType)  未初始化")
            return
        }
        statThread.execute {
            // 获取通用参数
            val log = CommonParams.appendCommonParams(ServerLog(PLItemKey.ZN_APP_EVENT))

            // 追加特有参数
            log.appendContent("page_code", pageCode) // 页面标识
            log.appendContent("code", identify) //点击事件唯一标识
            log.appendContent("pre_page_code", getPrePageCode(pageCode)) // 前一页标识
            log.appendContent("log_time", TimeManager.serviceTimeMillis().toString())//日志产生时间（毫秒数）


            EventLogRecorder.updateEvent(pageCode, identify)

            //事件对应的额外的参数部分
            if (extraParam != null) {
                log.appendContent("data", DataFieldTransfer.transfer(extraParam))
            }

            if (identify == Ids.ID_APP_INIT || identify.contentEquals(Ids.ID_APP_PUSH_NOW)) {
                // App启动点位，立即上传日志
                log.eventType = LogType.IMMEDIATELY
            } else {
                log.eventType = logType
            }

            LogCollectorStorage.accept(log)


        }
    }

    @JvmStatic
    @JvmOverloads
    fun onAdShowEvent(book: SimpleBook, ad: SimpleAd, extraParam: Map<String, String>? = null) {
        handleAdEvent(Ids.ID_AD_SHOW, book, ad, extraParam)
    }

    @JvmStatic
    @JvmOverloads
    fun onAdClickEvent(book: SimpleBook, ad: SimpleAd, extraParam: Map<String, String>? = null) {
        handleAdEvent(Ids.ID_AD_CLICK, book, ad, extraParam)
    }

    @JvmStatic
    private fun handleAdEvent(type: String, book: SimpleBook, ad: SimpleAd, extraParam: Map<String, String>? = null) {
        if (this.isDisableStatServiceFunc()) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("handleAdEvent()  未初始化")
            return
        }
        statThread.execute {
            if (!book.validateForAd() || !ad.validateForAd()) {
                if (config.enableToast) {
                    toastUtil?.postMessage("参数缺失")
                }
                SdkLog.e("参数缺失")
                return@execute
            }

            val log = CommonParams.appendCommonImportantParams(ServerLog(PLItemKey.ZN_AD))

            log.appendContent("code", type)
            log.appendContent("log_time", TimeManager.serviceTimeMillis().toString())//日志产生时间（毫秒数）

            log.appendContent("mark_id", ad.adSlotId)
            log.appendContent("request_id", ad.requestId)

            log.appendContent("book_id", book.bookId)
            log.appendContent("chapter_id", book.chapterId)

            log.appendContent("show_num", if (Ids.ID_AD_SHOW == type) {
                1
            } else {
                0
            }.toString())
            log.appendContent("click_num", if (Ids.ID_AD_CLICK == type) {
                1
            } else {
                0
            }.toString())

            if (extraParam != null) {
                log.appendContent("data", DataFieldTransfer.transfer(extraParam))
            }

            log.eventType = LogType.MINORITY

            LogCollectorStorage.accept(log)
        }
    }

    @JvmStatic
    @JvmOverloads
    fun sendPVData(book: SimpleBook) {
        if (this.isDisableStatServiceFunc()) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("sendPVData()  未初始化")
            return
        }
        statThread.execute {
            if (!book.validateForPV()) {
                if (config.enableToast) {
                    toastUtil?.postMessage("参数缺失")
                }
                SdkLog.e("参数缺失")
                return@execute
            }
            // 获取通用参数
            val log = CommonParams.appendCommonImportantParams(ServerLog(PLItemKey.ZN_PV))

            // 追加特有参数
            log.appendContent("book_id", book.bookId)
            log.appendContent("book_source_id", book.sourceIds)
            log.appendContent("chapter_id", book.chapterId)
            log.appendContent("channel_code", book.channel)
            log.appendContent("chapter_read", "1")
            log.appendContent("listen", if (book.isListen) "1" else "0")
            log.appendContent("chapter_pages", book.chapterPages.toString())
            log.appendContent("start_time", (book.startReadTimeMill / 1000L).toString())
            log.appendContent("end_time", (System.currentTimeMillis() / 1000L).toString())

            LogCollectorStorage.accept(log)
        }
    }

    /**
     * 章节反馈
     * successMessage为反馈成功提示信息,默认为 反馈成功
     * sameErrorMessage为同一章节同一问题反馈次数超过sameFeedbackTime预设定的值时的提示信息,默认为 请勿重复反馈
     */
    @JvmStatic
    @JvmOverloads
    fun onChapterError(chapterFeedback: BaseChapterFeedback?, successMessage: String? = null, sameErrorMessage: String? = null) {
        if (this.isDisableStatServiceFunc() || chapterFeedback == null) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("onChapterError()  未初始化")
            return
        }
        statThread.execute {
            // 判断同一章同一问题反馈是否超过次数
            if (chapterFeedback.bookName.isNullOrBlank() || chapterFeedback.chapterId.isNullOrBlank()) return@execute

            val repository = SdkRepositoryFactory.loadSdkRepository().chapterRepository
            val count = repository.getSameFeedbackCount(chapterFeedback.bookName!!, chapterFeedback.chapterId!!, chapterFeedback.type)
            if (count >= sameFeedbackTime) {
                toastUtil?.postMessage(sameErrorMessage ?: "请勿重复反馈问题")
                return@execute
            }
            // 存储章节反馈
            repository.insertOrUpdate(chapterFeedback)

            // 获取通用参数
            val log = CommonParams.appendCommonParams(ServerLog(PLItemKey.ZN_APP_FEEDBACK))

            // 追加特有参数
            chapterFeedback.toMap().map { log.appendContent(it.key, it.value) }

            LogCollectorStorage.accept(log)
            toastUtil?.postMessage(successMessage ?: "反馈成功")
        }
    }

    /**
     * 上传自定义异常
     * @param throwable 异常
     * @param extraParam 额外信息
     */
    @JvmStatic
    fun onError(throwable: Throwable?, extraParam: Map<String, String>? = null) {
        if (this.isDisableStatServiceFunc() || throwable == null) return
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("onError()  未初始化")
            return
        }
        val thread = Thread.currentThread()
        statThread.execute {
            CrashHandler.recordCrashLog(throwable, thread, "custom", extraParam)
        }
    }

    private fun recordInitLogs() {
        this.sendZnUserLog()
        this.upLoadApps()
    }

    /**
     * 上传App列表
     */
    private fun upLoadApps() {
        if (this.isDisableStatServiceFunc()) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("upLoadApps()  未初始化")
            return
        }
        statThread.execute {
            try {
                if (AppUtil.isSameDay(sharedUtil.getLong("upload_app_time"), System.currentTimeMillis())) return@execute // 每天上传一次
                // 获取通用参数
                val log = CommonParams.appendCommonImportantParams(ServerLog(PLItemKey.ZN_APP_APPSTORE))

                // 追加特有参数
                log.appendContent("apps", AppUtil.scanLocalInstallAppList(ApplicationWrapper.getInstance().context.packageManager))
                log.appendContent("data", AppUtil.loadUserApplicationList(ApplicationWrapper.getInstance().context))
                log.appendContent("time", TimeManager.serviceTimeMillis().toString())

                LogCollectorStorage.accept(log)
                sharedUtil.putLong("upload_app_time", System.currentTimeMillis()) // 记录上传时间
            } catch (e: Exception) {
                onError(LogException("DyStatServiceError", "fail to send upLoadApps(上传App列表)", e, ""))
            }
        }
    }

    /**
     * 上传用户信息
     */
    private fun sendZnUserLog() {
        if (this.isDisableStatServiceFunc()) {
            return
        }
        if (!ApplicationWrapper.getInstance().isInited) {
            SdkLog.e("sendZnUserLog()  未初始化")
            return
        }
        statThread.execute {
            if (AppUtil.isSameDay(sharedUtil.getLong("upload_user_time"), System.currentTimeMillis())) return@execute // 每天上传一次
            // 获取通用参数
            val log = CommonParams.appendCommonParams(ServerLog(PLItemKey.ZN_USER))

            LogCollectorStorage.accept(log)
            sharedUtil.putLong("upload_user_time", System.currentTimeMillis()) // 记录上传时间
        }
    }

    /**
     * 获取prePageCode
     */
    @Synchronized
    private fun getPrePageCode(pageCode: String): String {
        if (prePageList.size == 0 || (prePageList.size > 0 && prePageList[prePageList.size - 1] != pageCode)) {
            prePageList.add(pageCode)
            removePre(prePageList)
        }
        return if (prePageList.size != 0) {
            prePageList.map { SdkLog.e(it) }
            if (prePageList.size > 1) {
                prePageList[prePageList.size - 2]
            } else {
                prePageList[prePageList.size - 1]
            }
        } else ""
    }

    /**
     * 移除prePageCode
     */
    private fun removePre(prePageList: MutableList<String>) {
        if (prePageList.size > 6) {
            for (i in 0..1) {
                prePageList.remove(prePageList[i])
            }
        }
    }

    @JvmStatic
    @Synchronized
    fun onLogPushSuccess(logList: List<ServerLog>) {
        LogCollectorStorage.onLogPushSuccess(logList)
    }
}