package net.iyouqu.video.app.imagecache;

import net.iyouqu.video.app.data.LruBitmapCache;
import net.iyouqu.video.app.request.RequestManager;
import android.content.Context;

import com.android.volley.toolbox.ImageLoader;
import com.android.volley.toolbox.ImageLoader.ImageCache;

public class ImageCacheManager {

	public enum ImageCacheType {
		DISK, MEMORY, COMPLEX
	}

	/**
	 * Volley recommends in-memory L1 cache but both a disk and memory cache are
	 * provided. Volley includes a L2 disk cache out of the box but you can
	 * technically use a disk cache as an L1 cache provided you can live with
	 * potential i/o blocking.
	 * 
	 */
	private static ImageCacheManager mInstance;

	/**
	 * Volley image loader
	 */
	private ImageLoader mImageLoader;

	/**
	 * Image cache implementation
	 */
	private ImageCache mImageCache;

	/**
	 * @return instance of the cache manager
	 */
	public static ImageCacheManager getInstance() {
		if (mInstance == null) {
			initInstance();
		}

		return mInstance;
	}

	private synchronized static void initInstance() {
		if (mInstance == null) {
			mInstance = new ImageCacheManager();
		}

	}

	/**
	 * Initializer for the manager. Must be called prior to use.
	 * 
	 * @param context
	 *            application context
	 * @param path
	 *            name for the cache location
	 * @param cacheSize
	 *            max size for the cache
	 * @param compressFormat
	 *            file type compression format.
	 * @param quality
	 */
	public void init(Context context, String path, int cacheSize, ImageCacheType type) {

		switch (type) {
		case DISK:
			mImageCache = new DiskCache(context, path);
			break;
		case MEMORY:
			mImageCache = new BitmapLruImageCache(cacheSize);
//			mImageCache = new LruBitmapCache(LruBitmapCache.getCacheSize(context));
			break;
		case COMPLEX:
			mImageCache = new ComplexImageCache(context, cacheSize, path);
			break;
		default:
			mImageCache = new BitmapLruImageCache(cacheSize);
			break;
		}
		mImageLoader = new ImageLoader(RequestManager.getRequestQueue(), mImageCache);
	}

	/**
	 * @return instance of the image loader
	 */
	public ImageLoader getImageLoader() {
		return mImageLoader;
	}

}
