package net.iyouqu.video.basecommon;

import android.app.Dialog;
import android.content.Context;
import android.content.Intent;
import android.content.res.ColorStateList;
import android.net.ConnectivityManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.os.Process;
import android.support.v4.app.FragmentActivity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup.LayoutParams;
import android.widget.FrameLayout;
import android.widget.Toast;

import net.iyouqu.video.basecommon.manager.NetManager;
import net.iyouqu.video.basecommon.widget.EsDialog;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * 
 * File Name: BaseEsActivity.java
 * Create by:Bruce on 2014-12-12 下午2:40:29 
 * @version 1.0.0
 */
public abstract class BaseActivity extends FragmentActivity {
	/** 判断当前activitys是否销毁 */
	public boolean mIsDestroyed = false;
	/** UI线程 id */
	private int mUIThreadId;
	/** handler */
	private Handler mDefaultHandler = new Handler() {
		public void handleMessage(Message msg) {
			onHandlerMessage(msg);
		};
	};
	/** 记录当前Activity中的所有开启的对话框 */
	private List<EsDialog> mDialogs = new LinkedList<EsDialog>();
	/** 记录出于前台的Activity */
	private static BaseActivity mForegroundActivity = null;
	/** 记录所有活动的Activity */
	private static final List<BaseActivity> mActivities = new LinkedList<BaseActivity>();
	/** 需要在对话框结束之后执行的任务 */
	private static List<Runnable> mToDoAfterDialogDismiss = new LinkedList<Runnable>();
	protected ConnectivityManager mSystemService;
	/**
	 * 获取UI线程ID
	 * 
	 * @return UI线程ID
	 */
	public int getUIThreadId() {
		return mUIThreadId;
	}

	/**
	 * 获取内置handler
	 * 
	 * @return 内置handler
	 */
	public Handler getDefaultHandler() {
		return mDefaultHandler;
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		mUIThreadId = Process.myTid();
		super.onCreate(savedInstanceState);
		mActivities.add(this);
		mSystemService = (ConnectivityManager)getSystemService(Context.CONNECTIVITY_SERVICE);
		init();
		initLayout();
		initData();
	}

	@Override
	protected void onNewIntent(Intent intent) {
		mUIThreadId = Process.myTid();
		super.onNewIntent(intent);
	}

	@Override
	protected void onResume() {
		mForegroundActivity = this;
		super.onResume();
	}

	@Override
	protected void onPause() {
		mForegroundActivity = null;
		super.onPause();
	}

	@Override
	protected void onDestroy() {
		mIsDestroyed = true;
		mActivities.remove(this);
		super.onDestroy();
	}

	protected void onHandlerMessage(Message msg) {

	}

	@Override
	public void setContentView(View view) {
		/* Activity的根View，包装一层，方便做统一的修改 */
		FrameLayout mContentFrame = new FrameLayout(this);
		mContentFrame.addView(view, LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
		super.setContentView(mContentFrame);
	}

	@Override
	public void setContentView(int layoutResID) {
		View view = inflate(layoutResID);
		setContentView(view);
	}

	/**
	 * 根据指定的layout索引，创建一个View
	 * 
	 * @param resId
	 *            指定的layout索引
	 * @return 新的View
	 */
	public View inflate(int resId) {
		LayoutInflater inflater = LayoutInflater.from(this);
		return inflater.inflate(resId, null);
	}

	/**
	 * 向内置handler的消息队列中增加一个任务。该任务会在将来的某一时刻在UI线程执行。
	 * 
	 * @param r
	 *            任务
	 * @return 是否成功
	 */
	public boolean post(Runnable r) {
		return mDefaultHandler.post(r);
	}

	/**
	 * 向内置handler的消息队列中增加一个任务。该任务会在指定延时后的某一时刻在UI线程执行。
	 * 
	 * @param r
	 *            任务
	 * @param delayMillis
	 *            延迟的毫秒数
	 * @return
	 */
	public boolean postDelayed(Runnable r, long delayMillis) {
		return mDefaultHandler.postDelayed(r, delayMillis);
	}

	/** 向内置handler发送消息。 */
	public void sendMessage(int what) {
		mDefaultHandler.sendEmptyMessage(what);
	}

	/** 向内置handler发送消息。 */
	public void sendMessage(int what, Object obj) {
		mDefaultHandler.obtainMessage(what, obj).sendToTarget();
	}

	/** 向内置handler发送消息。 */
	public void sendMessage(int what, int arg1, int arg2) {
		mDefaultHandler.obtainMessage(what, arg1, arg2).sendToTarget();
	}

	/** 向内置handler发送消息。 */
	public void sendMessage(int what, int arg1, int arg2, Object obj) {
		mDefaultHandler.obtainMessage(what, arg1, arg2, obj).sendToTarget();
	}

	/** 从队列中移除回调 */
	public void removeCallback(Runnable r) {
		mDefaultHandler.removeCallbacks(r);
	}

	public void refreshViewSafe() {
		if (Process.myTid() == mUIThreadId) {
			// 调用在UI线程
			refreshView();
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					refreshView();
				}
			});
		}
	}

	/**
	 * 对toast的简易封装。线程安全，可以在非UI线程调用。
	 * 
	 * @param resId
	 *            Toast内容的资源id
	 * @param duration
	 *            Toast的持续时间
	 */
	public void showToastSafe(final int resId, final int duration) {
		if (Process.myTid() == mUIThreadId) {
			// 调用在UI线程
			Toast.makeText(getBaseContext(), resId, duration).show();
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					Toast.makeText(getBaseContext(), resId, duration).show();
				}
			});
		}
	}

	/**
	 * 对toast的简易封装。线程安全，可以在非UI线程调用。
	 * 
	 * @param text
	 *            Toast内容
	 * @param duration
	 *            Toast的持续时间
	 */
	public void showToastSafe(final CharSequence text, final int duration) {
		if (Process.myTid() == mUIThreadId) {
			// 调用在UI线程
			Toast.makeText(getBaseContext(), text, duration).show();
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					Toast.makeText(getBaseContext(), text, duration).show();
				}
			});
		}
	}

	/**
	 * 对showDialog的简易封装。线程安全，可以在非UI线程调用。
	 * 
	 * @param dialog
	 *            Dialog实例
	 */
	public void showDialogSafe(final EsDialog dialog) {
		if (Process.myTid() == mUIThreadId) {
			// 调用在UI线程
			showDialog(dialog);
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					showDialog(dialog);
				}
			});
		}
	}

	private void showDialog(EsDialog dialog) {
		if (!mDialogs.contains(dialog)) {
			dialog.show();
			mDialogs.add(dialog);
		}
	}

	/**
	 * 对dismissDialog的简易封装。线程安全，可以在非UI线程调用。
	 * 
	 * @param dialog
	 *            Dialog实例
	 */
	public void dismissDialogSafe(final EsDialog dialog) {
		if (Process.myTid() == mUIThreadId) {
			// 调用在UI线程
			dismissDialog(dialog);
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					dismissDialog(dialog);
				}
			});
		}
	}

	private void dismissDialog(EsDialog dialog) {
		dialog.dismiss();
		mDialogs.remove(dialog);
		// if (getAllShowingDialogCount() == 0) {
		// runDialogDismissPendingRunnables();
		// }
	}

	/** 关闭所有Activity */
	public static void finishAll() {
		List<BaseActivity> copy;
		synchronized (mActivities) {
			copy = new ArrayList<BaseActivity>(mActivities);
		}
		for (BaseActivity activity : copy) {
			activity.finish();
		}
		copy.clear();
	}
	
	/** 是否有启动的Activity */
	public static boolean hasActivity() {
		return mActivities.size() > 0;
	}
	
	/** 获取当前处于前台的activity */
	public static BaseActivity getForegroundActivity() {
		return mForegroundActivity;
	}

	/** 获取当前处于栈顶的activity，无论其是否处于前台 */
	public static BaseActivity getCurrentActivity() {
		List<BaseActivity> copy;
		synchronized (mActivities) {
			copy = new ArrayList<BaseActivity>(mActivities);
		}
		if (copy.size() > 0) {
			return copy.get(copy.size() - 1);
		}
		return null;
	}
	
	/** 退出应用 */
	public void exitApp() {
		finishAll();
		android.os.Process.killProcess(android.os.Process.myPid());
	}

	/** dip转换px */
	public int dip2px(int dip) {
		final float scale = getResources().getDisplayMetrics().density;
		return (int) (dip * scale + 0.5f);
	}

	/** pxz转换dip */
	public int px2dip(int px) {
		final float scale = getResources().getDisplayMetrics().density;
		return (int) (px / scale + 0.5f);
	}

	public int getDimens(int resId) {
		return getResources().getDimensionPixelSize(resId);
	}

//	public Drawable getDrawableByResource(int resId) {
//		return getResources().getDrawable(resId);
//	}

	public int getColor(int resId) {
		return getResources().getColor(resId);
	}

	public ColorStateList getColorStateList(int resId) {
		return getResources().getColorStateList(resId);
	}
	/*** 判断当前网络 */
	public boolean isNetworkConnected() {
		if (NetManager.getInstance().isNetConnected()) {
			return true;
		} else {
			return false;
		}
	}
	
	public void loadIntent(Class<?> cls) {
		Intent intent = new Intent(this, cls);
		startActivity(intent);
	}
	
	/** 显示通用对话框 new ResultsDialog(this, getStrText(resId)); */
	public void showBaseDialog(Dialog dialog) {
		if (isFinishing() || mIsDestroyed || dialog == null) {
			return;
		}
		dialog.show();
	}
	
	protected void startActivity(Class<?> cls){
		startActivity(new Intent(this, cls).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP));
	}
	
	//初始化相关数据 (在加载视图之前)
	protected abstract void init();
	//加载视图
	protected abstract void initLayout();
	//加载数据
	protected abstract void initData();
	//刷新UI
	protected abstract void refreshView();
}
