package net.iyouqu.video.basecommon.dao;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;

import net.iyouqu.video.basecommon.utils.DateFormattUtils;
import net.iyouqu.video.basecommon.utils.LogUtils;
import net.iyouqu.video.basecommon.utils.StringUtils;

/**
 * File Name: DbColumn.java
 * Create by:Bruce on 2014-12-12 下午5:50:11 
 * @version 1.0.0
 */
public class DbColumn {
	// ==========================================================================
	// 静态常量
	// ==========================================================================
	public final static int NONE = 0x0;
	public final static int ID = 0x1;
	public final static int UNIQUE = 0x2;
	public final static int NOTNULL = 0x4;
	public final static int AUTOINCREMENT = 0x8;
	public final static int FOREIGN = 0x10;
	public final static int TRANSIENT = 0x20;

	// ==========================================================================
	// 全局变量
	// ==========================================================================
	private final Column mAnnotation;
	private final String mName;
	private final Field mField;
	private final Class<?> mType;
	private final Method mGet;
	private final Method mSet;

	private final static ConcurrentHashMap<String, String> mFieldTypesToDb = new ConcurrentHashMap<String, String>();

	static {
		mFieldTypesToDb.put(boolean.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Boolean.class.getName(), "INTEGER");

		mFieldTypesToDb.put(byte[].class.getName(), "BLOB");
		mFieldTypesToDb.put(byte.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Byte.class.getName(), "INTEGER");

		mFieldTypesToDb.put(char.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Character.class.getName(), "INTEGER");

		mFieldTypesToDb.put(short.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Short.class.getName(), "INTEGER");

		mFieldTypesToDb.put(int.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Integer.class.getName(), "INTEGER");

		mFieldTypesToDb.put(long.class.getName(), "INTEGER");
		mFieldTypesToDb.put(Long.class.getName(), "INTEGER");

		mFieldTypesToDb.put(double.class.getName(), "REAL");
		mFieldTypesToDb.put(Double.class.getName(), "REAL");

		mFieldTypesToDb.put(float.class.getName(), "REAL");
		mFieldTypesToDb.put(Float.class.getName(), "REAL");

		mFieldTypesToDb.put(Date.class.getName(), "INTEGER");
		mFieldTypesToDb.put(java.sql.Date.class.getName(), "INTEGER");
		mFieldTypesToDb.put(String.class.getName(), "TEXT");
	}

	// ==========================================================================
	// 构造函数
	// ==========================================================================
	public DbColumn(Class<?> clazz, Field field) {
		if (clazz == null || field == null) {
			throw new RuntimeException("构建Column时，实体类型和字段不能为null");
		}
		mField = field;
		mType = field.getType();
		mAnnotation = field.getAnnotation(Column.class);
		mName = mField.getName();
		mGet = getColumnGetMethod(clazz, mField);
		mSet = getColumnSetMethod(clazz, mField);
	}

	// ==========================================================================
	// Gett、Set方法
	// ==========================================================================
	public String getName() {
		return mName;
	}

	public Method getGet() {
		return mGet;
	}

	public Method getSet() {
		return mSet;
	}

	public Class<?> getType() {
		return mType;
	}

	public String getColumnDbType() {
		String result = mFieldTypesToDb.get(mType.getName());
		if (result == null) {
			result = "TEXT";
		}
		return result;
	}

	public static String getColumnDbType(Class<?> clazz) {
		String result = mFieldTypesToDb.get(clazz.getName());
		if (result == null) {
			result = "TEXT";
		}
		return result;
	}

	// ==========================================================================
	// 工具方法
	// ==========================================================================
	public Method getColumnGetMethod(Class<?> clazz, Field field) {
		String fieldName = field.getName();
		Method getMethod = null;
		String methodName;
		if (field.getType() == boolean.class) {
			methodName = "is" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
		} else {
			methodName = "get" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
		}
		try {
			getMethod = clazz.getDeclaredMethod(methodName);
		} catch (NoSuchMethodException e) {
			LogUtils.e(e);
		}
		if (getMethod == null && !Object.class.equals(clazz.getSuperclass())) {
			return getColumnGetMethod(clazz.getSuperclass(), field);
		}
		return getMethod;
	}

	public Method getColumnSetMethod(Class<?> clazz, Field field) {
		String fieldName = field.getName();
		Method setMethod = null;
		String methodName = "set" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
		try {
			setMethod = clazz.getDeclaredMethod(methodName, field.getType());
		} catch (NoSuchMethodException e) {
			LogUtils.e(e);
		}
		if (setMethod == null && !Object.class.equals(clazz.getSuperclass())) {
			return getColumnSetMethod(clazz.getSuperclass(), field);
		}
		return setMethod;
	}

	public void setValue(Object receiver, Object value) {
		if (mSet != null && value != null) {
			Object args;
			try {
				if (mType == String.class) {
					args = value.toString();
				} else if (mType == int.class || mType == Integer.class) {
					args = Integer.parseInt(value.toString());
				} else if (mType == float.class || mType == Float.class) {
					args = Float.parseFloat(value.toString());
				} else if (mType == double.class || mType == Double.class) {
					args = Double.parseDouble(value.toString());
				} else if (mType == long.class || mType == Long.class) {
					args = Long.parseLong(value.toString());
				} else if (mType == java.util.Date.class || mType == java.sql.Date.class) {
					args = DateFormattUtils.string2Date(value.toString());
				} else if (mType == boolean.class || mType == Boolean.class) {
					args = !"0".equals(value.toString());
				} else {
					args = value;
				}
				mSet.invoke(receiver, args);
			} catch (Exception e) {
				LogUtils.e(e);
			}
		} else {
			try {
				mField.setAccessible(true);
				mField.set(receiver, value);
			} catch (Exception e) {
				LogUtils.e(e);
			}
		}
	}

	public <T> T getValue(Object entity) {
		T value = null;
		if (entity != null) {
			if (mGet != null) {
				try {
					value = (T) mGet.invoke(entity);
				} catch (Exception e) {
					LogUtils.e(e);
				}
			} else {
				try {
					mField.setAccessible(true);
					value = (T) mField.get(entity);
				} catch (Exception e) {
					LogUtils.e(e);
				}
			}
		}
		return value;
	}

	// ==========================================================================
	// 条件判断
	// ==========================================================================
	public boolean isSupportType() {
		return mFieldTypesToDb.containsKey(mType.getName());
	}

	public boolean isId() {
		return mAnnotation != null && (mAnnotation.constraint() & ID) == ID;
	}

	public boolean isTransient() {
		return mAnnotation != null && (mAnnotation.constraint() & TRANSIENT) == TRANSIENT;
	}

	public boolean isForeign() {
		return mAnnotation != null && (mAnnotation.constraint() & FOREIGN) == FOREIGN;
	}

	public boolean isAutoIncrement() {
		return mAnnotation != null && (mAnnotation.constraint() & AUTOINCREMENT) == AUTOINCREMENT && (mType == int.class || mType == Integer.class || mType == long.class || mType == Long.class);
	}

	public boolean isUnique() {
		return mAnnotation != null && (mAnnotation.constraint() & UNIQUE) == UNIQUE;
	}

	public boolean isNotNull() {
		return mAnnotation != null && (mAnnotation.constraint() & NOTNULL) == NOTNULL;
	}

	public String getCheck() {
		if (mAnnotation != null && !StringUtils.isEmpty(mAnnotation.check())) {
			return mAnnotation.check();
		}
		return null;
	}

	public String getDefaultValue() {
		if (mAnnotation != null && !StringUtils.isEmpty(mAnnotation.defaultValue())) {
			return mAnnotation.defaultValue();
		}
		return null;
	}

	// ==========================================================================
	// 注解
	// ==========================================================================
	@Target(ElementType.FIELD)
	@Retention(RetentionPolicy.RUNTIME)
	public @interface Column {
		int constraint() default NONE; // 是否id，默认为0，则不是id。

		String defaultValue() default "";

		String check() default "";
	}
}
