package net.iyouqu.video.basecommon.dao;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.text.TextUtils;

import net.iyouqu.video.basecommon.dao.DbSql.KeyValue;
import net.iyouqu.video.basecommon.utils.LogUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class DbHelper {
	// ==========================================================================
	// 全局变量
	// ==========================================================================
	private SQLiteDatabase mDatabase;
	private DbConfig mConfig;
	private boolean mDebug = true;
	private boolean mTransaction = false;
	private Lock mWriteLock = new ReentrantLock();
	private volatile boolean mWriteLocked = false;
	private static HashMap<String, DbHelper> mHelperMap = new HashMap<String, DbHelper>();

	// ==========================================================================
	// 创建、获取Helper实例
	// ==========================================================================

	/** 根据配置类获取一个DbHelper实例 */
	public static DbHelper create(DbConfig daoConfig) {
		return getInstance(daoConfig);
	}

	/** 获取一个默认的DbHelper实例 */
	public static DbHelper create(Context context) {
		DbConfig config = new DbConfig(context);
		return getInstance(config);
	}

	private DbHelper(DbConfig config) {
		if (config == null) {
			throw new IllegalArgumentException("daoConfig may not be null");
		}
		if (config.getContext() == null) {
			throw new IllegalArgumentException("context mey not be null");
		}
		mConfig = config;
		mDatabase = createDatabase(config);
		mDatabase.execSQL("PRAGMA foreign_keys=ON;");
	}

	/** 根据数据库配置类来创建数据库实例 */
	private SQLiteDatabase createDatabase(DbConfig config) {
		SQLiteDatabase sqlite;
		String dbDir = config.getDbDir();
		if (!TextUtils.isEmpty(dbDir)) {
			File dir = new File(dbDir);
			if (!dir.exists()) {
				dir.mkdirs();
			}
			File dbFile = new File(dbDir, config.getDbName());
			sqlite = SQLiteDatabase.openOrCreateDatabase(dbFile, null);
		} else {
			sqlite = config.getContext().openOrCreateDatabase(config.getDbName(), 0, null);
		}
		return sqlite;
	}

	private synchronized static DbHelper getInstance(DbConfig daoConfig) {
		DbHelper helper = mHelperMap.get(daoConfig.getDbName());
		if (helper == null) {
			helper = new DbHelper(daoConfig);
			mHelperMap.put(daoConfig.getDbName(), helper);
		} else {
			helper.mConfig = daoConfig;
		}
		// 如果是从缓存中取出的helper，有可能新的配置config中提高了版本，需要进行升级操作
		SQLiteDatabase database = helper.mDatabase;
		int oldVersion = database.getVersion();
		int newVersion = daoConfig.getDbVersion();
		if (oldVersion != newVersion) {
			if (oldVersion != 0) {
				DbUpgradeListener upgradeListener = daoConfig.getDbUpgradeListener();
				if (upgradeListener != null) {// 如果有监听，交给监听做
					upgradeListener.onUpgrade(helper, oldVersion, newVersion);
				} else {// 没有，则删除所有数据
					try {
						helper.dropDb();
					} catch (Exception e) {
						LogUtils.e(e.getMessage(), e);
					}
				}
			}
			database.setVersion(newVersion);
		}
		return helper;
	}

	// ==========================================================================
	// get、set方法
	// ==========================================================================

	/** 是否打印SQL语句 */
	public DbHelper setDebug(boolean debug) {
		this.mDebug = debug;
		return this;
	}

	/** 是否开启事物 */
	public DbHelper setTransaction(boolean transaction) {
		this.mTransaction = transaction;
		return this;
	}

	/** 获取数据库实例 */
	public SQLiteDatabase getDatabase() {
		return mDatabase;
	}

	/** 获取配置类实例 */
	public DbConfig getDbConfig() {
		return mConfig;
	}

	// ==========================================================================
	// Methods
	// ==========================================================================
	// ==========================================================================
	// 数据库操作方法
	// ==========================================================================

	/** 保存实例对象到数据库 */
	public void save(Object entity) {
		if (entity == null) {
			return;
		}
		try {
			beginTransaction();
			createTableIfNotExist(entity.getClass());
			save(DbTable.getForeign(entity));
			execNonQuery(DbSql.buildInsertSql(entity));
			setTransactionSuccessful();
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			endTransaction();
		}
	}

	/** 保存实例对象到数据库 */
	public void save(List<?> entities) {
		if (entities == null || entities.size() == 0) {
			return;
		}
		try {
			beginTransaction();
			createTableIfNotExist(entities.get(0).getClass());
			for (Object entity : entities) {
				execNonQuery(DbSql.buildInsertSql(entity));
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 保存实例对象到数据库，并绑定id */
	public boolean saveBindId(Object entity) {
		boolean result = false;
		try {
			beginTransaction();
			createTableIfNotExist(entity.getClass());
			result = saveBindIdWithoutTransaction(entity);
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
		return result;
	}

	/** 保存实例对象到数据库，并绑定id */
	public boolean saveBindId(List<?> entities) {
		boolean result = true;
		if (entities == null || entities.size() == 0) return !result;
		try {
			beginTransaction();
			createTableIfNotExist(entities.get(0).getClass());
			for (Object entity : entities) {
				result &= saveBindIdWithoutTransaction(entity);
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
		return result;
	}

	/** 根据实体对象中唯一属性来判断，如果数据库中存在，则修改唯一属性外的其他属性，如果不存在，则保存实体对象 */
	public void saveOrUpdate(Object entity) {
		try {
			beginTransaction();
			createTableIfNotExist(entity.getClass());
			saveOrUpdateWithoutTransaction(entity);
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象中唯一属性来判断，如果数据库中存在，则修改唯一属性外的其他属性，如果不存在，则保存实体对象 */
	public void saveOrUpdate(List<?> entities) {
		if (entities == null || entities.size() == 0) return;
		try {
			beginTransaction();
			createTableIfNotExist(entities.get(0).getClass());
			for (Object entity : entities) {
				saveOrUpdateWithoutTransaction(entity);
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	private void saveOrUpdateWithoutTransaction(Object entity) {
		Class<?> clazz = entity.getClass();
		DbTable table = DbTable.getTable(clazz);
		DbColumn info = table.getId();
		if (info.isAutoIncrement()) {
			if (info.getValue(entity) != null) {
				execNonQuery(DbSql.buildUpdateSql(entity, null));
			} else {
				saveBindIdWithoutTransaction(entity);
			}
		} else {
			execNonQuery(DbSql.buildReplaceSql(entity));
		}
	}

	private boolean saveBindIdWithoutTransaction(Object entity) {
		Class<?> clazz = entity.getClass();
		DbTable table = DbTable.getTable(clazz);
		String tableName = table.getName();
		DbColumn info = table.getId();
		if (info.isAutoIncrement()) {
			List<KeyValue> entityKvList = DbSql.entity2KeyValueList(entity);
			if (entityKvList != null && entityKvList.size() > 0) {
				ContentValues cv = new ContentValues();
				DbHelper.fillContentValues(cv, entityKvList);
				long id = mDatabase.insert(tableName, null, cv);
				if (id == -1) {
					return false;
				}
				info.setValue(entity, id);
				return true;
			}
		} else {
			execNonQuery(DbSql.buildInsertSql(entity));
			return true;
		}
		return false;
	}

	private static void fillContentValues(ContentValues contentValues, List<KeyValue> list) {
		if (list != null && contentValues != null) {
			for (KeyValue kv : list) {
				Object value = kv.getValue();
				if (value != null) {
					contentValues.put(kv.getKey(), value.toString());
				}
			}
		} else {
			LogUtils.w("List<KeyValue> is empty or ContentValues is empty!");
		}
	}

	/** 根据实体对象替换数据库中的数据 */
	public void replace(Object entity) {
		try {
			beginTransaction();
			createTableIfNotExist(entity.getClass());
			execNonQuery(DbSql.buildReplaceSql(entity));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象替换数据库中的数据 */
	public void replace(List<?> entities) {
		if (entities == null || entities.size() == 0) return;
		try {
			beginTransaction();
			createTableIfNotExist(entities.get(0).getClass());
			for (Object entity : entities) {
				execNonQuery(DbSql.buildReplaceSql(entity));
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象更新数据库中指定的字段 */
	public void update(Object entity, String[] updateColumnNames) {
		if (!tableIsExist(entity.getClass())) return;
		try {
			beginTransaction();
			execNonQuery(DbSql.buildUpdateSql(entity, updateColumnNames));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象更新数据库中指定的字段 */
	public void update(List<?> entities, String[] updateColumnNames) {
		if (entities == null || entities.size() == 0 || !tableIsExist(entities.get(0).getClass()))
			return;
		try {
			beginTransaction();
			for (Object entity : entities) {
				execNonQuery(DbSql.buildUpdateSql(entity, updateColumnNames));
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象和条件更新数据库中指定的字段 */
	public void update(Object entity, String where, String[] updateColumnNames) {
		if (!tableIsExist(entity.getClass())) return;
		try {
			beginTransaction();
			execNonQuery(DbSql.buildUpdateSql(entity, where, updateColumnNames));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象和条件更新数据库中指定的字段 */
	public void update(List<?> entities, String where, String[] updateColumnNames) {
		if (entities == null || entities.size() == 0 || !tableIsExist(entities.get(0).getClass()))
			return;
		try {
			beginTransaction();

			for (Object entity : entities) {
				execNonQuery(DbSql.buildUpdateSql(entity, where, updateColumnNames));
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据id删除数据库中的数据 */
	public void deleteById(Class<?> entityType, Object idValue) {
		if (!tableIsExist(entityType)) return;
		try {
			beginTransaction();
			execNonQuery(DbSql.buildDeleteSql(entityType, idValue));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象删除数据库中的数据 */
	public void delete(Object entity) {
		if (!tableIsExist(entity.getClass())) return;
		try {
			beginTransaction();
			execNonQuery(DbSql.buildDeleteSql(entity));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体对象集合删除数据库中的数据 */
	public void delete(List<?> entities) {
		if (entities == null || entities.size() == 0 || !tableIsExist(entities.get(0).getClass()))
			return;
		try {
			beginTransaction();
			for (Object entity : entities) {
				execNonQuery(DbSql.buildDeleteSql(entity));
			}
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据指定的条件删除数据库中的数据 */
	public void delete(Class<?> entityType, String where) {
		if (!tableIsExist(entityType)) return;
		try {
			beginTransaction();
			execNonQuery(DbSql.buildDeleteSql(entityType, where));
			setTransactionSuccessful();
		} finally {
			endTransaction();
		}
	}

	/** 根据实体类型删除数据库中所有的数据 */
	public void deleteAll(Class<?> entityType) {
		delete(entityType, "");
	}

	/** 根据Id从数据库中查找数据 */
	public <T> T findById(Class<T> entityType, Object idValue) {
		if (!tableIsExist(entityType)) {
			return null;
		}
		return findFirstBySql(entityType, DbSql.buildSelectSql(entityType, idValue).getSql());
	}

	/** 根据实体类型从数据库中查找数据 */
	public <T> List<T> findAll(Class<T> clazz) {
		if (!tableIsExist(clazz)) {
			return null;
		}
		return findAllBySql(clazz, DbSql.buildSelectSql(clazz).getSql());
	}

	/** 根据Id从数据库中查找数据 */
	public <T> T findFirst(Class<T> clazz) {
		if (!tableIsExist(clazz)) {
			return null;
		}
		return findFirstBySql(clazz, DbSql.buildSelectSql(clazz).getSql());
	}

	/**
	 * 根据条件查找所有数据
	 * @param clazz 需要查找的类型
	 * @param where 条件为空的时候查找所有数据
	 */
	public <T> List<T> findAllByWhere(Class<T> clazz, String where) {
		if (!tableIsExist(clazz)) {
			return null;
		}
		return findAllBySql(clazz, DbSql.buildSelectSql(clazz, where, null, null, 0, 0).getSql());
	}

	/** 根据Id从数据库中查找数据 where id 有问题*/
	@Deprecated
	public <T> T findFirst(Class<T> clazz, String where) {
		if (!tableIsExist(clazz)) {
			return null;
		}
		return findFirstBySql(clazz, DbSql.buildSelectSql(clazz, where, null, null, 0, 0).getSql());
	}

	/**
	 * 查找所有数据
	 * @param clazz   需要查找的类型
	 * @param orderBy 排序的字段
	 */
	public <T> List<T> findAll(Class<T> clazz, String where, String orderBy, String groupBy, int limit, int offset) {
		if (!tableIsExist(clazz)) {
			return null;
		}
		return findAllBySql(clazz, DbSql.buildSelectSql2(clazz, where, orderBy, groupBy, limit, offset).getSql());
	}

	/**
	 * 根据条件查找所有数据
	 */
	private <T> List<T> findAllBySql(Class<T> clazz, String sql) {
		debugSql(sql);
		Cursor cursor = mDatabase.rawQuery(sql, null);
		List<T> list = new ArrayList<T>();
		try {
			while (cursor.moveToNext()) {
				T entity = getEntity(cursor, clazz);
				list.add(entity);
			}
		} finally {
			closeCursor(cursor);
		}
		return list;
	}

	/**
	 * 根据条件查找所有数据
	 */
	private <T> T findFirstBySql(Class<T> clazz, String sql) {
		debugSql(sql);
		Cursor cursor = mDatabase.rawQuery(sql, null);
		try {
			if (cursor.moveToNext()) {
				return getEntity(cursor, clazz);
			}
		} finally {
			closeCursor(cursor);
		}
		return null;
	}

	public <T> T getEntity(final Cursor cursor, Class<T> clazz) {
		if (cursor == null) {
			return null;
		}
		T entity = null;
		try {
			int columnCount = cursor.getColumnCount();
			if (columnCount > 0) {
				DbTable table = DbTable.getTable(clazz);
				DbColumn id = table.getId();
				entity = clazz.newInstance();
				Map<String, DbColumn> columnMap = table.getColumnMap();
				Map<String, DbColumn> foreignMap = table.getForeignMap();
				for (int i = 0; i < columnCount; i++) {
					String column = cursor.getColumnName(i);
					DbColumn dbColumn = columnMap.get(column);
					DbColumn dbForeign = foreignMap.get(column);
					if (dbColumn != null) {
						dbColumn.setValue(entity, cursor.getString(i));
					} else if (id.getName().equals(column)) {
						id.setValue(entity, cursor.getString(i));
					} else if (dbForeign != null) {
						Object object = findById(dbForeign.getType(), cursor.getInt(i));
						dbForeign.setValue(entity, object);
					}
				}
			}
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return entity;
	}

	/** 根据实体类型查找数据库中的所有符合的数据个数 */
	public long count(Class<?> clazz) {
		return count(clazz, null);
	}

	/** 根据实体类型和约束条件查找数据库中的所有符合的数据个数 */
	public long count(Class<?> clazz, String where) {
		if (!tableIsExist(clazz)) {
			return 0;
		}
		Cursor cursor = execQuery(DbSql.buildCountSql(clazz, where));
		try {
			if (cursor.moveToNext()) {
				return cursor.getLong(0);
			}
		} finally {
			closeCursor(cursor);
		}
		return 0;
	}

	// ==========================================================================
	// 执行SQL语句
	// ==========================================================================
	public void execNonQuery(DbSql sql) {
		debugSql(sql.getSql());
		try {
			if (sql.getBindArgs() != null) {
				mDatabase.execSQL(sql.getSql(), sql.getBindArgsAsArray());
			} else {
				mDatabase.execSQL(sql.getSql());
			}
		} catch (Exception e) {
			LogUtils.e(e);
		}
	}

	public void execNonQuery(String sql) {
		debugSql(sql);
		try {
			mDatabase.execSQL(sql);
		} catch (Throwable e) {
			LogUtils.e(e);
		}
	}

	public Cursor execQuery(DbSql sql) {
		debugSql(sql.getSql());
		Cursor result = null;
		try {
			return mDatabase.rawQuery(sql.getSql(), sql.getBindArgsAsStrArray());
		} catch (Throwable e) {
			LogUtils.e(e);
		}
		return result;
	}

	public Cursor execQuery(String sql) {
		debugSql(sql);
		Cursor result = null;
		try {
			return mDatabase.rawQuery(sql, null);
		} catch (Throwable e) {
			LogUtils.e(e);
		}
		return result;
	}

	// ==========================================================================
	// 其他方法
	// ==========================================================================

	/** 打印SQL语句 */
	private void debugSql(String sql) {
		if (mDebug) {
			LogUtils.d(sql);
		}
	}

	private void closeCursor(Cursor cursor) {
		if (cursor != null) {
			try {
				cursor.close();
			} catch (Throwable e) {
				LogUtils.e(e);
			}
		}
	}

	/** 开启事物 */
	private void beginTransaction() {
		if (mTransaction) {
			mDatabase.beginTransaction();
		} else {// 如果不用开始事物，则开启写锁，防止多次写操作造成脏读数据
			mWriteLock.lock();
			mWriteLocked = true;
		}
	}

	/** 设置事物成功 */
	private void setTransactionSuccessful() {
		if (mTransaction) {
			mDatabase.setTransactionSuccessful();
		}
	}

	/** 结束事物 */
	private void endTransaction() {
		if (mTransaction) {
			mDatabase.endTransaction();
		}
		if (mWriteLocked) {
			mWriteLocked = false;
			mWriteLock.unlock();
		}
	}

	/** 判断表是否存在，不存在就创建 */
	public void createTableIfNotExist(Class<?> entityType) {
		if (!tableIsExist(entityType)) {
			DbSql sql = DbSql.buildCreateTableSql(entityType);
			execNonQuery(sql);
		}
	}

	/** 判断表是否存在 */
	public boolean tableIsExist(Class<?> entityType) {
		DbTable table = DbTable.getTable(entityType);
		if (table.isExist()) {
			return true;
		}

		Cursor cursor = null;
		try {
			cursor = execQuery("SELECT COUNT(*) AS c FROM sqlite_master WHERE type ='table' AND name ='" + table.getName() + "'");
			if (cursor != null && cursor.moveToNext()) {
				int count = cursor.getInt(0);
				if (count > 0) {
					table.setExist(true);
					return true;
				}
			}
		} finally {
			closeCursor(cursor);
		}
		return false;
	}

	/** 删除数据库中的表 */
	public void dropDb() {
		Cursor cursor = null;
		try {
			cursor = execQuery("SELECT name FROM sqlite_master WHERE type ='table'");
			if (cursor != null) {
				while (cursor.moveToNext()) {
					try {
						String tableName = cursor.getString(0);
						execNonQuery("DROP TABLE " + tableName);
						DbTable.remove(tableName);
					} catch (Throwable e) {
						LogUtils.e(e.getMessage(), e);
					}
				}
			}
		} finally {
			closeCursor(cursor);
		}
	}

	/** 根据实体类型删除对应的表 */
	public void dropTable(Class<?> entityType) {
		if (!tableIsExist(entityType)) return;
		String tableName = DbTable.getTableName(entityType);
		execNonQuery("DROP TABLE " + tableName);
		DbTable.remove(entityType);
	}

	// ==========================================================================
	// 内部类、内部接口
	// ==========================================================================
	public static class DbConfig {
		private Context context;
		private String dbName = "iyouquvideo.db"; // 默认的数据库名
		private int dbVersion = 1;
		private DbUpgradeListener dbUpgradeListener;

		private String dbDir;

		public DbConfig(Context context) {
			this.context = context;
		}

		public Context getContext() {
			return context;
		}

		public String getDbName() {
			return dbName;
		}

		public void setDbName(String dbName) {
			if (!TextUtils.isEmpty(dbName)) {
				this.dbName = dbName;
			}
		}

		public int getDbVersion() {
			return dbVersion;
		}

		public void setDbVersion(int dbVersion) {
			this.dbVersion = dbVersion;
		}

		public DbUpgradeListener getDbUpgradeListener() {
			return dbUpgradeListener;
		}

		public void setDbUpgradeListener(DbUpgradeListener dbUpgradeListener) {
			this.dbUpgradeListener = dbUpgradeListener;
		}

		public String getDbDir() {
			return dbDir;
		}

		/** 设置数据库的保存路径 */
		public void setDbDir(String dbDir) {
			this.dbDir = dbDir;
		}
	}

	/** 数据库更新时的监听 */
	public interface DbUpgradeListener {
		public void onUpgrade(DbHelper db, int oldVersion, int newVersion);
	}
}
