package net.iyouqu.video.basecommon.dao;
import net.iyouqu.video.basecommon.utils.LogUtils;
import net.iyouqu.video.basecommon.utils.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

/**
 * File Name: DbSql.java
 * Create by:Bruce on 2014-12-12 下午5:51:39 
 * @version 1.0.0
 */
public class DbSql {

	private String sql;
	private LinkedList<Object> bindArgs;

	public DbSql() {
	}

	public DbSql(String sql) {
		this.sql = sql;
	}

	public DbSql(String sql, Object... bindArgs) {
		this.sql = sql;
		addBindArgs(bindArgs);
	}

	public String getSql() {
		return sql;
	}

	public void setSql(String sql) {
		this.sql = sql;
	}

	public LinkedList<Object> getBindArgs() {
		return bindArgs;
	}

	public Object[] getBindArgsAsArray() {
		if (bindArgs != null) {
			return bindArgs.toArray();
		}
		return null;
	}

	public String[] getBindArgsAsStrArray() {
		if (bindArgs != null) {
			String[] strings = new String[bindArgs.size()];
			for (int i = 0; i < bindArgs.size(); i++) {
				Object value = bindArgs.get(i);
				strings[i] = value == null ? null : value.toString();
			}
			return strings;
		}
		return null;
	}

	public void addBindArg(Object arg) {
		if (bindArgs == null) {
			bindArgs = new LinkedList<Object>();
		}
		bindArgs.add(arg);
	}

	public void addBindArgWithoutConverter(Object arg) {
		if (bindArgs == null) {
			bindArgs = new LinkedList<Object>();
		}

		bindArgs.add(arg);
	}

	public void addBindArgs(Object... bindArgs) {
		if (bindArgs != null) {
			for (Object arg : bindArgs) {
				addBindArg(arg);
			}
		}
	}

	// ==========================================================================
	// 创建insert sql
	// ==========================================================================
	public static DbSql buildInsertSql(Object entity) {
		List<KeyValue> keyValueList = entity2KeyValueList(entity);
		if (keyValueList.size() == 0) {
			return null;
		}
		DbSql result = new DbSql();
		StringBuilder sqlBuilder = new StringBuilder();
		sqlBuilder.append("INSERT INTO ");
		sqlBuilder.append(DbTable.getTableName(entity.getClass()));
		sqlBuilder.append(" (");
		for (KeyValue kv : keyValueList) {
			sqlBuilder.append(kv.getKey()).append(",");
			result.addBindArgWithoutConverter(kv.getValue());
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		sqlBuilder.append(") VALUES (");
		int length = keyValueList.size();
		for (KeyValue aKeyValue : keyValueList) {
			sqlBuilder.append("?,");
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		sqlBuilder.append(")");
		result.setSql(sqlBuilder.toString());
		return result;
	}

	// ==========================================================================
	// 创建replace sql
	// ==========================================================================
	public static DbSql buildReplaceSql(Object entity) {
		List<KeyValue> keyValueList = entity2KeyValueList(entity);
		if (keyValueList.size() == 0) {
			return null;
		}
		DbSql result = new DbSql();
		StringBuilder sqlBuilder = new StringBuilder();
		sqlBuilder.append("REPLACE INTO ");
		sqlBuilder.append(DbTable.getTableName(entity.getClass()));
		sqlBuilder.append(" (");
		for (KeyValue kv : keyValueList) {
			sqlBuilder.append(kv.getKey()).append(",");
			result.addBindArgWithoutConverter(kv.getValue());
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		sqlBuilder.append(") VALUES (");
		int length = keyValueList.size();
		for (KeyValue aKeyValue : keyValueList) {
			sqlBuilder.append("?,");
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		sqlBuilder.append(")");
		result.setSql(sqlBuilder.toString());
		return result;
	}

	// ==========================================================================
	// 创建delete sql
	// ==========================================================================
	public static DbSql buildDeleteSql(Object entity) {
		Class<?> clazz = entity.getClass();
		DbColumn id = DbTable.getTable(clazz).getId();
		Object idValue = id.getValue(entity);
		return buildDeleteSql(clazz, idValue);
	}

	public static DbSql buildDeleteSql(Class<?> clazz, Object idValue) {
		DbSql result = new DbSql();
		if (null == idValue) {
			LogUtils.e("创建删除语句失败，id不能为null");
		} else {
			DbTable table = DbTable.getTable(clazz);
			String tableName = table.getName();
			DbColumn id = table.getId();
			StringBuilder sqlBuilder = new StringBuilder("DELETE FROM ");
			sqlBuilder.append(tableName);
			if(idValue instanceof String){
				sqlBuilder.append(" WHERE ").append(id.getName()).append("='").append(idValue).append("'");
			}else {
				sqlBuilder.append(" WHERE ").append(id.getName()).append("=").append(idValue);
			}
			result.setSql(sqlBuilder.toString());
		}
		return result;
	}

	public static DbSql buildDeleteSql(Class<?> clazz, String where) {
		String tableName = DbTable.getTableName(clazz);
		StringBuilder sqlBuilder = new StringBuilder("DELETE FROM ").append(tableName);
		if (!StringUtils.isEmpty(where)) {
			sqlBuilder.append(" WHERE ").append(where);
		}
		return new DbSql(sqlBuilder.toString());
	}

	// ==========================================================================
	// 创建update sql
	// ==========================================================================
	public static DbSql buildUpdateSql(Object entity, String[] updateColumnNames) {
		List<KeyValue> keyValueList = entity2KeyValueList(entity);
		if (keyValueList.size() == 0) {
			return null;
		}
		HashSet<String> updateColumnNameSet = null;
		if (updateColumnNames != null && updateColumnNames.length > 0) {
			updateColumnNameSet = new HashSet<String>(updateColumnNames.length);
			Collections.addAll(updateColumnNameSet, updateColumnNames);
		}
		Class<?> clazz = entity.getClass();
		String tableName = DbTable.getTableName(clazz);
		DbColumn id = DbTable.getTable(clazz).getId();
		Object idValue = id.getValue(entity);
		DbSql result = new DbSql();
		if (idValue == null) {
			LogUtils.e("创建删除语句失败，id不能为null");
		} else {
			StringBuilder sqlBuilder = new StringBuilder("UPDATE ");
			sqlBuilder.append(tableName);
			sqlBuilder.append(" SET ");
			for (KeyValue kv : keyValueList) {
				if (updateColumnNameSet == null || updateColumnNameSet.contains(kv.getKey())) {
					sqlBuilder.append(kv.getKey()).append("=?,");
					result.addBindArgWithoutConverter(kv.getValue());
				}
			}
			sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
			if(idValue instanceof String){
				sqlBuilder.append(" WHERE ").append(id.getName()).append("='").append(idValue).append("'");
			}else {
				sqlBuilder.append(" WHERE ").append(id.getName()).append("=").append(idValue);
			}
			result.setSql(sqlBuilder.toString());
		}
		return result;
	}

	public static DbSql buildUpdateSql(Object entity, String where, String[] updateColumnNames) {
		List<KeyValue> keyValueList = entity2KeyValueList(entity);
		if (keyValueList.size() == 0) {
			return null;
		}
		HashSet<String> updateColumnNameSet = null;
		if (updateColumnNames != null && updateColumnNames.length > 0) {
			updateColumnNameSet = new HashSet<String>(updateColumnNames.length);
			Collections.addAll(updateColumnNameSet, updateColumnNames);
		}
		Class<?> clazz = entity.getClass();
		DbTable table = DbTable.getTable(clazz);
		String tableName = table.getName();
		DbSql result = new DbSql();
		StringBuilder sqlBuilder = new StringBuilder("UPDATE ");
		sqlBuilder.append(tableName);
		sqlBuilder.append(" SET ");
		for (KeyValue kv : keyValueList) {
			if (updateColumnNameSet == null || updateColumnNameSet.contains(kv.getKey())) {
				sqlBuilder.append(kv.getKey()).append("=?,");
				result.addBindArgWithoutConverter(kv.getValue());
			}
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		if (!StringUtils.isEmpty(where)) {
			sqlBuilder.append(" WHERE ").append(where);
		}
		result.setSql(sqlBuilder.toString());
		return result;
	}

	// ==========================================================================
	// 查找
	// ==========================================================================
	public static DbSql buildSelectSql(Class<?> clazz) {
		DbSql result = new DbSql();
		DbTable table = DbTable.getTable(clazz);
		String name = table.getName();
		StringBuilder sqlBuilder = new StringBuilder("SELECT * FROM ").append(name);
		result.setSql(sqlBuilder.toString());
		return result;
	}

	public static DbSql buildSelectSql(Class<?> clazz, Object idValue) {
		DbSql result = new DbSql();
		if (null == idValue) {
			LogUtils.e("创建查找语句失败，id不能为null");
		} else {
			DbTable table = DbTable.getTable(clazz);
			String name = table.getName();
			DbColumn id = table.getId();
			StringBuilder sqlBuilder = new StringBuilder("SELECT * FROM ").append(name);
			if(idValue instanceof String){
				sqlBuilder.append(" WHERE ").append(id.getName()).append("='").append(idValue).append("'");
			}else{
				sqlBuilder.append(" WHERE ").append(id.getName()).append("=").append(idValue);
			}
			result.setSql(sqlBuilder.toString());
		}
		return result;
	}

    public static DbSql buildSelectSql2(Class<?> clazz, String where, String orderBy, String groupBy, int limit, int offset) {
        DbSql result = new DbSql();
        String name = DbTable.getTable(clazz).getName();
        StringBuilder sqlBuilder = new StringBuilder("SELECT * FROM ").append(name);
        if (!StringUtils.isEmpty(where)) {
            sqlBuilder.append(" WHERE ").append(where);
        }
        if (!StringUtils.isEmpty(orderBy)) {
            sqlBuilder.append(" ORDER BY ").append(orderBy);
        }
        if (!StringUtils.isEmpty(groupBy)) {
            sqlBuilder.append(" GROUP BY ").append(groupBy);
        }
        if (limit > 0) {
            sqlBuilder.append(" LIMIT ").append(limit);
            sqlBuilder.append(" OFFSET ").append(offset);
        }
        result.setSql(sqlBuilder.toString());
        return result;
    }

	public static DbSql buildSelectSql(Class<?> clazz, String where, String orderBy, String groupBy, int limit, int offset) {
		DbSql result = new DbSql();
		String name = DbTable.getTable(clazz).getName();
		StringBuilder sqlBuilder = new StringBuilder("SELECT * FROM ").append(name);
		if (!StringUtils.isEmpty(where)) {
			if(where instanceof String){
				sqlBuilder.append(" WHERE ").append("'").append(where).append("'");
			}else {
				sqlBuilder.append(" WHERE ").append(where);
			}
		}
		if (!StringUtils.isEmpty(orderBy)) {
			sqlBuilder.append(" ORDER BY ").append(orderBy);
		}
		if (!StringUtils.isEmpty(groupBy)) {
			sqlBuilder.append(" GROUP BY ").append(groupBy);
		}
		if (limit > 0) {
			sqlBuilder.append(" LIMIT ").append(limit);
			sqlBuilder.append(" OFFSET ").append(offset);
		}
		result.setSql(sqlBuilder.toString());
		return result;
	}

	public static DbSql buildCountSql(Class<?> clazz, String where) {
		DbSql result = new DbSql();
		String name = DbTable.getTable(clazz).getName();
		StringBuilder sqlBuilder = new StringBuilder("SELECT COUNT(*) FROM ").append(name);
		if (!StringUtils.isEmpty(where)) {
			sqlBuilder.append(" WHERE ").append(where);
		}
		result.setSql(sqlBuilder.toString());
		return result;
	}

	// ==========================================================================
	// 其他方法
	// ==========================================================================
	public static DbSql buildCreateTableSql(Class<?> clazz) {
		DbTable table = DbTable.getTable(clazz);
		String tableName = table.getName();
		DbColumn id = table.getId();

		StringBuilder sqlBuilder = new StringBuilder();
		sqlBuilder.append("CREATE TABLE IF NOT EXISTS ");
		sqlBuilder.append(tableName);
		sqlBuilder.append(" ( ");

		if (id.isAutoIncrement()) {
			sqlBuilder.append("\"").append(id.getName()).append("\"  ").append("INTEGER PRIMARY KEY AUTOINCREMENT,");
		} else {
			sqlBuilder.append("\"").append(id.getName()).append("\"  ").append(id.getColumnDbType()).append(" PRIMARY KEY,");
		}
		Collection<DbColumn> columns = table.getColumnMap().values();
		for (DbColumn info : columns) {
			sqlBuilder.append("\"").append(info.getName()).append("\"  ");
			sqlBuilder.append(info.getColumnDbType());
			if (info.isUnique()) {
				sqlBuilder.append(" UNIQUE");
			}
			if (info.isNotNull()) {
				sqlBuilder.append(" NOT NULL");
			}
			String check = info.getCheck();
			if (check != null) {
				sqlBuilder.append(" CHECK(").append(check).append(")");
			}
			sqlBuilder.append(",");
		}
		Collection<DbColumn> foreigns = table.getForeignMap().values();
		for (DbColumn info : foreigns) {
			sqlBuilder.append(info.getName()).append(" INTEGER").append(",");
		}
		sqlBuilder.deleteCharAt(sqlBuilder.length() - 1);
		sqlBuilder.append(" )");
		return new DbSql(sqlBuilder.toString());
	}

	/** 把实体对象转换成键值对 */
	private static KeyValue column2KeyValue(Object entity, DbColumn ifno) {
		KeyValue kv = null;
		String key = ifno.getName();
		Object value = null;
		if (ifno.isForeign()) {
			try {
				DbTable table = DbTable.getTable(ifno.getType());
				Object obj = ifno.getGet().invoke(entity);
				value = table.getId().getGet().invoke(obj);
			} catch (Exception e) {
				LogUtils.e(e);
			}
		} else {
			value = ifno.getValue(entity);
			value = value == null ? ifno.getDefaultValue() : value;
		}
		if (key != null) {
			kv = new KeyValue(key, value);
		}
		return kv;
	}

	/** 把实体对象转换成键值对 */
	public static List<KeyValue> entity2KeyValueList(Object entity) {
		List<KeyValue> keyValueList = new ArrayList<KeyValue>();
		Class<?> clazz = entity.getClass();
		DbTable table = DbTable.getTable(clazz);
		DbColumn id = table.getId();
		if (!id.isAutoIncrement()) {
			Object idValue = id.getValue(entity);
			KeyValue kv = new KeyValue(id.getName(), idValue);
			keyValueList.add(kv);
		}
		Collection<DbColumn> columns = table.getColumnMap().values();
		for (DbColumn info : columns) {
			KeyValue kv = column2KeyValue(entity, info);
			if (kv != null) {
				keyValueList.add(kv);
			}
		}
		Collection<DbColumn> foreigns = table.getForeignMap().values();
		for (DbColumn info : foreigns) {
			KeyValue kv = column2KeyValue(entity, info);
			if (kv != null) {
				keyValueList.add(kv);
			}
		}
		return keyValueList;
	}

	public static class KeyValue {
		private String key;
		private Object value;

		public KeyValue(String key, Object value) {
			this.key = key;
			this.value = value;
		}

		public String getKey() {
			return key;
		}

		public Object getValue() {
			return value;
		}
	}
}
