package net.iyouqu.video.basecommon.dao;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import net.iyouqu.video.basecommon.utils.LogUtils;

/**
 * File Name: DbTable.java
 * Create by:Bruce on 2014-12-12 下午5:52:11 
 * @version 1.0.0
 */
public class DbTable {
	// ==========================================================================
	// Constants
	// ==========================================================================

	// ==========================================================================
	// Fields
	// ==========================================================================
	private String mName;
	private DbColumn mId;
	private boolean mExist;
	private final HashMap<String, DbColumn> mColumnMap = new HashMap<String, DbColumn>();
	private final HashMap<String, DbColumn> mForeignMap = new HashMap<String, DbColumn>();

	private static ConcurrentHashMap<String, DbTable> mTableMap = new ConcurrentHashMap<String, DbTable>();

	// ==========================================================================
	// Constructors
	// ==========================================================================
	public static synchronized DbTable getTable(Class<?> clazz) {
		if (clazz == null) {
			LogUtils.e("构建Table时，实体类型不能为null");
			return null;
		}
		String name = getTableName(clazz);
		DbTable table = mTableMap.get(name);
		if (table == null) {
			table = new DbTable(clazz);
			table.setName(name);
			mTableMap.put(name, table);
		}
		return table;
	}

	private DbTable(Class<?> clazz) {
		init(clazz);
	}

	// ==========================================================================
	// Getters And Setters
	// ==========================================================================

	public boolean isExist() {
		return mExist;
	}

	public void setExist(boolean exist) {
		mExist = exist;
	}

	public String getName() {
		return mName;
	}

	private void setName(String name) {
		mName = name;
	}

	public DbColumn getId() {
		return mId;
	}

	public Map<String, DbColumn> getColumnMap() {
		return mColumnMap;
	}

	public Map<String, DbColumn> getForeignMap() {
		return mForeignMap;
	}

	public boolean hasForeign() {
		return mForeignMap != null && mForeignMap.size() > 0;
	}

	// ==========================================================================
	// Methods
	// ==========================================================================
	public void init(Class<?> clazz) {
		Field[] fields = clazz.getDeclaredFields();
		for (Field field : fields) {
			if (Modifier.isStatic(field.getModifiers())) {// 静态字段的不用保存，创建对象时会自动创建
				continue;
			}
			DbColumn info = new DbColumn(clazz, field);
			String name = info.getName();
			if (info.isTransient()) {// 如果标记了不需要保存，则跳过
				continue;
			}
			if (info.isSupportType()) {
				if (info.isId()) {
					mId = info;
				} else {
					mColumnMap.put(name, info);
				}
			} else if (info.isForeign()) {
				mForeignMap.put(name, info);
			}
		}
		if (!Object.class.equals(clazz.getSuperclass())) {
			init(clazz.getSuperclass());
		}
	}

	// ==========================================================================
	// Implementation
	// ==========================================================================
	public static String getTableName(Class<?> clazz) {
		String name = clazz.getName();
		if (name != null) {
			return name.replace('.', '_');
		}
		return null;
	}

	public static void remove(Class<?> clazz) {
		String name = getTable(clazz).getName();
		mTableMap.remove(name);
	}

	public static void remove(String tableName) {
		mTableMap.remove(tableName);
	}

	public static List<Object> getForeign(Object entity) {
		Class<?> clazz = entity.getClass();
		DbTable table = DbTable.getTable(clazz);
		Collection<DbColumn> values = table.getForeignMap().values();
		List<Object> list = new ArrayList<Object>();
		try {
			for (DbColumn column : values) {
				Object invoke = column.getGet().invoke(entity);
				list.add(invoke);
			}
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return list;
	}

	// ==========================================================================
	// Inner/Nested Classes
	// ==========================================================================
}
