package net.iyouqu.video.basecommon.manager;

import java.lang.reflect.Field;
import java.net.InetAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import net.iyouqu.video.basecommon.utils.LogUtils;
import net.iyouqu.video.basecommon.utils.ui.UIUtils;

import org.apache.http.HttpEntity;
import org.apache.http.conn.BasicManagedEntity;
import org.apache.http.impl.SocketHttpClientConnection;
import org.apache.http.impl.conn.AbstractClientConnAdapter;


import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.database.Cursor;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Proxy;
import android.net.Uri;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;

/**
 * File Name: NetManager.java
 * Create by:Bruce on 2014-12-12 下午4:43:38 
 * @version 1.0.0
 */
public class NetManager {
	//==========================================================================
		// 静态常量
		//==========================================================================
		/** 网络状态类型: */
		public static final int NET_TPYE_NONE = 0;// 不可用
		public static final int NET_TPYE_WIFI = 1;// wifi
		public static final int NET_TPYE_NET = 2;// 3G
		public static final int NET_TPYE_CMWAP = 3;// 移动WAP
		public static final int NET_TPYE_CTWAP = 4;// 电信WAP
		public static final int NET_TPYE_UNIWAP = 5;// 联通WAP
		public static final int NET_TPYE_OTHER = 6;// 其他

		/** 移动，联通WAP代理IP */
		private final static String PROXY_CM_UNI = "10.0.0.172";
		/** 电信WAP代理IP */
		private final static String PROXY_CT = "10.0.0.200";

		/** 移动MNC编号 00 02 07 */
		public final static String CM_MNC_1 = "00";
		public final static String CM_MNC_2 = "02";
		public final static String CM_MNC_7 = "07";
		/** 电信MNC编号 03 */
		public final static String CT_MNC = "03";
		/** 联通MNC编号 01 */
		public final static String UNI_MNC = "01";
		//==========================================================================
		// 全局变量
		//==========================================================================
		private static NetManager sInstance;
		private final List<NetSateObserver> mNetSateObserver;
		private AtomicBoolean mIsRegisterReceiver = new AtomicBoolean(false);
		private NetReceiver mNetReceiver;

		//==========================================================================
		// 构造方法
		//==========================================================================
		private NetManager() {
			mNetSateObserver = new ArrayList<NetSateObserver>();
			mNetReceiver = new NetReceiver();
		}

		//==========================================================================
		// get、set方法
		//==========================================================================
		public static synchronized NetManager getInstance() {
			if (sInstance == null) {
				sInstance = new NetManager();
			}
			return sInstance;
		}

		/** 判断是否有网络连接 */
		public boolean isNetConnected() {
			NetworkInfo netWorkInfo = getActiveNetInfo();
			return netWorkInfo != null && netWorkInfo.isAvailable();
		}

		/** 判断当前网络是否可用，有连接不代表可用 */
		public boolean isCurrentNetAvailable() {
			ApnInfo apnInfo = getCurrentApnInfo();
			return !(apnInfo == null || apnInfo.getNetType() == NET_TPYE_NONE);
		}

		/** 判断是否有可用的网络 */
		public boolean isNetAvailable() {
			ConnectivityManager mgr = getConnManager();
			NetworkInfo[] info = mgr.getAllNetworkInfo();
			if (info != null) {
				for (NetworkInfo anInfo : info) {
					if (anInfo.getState() == NetworkInfo.State.CONNECTED) {
						return true;
					}
				}
			}
			return false;
		}

		/** 取得当前连接类型 */
		public int getCurrentNetType() {
			return getCurrentApnInfo().getNetType();
		}

		/** 判断WIFI是否可用 */
		public boolean isWifiAvailable() {
			Context context = UIUtils.getContext();
			if (context == null) {
				return false;
			}
			NetworkInfo netInfo = getActiveNetInfo();
			WifiManager wifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
			return ((netInfo != null && netInfo.getType() == ConnectivityManager.TYPE_WIFI) && wifiManager.isWifiEnabled());
		}

		/** 判断2g/3g是否可用 */
		public boolean isMobileAvailable() {
			NetworkInfo netInfo = getActiveNetInfo();
			return (netInfo != null && netInfo.getType() == ConnectivityManager.TYPE_MOBILE);
		}

		/** 获取网络连接管理者ConnectivityManager */
		private ConnectivityManager getConnManager() {
			Context context = UIUtils.getContext();
			if (context == null) {
				return null;
			}
			Object service = context.getSystemService(Context.CONNECTIVITY_SERVICE);
			if (null != service) {
				return (ConnectivityManager) service;
			}
			return null;
		}

		/** 获取连接的网络信息 */
		private NetworkInfo getActiveNetInfo() {
			ConnectivityManager mgr = getConnManager();
			return mgr == null ? null : mgr.getActiveNetworkInfo();
		}

		//==========================================================================
		// 方法
		//==========================================================================

		/** 获取当前网络状态的ApnInfo */
		public ApnInfo getCurrentApnInfo() {
			Context context = UIUtils.getContext();
			if (context == null) {
				return null;
			}
			ApnInfo apnInfo;
			NetworkInfo netInfo = getActiveNetInfo();
			if (netInfo != null) {
				if (netInfo.getType() == ConnectivityManager.TYPE_WIFI) {// Wifi网络
					apnInfo = new ApnInfo();
					apnInfo.setNetType(NET_TPYE_WIFI);
				} else if (netInfo.getType() == ConnectivityManager.TYPE_MOBILE) {// 移动网络
					String apn = netInfo.getExtraInfo();
					String host = Proxy.getDefaultHost();
					// int port = Proxy.getDefaultPort();
					Cursor cursor = null;
					Uri uri = Uri.parse("content://telephony/carriers/");
					try {
						String where = String.format("apn='%s' AND %s ", apn, host == null ? "(proxy IS NULL OR proxy = '')" : ("proxy ='" + host + "'"));
						cursor = context.getContentResolver().query(uri, null, where, null, null);
					} catch (Exception e) {
						LogUtils.e(e);
					}
					apnInfo = makeApnParam(cursor);
					// 如果没有查询到匹配的apn，填充已有数据，并当做net网络，确保返回不为NULL
					if (apnInfo == null) {
						apnInfo = new ApnInfo();
						apnInfo.setNetType(NET_TPYE_NET);
						apnInfo.setExtraNetInfo(apn);
					}
					// apnInfo.setPort(port);
					// apnInfo.setProxy(host);
					try {
						if (cursor != null) {
							cursor.close();
						}
					} catch (Exception e) {
						LogUtils.e(e);
					}
				} else {
					// 非Wifi、非Mobile的其他网络
					apnInfo = new ApnInfo();
					apnInfo.setNetType(NET_TPYE_OTHER);
					apnInfo.setExtraNetInfo(netInfo.getType() + "," + netInfo.getTypeName());
				}
			} else {
				// 当前无可用网络连接
				apnInfo = new ApnInfo();
				apnInfo.setNetType(NET_TPYE_NONE);
			}
			return apnInfo;
		}

		/** 将Apn数据库记录解析为ApnInfo对象 */
		private static ApnInfo makeApnParam(Cursor cur) {
			if (cur == null) {
				return null;
			}
			if (!cur.moveToFirst()) {
				return null;
			}
			ApnInfo apnInfo = new ApnInfo();
			int colId;
			String proxy = null;
			int port = -1;
			String apn = null;
			String mnc = null;

			colId = cur.getColumnIndex("proxy");
			if (-1 != colId) {
				proxy = cur.getString(colId);
				proxy = convertProxy(proxy);
				apnInfo.setProxy(proxy);
			} else {
				LogUtils.e("\"proxy\" column is not found in cursor!");
			}
			colId = cur.getColumnIndex("apn");
			if (-1 != colId) {
				apn = cur.getString(colId);
				apnInfo.setApn(apn);
			} else {
				LogUtils.e("\"apn\" column is not found in cursor!");
			}

			colId = cur.getColumnIndex("mnc");
			if (-1 != colId) {
				mnc = cur.getString(colId);
				colId = cur.getColumnIndex("mcc");
				if (-1 != colId) {
					String mcc = cur.getString(colId);
					apnInfo.setMcc(mcc + mnc);
				} else {
					LogUtils.e("\"mcc\" column is not found in cursor!");
				}
			} else {
				LogUtils.e("\"mnc\" column is not found in cursor!");
			}

			if (proxy != null) {
				colId = cur.getColumnIndex("port");
				if (-1 != colId) {
					port = cur.getInt(colId);
					apnInfo.setPort(port);
				} else {
					LogUtils.e("\"port\" column is not found in cursor!");
				}
				if (proxy.equals(PROXY_CM_UNI)) {
					apnInfo.setMnc(mnc);
					if (mnc != null) {
						if ((mnc.equals(CM_MNC_1) || mnc.equals(CM_MNC_2) || mnc.equals(CM_MNC_7))) {
							apnInfo.setNetType(NET_TPYE_CMWAP);
						} else {
							apnInfo.setNetType(NET_TPYE_UNIWAP);
						}
					}
				} else if (proxy.equals(PROXY_CT)) {
					apnInfo.setNetType(NET_TPYE_CTWAP);
				} else {
					// 如果用其他代理地址，当做net网络
					apnInfo.setNetType(NET_TPYE_NET);
					apnInfo.setExtraNetInfo(apn + ", " + proxy + ", " + port);
				}
			} else {
				// 无代理，可能是net网络
				apnInfo.setNetType(NET_TPYE_NET);
				apnInfo.setExtraNetInfo(apn);
			}
			return apnInfo;
		}

		private static String convertProxy(final String proxy) {
			int[] mSeg;
			String proxyIP = proxy;
			try {
				if (proxyIP != null && proxyIP.length() > 10) {// 代理IP地址位数过长,进行格式转化
					mSeg = new int[4];
					String[] strSeg = proxyIP.split("\\.");
					if (4 == strSeg.length) {
						for (int i = 0; i < 4; i++) {
							mSeg[i] = Integer.parseInt(strSeg[i]);
						}
						proxyIP = new StringBuilder(16).append(mSeg[0]).append('.').append(mSeg[1]).append('.').append(mSeg[2]).append('.').append(mSeg[3]).toString();
					}
				}
			} catch (NumberFormatException e) {
				LogUtils.i("Proxy IP FormatException " + e.getMessage());
				proxyIP = proxy;
			}
			return proxyIP;
		}

		/**
		 * 根据response的getEntity实体中获取网络地址，可以从该网络地址中通过getHostAddress来获取服务器ip地址
		 * @param oEntity 实体数据
		 * @return 返回网络地址
		 */
		@SuppressWarnings("rawtypes")
		public static InetAddress extractServerAddr(HttpEntity oEntity) {
			if (oEntity == null) return null;
			try {
				Class clz = BasicManagedEntity.class;
				Field fConn = clz.getDeclaredField("managedConn");
				fConn.setAccessible(true);
				Object oConn = fConn.get(oEntity);
				clz = AbstractClientConnAdapter.class;
				Field fWrapped = clz.getDeclaredField("wrappedConnection");
				fWrapped.setAccessible(true);
				Object oWrapped = fWrapped.get(oConn);
				clz = SocketHttpClientConnection.class;
				Field fSocket = clz.getDeclaredField("socket");
				fSocket.setAccessible(true);
				Socket oSocket = (Socket) fSocket.get(oWrapped);
				return oSocket.getInetAddress();
			} catch (Exception e) {
				return null;
			}
		}

		/** 使用Wifi时获取IP 设置用户权限 需要权限：android.permission.WAKE_LOCK android.permission.CHANGE_WIFI_STATE android.permission.ACCESS_WIFI_STATE */
		public static String getWifiIp(Context context) {
			// 获取wifi服务
			WifiManager wifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
			if (!wifiManager.isWifiEnabled()) {// 判断wifi是否开启
				wifiManager.setWifiEnabled(true);
			}
			WifiInfo wifiInfo = wifiManager.getConnectionInfo();
			int ipAddress = wifiInfo.getIpAddress();
			return intToIp(ipAddress);
		}

		private static String intToIp(int i) {
			return (i & 0xFF) + "." + ((i >> 8) & 0xFF) + "." + ((i >> 16) & 0xFF) + "." + (i >> 24 & 0xFF);
		}

		public void registerNetSateObserver(NetSateObserver ob) {
			synchronized (mNetSateObserver) {
				if (null != ob) {
					if (!mNetSateObserver.contains(ob)) {
						mNetSateObserver.add(ob);
					}
				}
			}
		}

		public void unregisterNetSateObserver(NetSateObserver ob) {
			synchronized (mNetSateObserver) {
				mNetSateObserver.remove(ob);
			}
		}

		public void onNetSateChanged() {
			if (isNetConnected() && isCurrentNetAvailable()) {
				notifyNetConnect();
			} else {
				notifyNetDisConnect();
			}
		}

		private void notifyNetConnect() {
			synchronized (mNetSateObserver) {
				for (NetSateObserver ob : mNetSateObserver) {
					ob.onNetConnect();
				}
			}
		}

		private void notifyNetDisConnect() {
			synchronized (mNetSateObserver) {
				for (NetSateObserver ob : mNetSateObserver) {
					ob.onNetDisConnect();
				}
			}
		}

		/** 注册广播 */
		public void registerReceiver() {
			if (mIsRegisterReceiver.compareAndSet(false, true)) {
				IntentFilter filter = new IntentFilter();
				filter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
				Context context = UIUtils.getContext();
				if (context != null) {
					context.registerReceiver(mNetReceiver, filter);
				}
			}
		}

		/** 反注册广播 */
		public void unregisterReceiver() {
			if (mIsRegisterReceiver.compareAndSet(true, false)) {
				Context context = UIUtils.getContext();
				if (context != null) {
					context.unregisterReceiver(mNetReceiver);
				}
			}
		}

		//==========================================================================
		// 内部类、内部接口
		//==========================================================================
		public interface NetSateObserver {

			public void onNetConnect();

			public void onNetDisConnect();
		}

		public static class ApnInfo {
			/** APN 名称 */
			private String apn;
			/** 代理IP */
			private String proxy;
			/** 端口号 */
			private int port;
			/** 网络类型 */
			private int netType;
			/** 网络类型补充信息 */
			private String extraNetInfo;
			/** MNC号 */
			private String mnc;
			/** MCC号 */
			private String mcc;

			public String getApn() {
				return apn;
			}

			public String getProxy() {
				return proxy;
			}

			public int getPort() {
				return port;
			}

			public int getNetType() {
				return netType;
			}

			public String getExtraNetInfo() {
				return extraNetInfo;
			}

			public String getMnc() {
				return mnc;
			}

			public String getMcc() {
				return mcc;
			}

			public void setApn(String apn) {
				this.apn = apn;
			}

			public void setProxy(String proxy) {
				this.proxy = proxy;
			}

			public void setPort(int port) {
				this.port = port;
			}

			public void setNetType(int netType) {
				this.netType = netType;
			}

			public void setExtraNetInfo(String extraNetInfo) {
				this.extraNetInfo = extraNetInfo;
			}

			public void setMnc(String mnc) {
				this.mnc = mnc;
			}

			public void setMcc(String mcc) {
				this.mcc = mcc;
			}
		}

		private final class NetReceiver extends BroadcastReceiver {

			@Override
			public void onReceive(Context context, Intent intent) {
				String action = intent.getAction();
				if (action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
					NetManager.getInstance().onNetSateChanged();
				}
			}
		}
}
